/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator;

import java.io.IOException;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;

import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater;

/**
 * @author marcelop
 * @since 0.2.0
 */
public class TestResourceChangeUpdater 
implements IResourceChangeUpdater
{
	private TestNavigator testNavigator;
	private List addedObjects;
	private List removedObjects;
	private List changedObjects;
	private List resourcesToSave;
	
	private Object[] expandedElements;
	private Object[] selectedElements;
	
	private boolean active;
	
	/**
	 * Constructor for TestResourceChangeUpdater
	 */
	public TestResourceChangeUpdater(TestNavigator testNavigator)
	{
		this.testNavigator = testNavigator;
		active = true;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testNavigator = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#started()
	 */
	public void started()
	{
		addedObjects = new UniqueEList();
		removedObjects = new UniqueEList();
		changedObjects = new UniqueEList();
		resourcesToSave = new UniqueEList();		
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#ended()
	 */
	public void ended()
	{
		if(!addedObjects.isEmpty())
		{
			for (Iterator i = addedObjects.iterator(); i.hasNext();)
			{
				Object object = i.next();
				Object parent = ((ITreeContentProvider)testNavigator.getViewer().getContentProvider()).getParent(object);
				if(parent == null)
				{
					testNavigator.getTreeViewer().refresh(true);
					return;
				}
				testNavigator.getTreeViewer().add(parent, object);
			}
		}

		if(!removedObjects.isEmpty())
			testNavigator.getTreeViewer().remove(removedObjects.toArray());

		if(!changedObjects.isEmpty())
		{
			for (Iterator i = changedObjects.iterator(); i.hasNext();)
				testNavigator.getTreeViewer().refresh(i.next(), true);
		}
		
		if(!resourcesToSave.isEmpty())
		{
			for (Iterator i = resourcesToSave.iterator(); i.hasNext();)
			{
				try
				{
					EMFUtil.save((Resource)i.next());
				}
				catch (Exception e)
				{
				}
			}			
		}
				
		addedObjects.clear();
		addedObjects = null;
		
		removedObjects.clear();
		removedObjects = null;
		
		changedObjects.clear();
		changedObjects = null;
		
		resourcesToSave.clear();
		resourcesToSave = null;

		if(expandedElements != null)
		{
			convertURIToEObject(expandedElements);
			testNavigator.getTreeViewer().setExpandedElements(expandedElements);
			convertURIToEObject(selectedElements);
			testNavigator.getTreeViewer().setSelection(new StructuredSelection(selectedElements), true);
	
			expandedElements = null;
			selectedElements = null;
		}
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#add(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean add(IResource parent, IResource[] affectedResources) {
		for (int i = 0, maxi = affectedResources.length; i < maxi; i++) {
			if (affectedResources[i].getType() != IResource.FILE) {
				//- not a file thus a project or folder, add it if it is no filtered
				if(testNavigator.isVisibleResource(affectedResources[i])) {
					addedObjects.add(affectedResources[i]);
				}
			} else {
				//- if the file should be converted through a special converter, use it
				Object convertedObject = testNavigator.getConvertedObjectFromFile(affectedResources[i]);
				if(convertedObject == null) {
					//- no file conversion,	insert the file as is if this file is known by the navigator
					if(testNavigator.isVisibleResource(affectedResources[i])) {
						addedObjects.add(affectedResources[i]);
					}
				} else {
					//- file has been converted into another object, insert ojbect instead of the file
					addedObjects.add(convertedObject);
				}
			}
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#remove(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource[])
	 */
	public boolean remove(IResource parent, IResource[] affectedResources)
	{
		for(int i = 0, maxi = affectedResources.length; i < maxi; i++) {
			if(affectedResources[i].getType() != IResource.FILE) {
				removedObjects.add(affectedResources[i]);
			} else {
				//- in case of file removed, we need to find the right corresponding element in the test navigator
				//- first is this file has been converted into a simple object by a converter ?
				Object convertedObject = testNavigator.getCorrespondingObjectFromFile(affectedResources[i]);
				if(convertedObject == null) {
					//- this file hasn't been converted, it should be as is in the test navigator
					removedObjects.add(affectedResources[i]);
				} else {
					//- this file has been converted so remove its conversion 
					removedObjects.add(convertedObject);
					//- map entry should be cleaned
					testNavigator.cleanConvertedObjectEntryFor(affectedResources[i]);
					//- if the converted object is a EMF object we need to clean the resource associated to this object
					if (convertedObject instanceof EObject) {
						Resource res = EMFUtil.getResource(null, (IFile)affectedResources[i]);
						EMFUtil.remove(res);
					}
				}
//				URI uri = getFileURI(affectedResources[i]);
//				if(uri == null)
//					continue;
//					
//				Resource resource = resourceSet.getResource(uri, false);
//				if(resource != null) {
//					removedObjects.addAll(resource.getContents());
//					resourcesToSave.addAll(Arrays.asList(EMFUtil.remove(resource)));
//				}
			}
		}
		
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#replaced(org.eclipse.core.resources.IResource)
	 */
	public boolean replaced(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateProperties(org.eclipse.core.resources.IResource)
	 */
	public boolean updateProperties(IResource affectedResource)
	{
		if(affectedResource.getType() == IResource.FILE)
		{
			EObject[] eObjects = EMFUtil.getEObjects(null, (IFile)affectedResource);
			if(eObjects.length > 0)
				testNavigator.getTreeViewer().update(eObjects, null);
		}
		else
		{
			testNavigator.getTreeViewer().update(affectedResource, null);
		}
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateChildrenType(org.eclipse.core.resources.IResource)
	 */
	public boolean updateChildrenType(IResource affectedResource)
	{
		return false;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#updateContent(org.eclipse.core.resources.IResource, org.eclipse.core.resources.IResource)
	 */
	public boolean updateContent(IResource parent, IResource affectedResource)
	{
		URI uri = getFileURI(affectedResource);
		if(uri != null)	{
			EMFUtil.unloadIfNeeded(uri);
			Object convertedObject = testNavigator.getCorrespondingObjectFromFile(affectedResource);
			if(convertedObject != null) {
				//- if the affected resource is converted in the navigator
				//- remove this converted object
				removedObjects.add(convertedObject);
				//- update the map
				testNavigator.cleanConvertedObjectEntryFor(affectedResource);
				//- add the new converted object from the new resource using the its converter
				convertedObject = testNavigator.getConvertedObjectFromFile(affectedResource);
				addedObjects.add(convertedObject);
			} else {
				changedObjects.add(affectedResource);
			}
			if(expandedElements == null)
			{
				expandedElements = testNavigator.getTreeViewer().getExpandedElements();
				convertEObjectToURI(expandedElements);
				selectedElements = ((IStructuredSelection)testNavigator.getTreeViewer().getSelection()).toArray();
				convertEObjectToURI(selectedElements);
			}				
		}
		return false;
	}


	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#setActive(boolean)
	 */
	public void setActive(boolean enable)
	{
		active = enable;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater#isActive()
	 */
	public boolean isActive()
	{
		return active;
	}
	
	/**
	 * Returns the uri associated with the resource if this is a file and has
	 * the correct file extension.
	 * @param resource
	 * @return URI
	 */
	protected URI getFileURI(IResource resource)
	{
		if(resource.getType() == IResource.FILE)
		{
			String fileExtension = resource.getFileExtension();
			String[] files = testNavigator.getVisibleFiles();
			for (int i = 0; i < files.length; i++) {
				if (files[i].equals(fileExtension)) {
					return URI.createPlatformResourceURI(resource.getFullPath().toString());
				}
			}
		}
		return null;
	}
	
	/**
	 * Returns whether the eObject is valid in the context of the 
	 * test navigator.
	 * @param eObject
	 * @return boolean
	 */
	protected boolean isValid(EObject eObject)
	{
		return (eObject instanceof CMNNamedElement);
	}
	
	/**
	 * @param expandedElements
	 */
	private void convertEObjectToURI(Object[] objects)
	{
		for (int i = 0, maxi = objects.length; i < maxi; i++)
		{
			if(objects[i] instanceof EObject)
				objects[i] = EcoreUtil.getURI((EObject)objects[i]);
		}
	}
	
	private void convertURIToEObject(Object[] objects)
	{
		for (int i = 0, maxi = objects.length; i < maxi; i++)
		{
			if(objects[i] instanceof URI)
				objects[i] = EMFUtil.getResourceSet().getEObject((URI)objects[i], false);
		}
	}
}
