/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.invocation;

/**
 * This is the supertype of classes that are essentially data holders that
 * are used during the marshalling and unmarshalling of data that is to be
 * sent between processes in support of remote method invocation semantics.
 * 
 * The state of <code>MarshalData</code> instances is intended to be
 * immutable. It should only be set during instantiation, by the constructors
 * provided. The intention is to prevent invalid data combinations.
 */
public abstract class MarshalData {
	
	/**
	 * the unique id associated with the object on which some action is
	 * to be performed
	 */
	protected Integer targetId;
	/**
	 * the argument types of the method that is to be or has already
	 * been invoked
	 */
	protected Class[] argTypes;
	/**
	 * the name of the method that is to be or has already been invoked
	 */
	protected String call;
	
	/**
	 * 
	 * @param targetId
	 * @param argTypes
	 * @param call
	 */
	protected MarshalData(Integer targetId, Class[] argTypes, String call) {
		this.targetId = targetId;
		this.argTypes = argTypes;
		this.call = call;
	}
	
	private MarshalData(){}
	
	/**
	 * Get the unique id associated with the target instance -- the object
	 * on which some action is to be performed.
	 * 
	 * @return an <code>Integer</code>
	 */
	public Integer getTargetId() {
		return targetId;
	}
	
	/**
	 * Get the argument type list of the method that is to be or already
	 * has been invoked.
	 * 
	 * @return an array of <code>Class</code> objects
	 */
	public Class[] getArgTypes() {
		return argTypes;
	}

	/**
	 * Get the name of the method that is to be or already has been invoked.
	 * 
	 * @return a method name
	 */
	public String getCall() {
		return call;
	}
	
	/**
	 * Determine if this instance was created for the same method call as
	 * the specified instance.
	 * 
	 * @param comp a <code>MarshalData</code> instance
	 * 
	 * @return <code>true</code> if this instance was created for the same
	 *         method call as that for which <code>comp</code> was created,
	 *         <code>false</code> otherwise
	 */
	public boolean isForSameCallAs(MarshalData comp) {
		if ( comp == null )
			return false;
		if ( !targetId.equals(comp.targetId) )
			return false;
		if ( !call.equals(comp.call) )
			return false;
		if ( argTypes.length != comp.argTypes.length )
			return false;
		for ( int idx=0; idx<argTypes.length; idx++ ) {
			if ( !argTypes[idx].equals(comp.argTypes[idx]) )
				return false;
		}
		return true;
	}

}
