/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.dialog;

import java.util.Iterator;
import java.util.Vector;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;

/**
 * This class implements a dialog to select one or more resources with a certain file extension
 * in the workspace.
 * 
 * @author bjiang
 * @since  3.0
 */
public class EObjectResourceSelectionDialog extends Dialog implements SelectionListener, IDoubleClickListener
{
	private String[] fileExtensions;
	
	private String title;
	private Button filterButton;
	private TreeViewer hostViewer;
	private EObjectResourceContentProvider contentProvider;
	private Image fileImage;
	
	private EObject[] selectedEObjects;
		
	/**
	 * @param parentShell
	 */
	public EObjectResourceSelectionDialog(Shell parentShell, String title, String[] theFileExtensions, Image fileImage)
	{
		super(parentShell);
		this.title = title;
		this.fileExtensions = theFileExtensions;
		this.fileImage = fileImage;
	}
		
	protected Control createDialogArea(Composite parent)
	{
		Composite composite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 300;
		gridData.widthHint = 250;
		composite.setLayoutData(gridData);
	
		filterButton = new Button(composite, SWT.CHECK);
		filterButton.setText(TestUIPlugin.getString("SHOW_FOLDERS")); //$NON-NLS-1$
		filterButton.addSelectionListener(this);
		createViewer(composite);
		hostViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
		((Tree)hostViewer.getControl()).addSelectionListener(this);
		((TreeViewer)hostViewer).expandToLevel(2);
		
		return composite;
	}
	
	protected StructuredViewer createViewer(Composite parent)
	{		
		Tree tree = new Tree(parent, SWT.MULTI | SWT.FULL_SELECTION | SWT.BORDER );
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 150;
		tree.setLayoutData(gridData);
			
		contentProvider = new EObjectResourceContentProvider(fileExtensions);
		hostViewer = new TreeViewer(tree);
		hostViewer.setContentProvider(contentProvider);
		hostViewer.setLabelProvider(new EObjectResourceLabelProvider(true, fileImage));
		hostViewer.addDoubleClickListener(this);

		contentProvider.setFilter(EObjectResourceContentProvider.FILTER_FOLDERS);

		ViewerSorter sorter = new ViewerSorter()
		{
			public void sort(Viewer viewer, Object[] elements)
			{
				if(elements.length > 0)
					super.sort(viewer, elements);
			}
		
			public int category(Object element)
			{
				if(element instanceof IResource)
					return 0;

				return 10;
			}		
		};
		hostViewer.setSorter(sorter);
	
		return hostViewer;
	}
	
	protected void configureShell(Shell shell) {
		super.configureShell(shell);
		if (title != null)
			shell.setText(title);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(e.widget == filterButton)						
		{
			contentProvider.setFilter((filterButton.getSelection() ? EObjectResourceContentProvider.SHOW_ALL : EObjectResourceContentProvider.FILTER_FOLDERS));
			ISelection selection = hostViewer.getSelection();				
			hostViewer.refresh();
			if(!selection.isEmpty())
				hostViewer.setSelection(selection, true);
		}
		
		if(e.widget == hostViewer.getControl())
		{
			setSelection();
		}
	}
	
	private void setSelection()
	{
		ISelection selection = hostViewer.getSelection();
		if(selection != null && !selection.isEmpty())
		{ 
			Iterator it = ((IStructuredSelection)selection).iterator();
			Vector objects = new Vector();
			while(it.hasNext())
			{
				Object element = it.next();
				if(element instanceof EObject)
					objects.add((EObject)element);
			}
			selectedEObjects = new EObject[objects.size()];
			for(int i = 0; i < objects.size(); i++)
				selectedEObjects[i] = (EObject)objects.get(i);
		}
	}
	
	public EObject[] getSelectedObjects()
	{
		if(selectedEObjects == null)
			selectedEObjects = new EObject[0];
					
		return this.selectedEObjects;
	}

	public IFile[] getSelectedFiles()
	{
		if(selectedEObjects == null)
			return new IFile[0];
			
		IFile[] selectedFiles = new IFile[selectedEObjects.length];
		for(int i = 0; i < selectedEObjects.length; i++)
			selectedFiles[i] = EMFUtil.getWorkspaceFile(selectedEObjects[i]);
			
		return selectedFiles;
	}
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
     */
    public void doubleClick(DoubleClickEvent event)
    {
        if(event.getViewer() == hostViewer)
        {
            setSelection();
            this.buttonPressed(IDialogConstants.OK_ID);
        }
        
    }
}
