/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.model.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.model.EMFUtil;

/**
 * This action removes the one or more children of an eObject.  The children 
 * are identified by the selection. 
 * 
 * @author marcelop
 * @since 0.2.0
 */
public class RemoveChildrenAction 
extends ChildrenSelectionAction
{
	private boolean saveChanges;
	private boolean showErrorDialog;
	private boolean unloadNotSavedResource;
	
	/**
	 * Constructor for RemoveChildrenAction
	 * @param text
	 */
	public RemoveChildrenAction(boolean saveChanges)
	{
		super(TestUIPlugin.getString("BTN_REMOVE")); //$NON-NLS-1$
		
		this.saveChanges = saveChanges;
		if(saveChanges)
			setText(TestUIPlugin.getString("LBL_DELETE")); //$NON-NLS-1$
			
		setToolTipText(getText());
		ISharedImages images = PlatformUI.getWorkbench().getSharedImages();
		setDisabledImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_DISABLED));
		setImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE));		
		setHoverImageDescriptor(images.getImageDescriptor(ISharedImages.IMG_TOOL_DELETE_HOVER));		
	}

	/**
	 * @see org.eclipse.ui.actions.SelectionListenerAction#updateSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected boolean updateSelection(IStructuredSelection selection)
	{
		if(!super.updateSelection(selection))
			return false;
		
		if((getStructuredViewer() == null) || (getChildren().size() != 1))
			return true;
			
		return (getStructuredViewer().getInput() != getChildren().get(0));
	}

	/**
	 * @param unloadNotSavedResource
	 */
	public void setUnloadNotSavedResource(boolean unloadNotSavedResource)
	{
		this.unloadNotSavedResource = unloadNotSavedResource;
	}

	/**
	 * @return
	 */
	public boolean isUnloadNotSavedResource()
	{
		return unloadNotSavedResource;
	}

	/**
	 * @return
	 */
	public boolean isSaveChanges()
	{
		return saveChanges;
	}

	/**
	 * @param showErrorDialog
	 */
	public void setShowErrorDialog(boolean showErrorDialog)
	{
		this.showErrorDialog = showErrorDialog;
	}

	/**
	 * @return
	 */
	public boolean isShowErrorDialog()
	{
		return showErrorDialog;
	}
	
	/**
	 * @see org.eclipse.jface.action.IAction#run()
	 */
	public void run()
	{
		setActionPerformed(false);
		Set changedResources = breakFeatures(getStructuredSelection());
		if(changedResources.isEmpty())
			return;
	
		List unsavedResources = saveResources(changedResources);
		changedResources.clear();
		
		if(unsavedResources.isEmpty())
			return;
		
		handleUnsavedResources(unsavedResources);
		unsavedResources.clear();	
	}
	
	protected Set breakFeatures(IStructuredSelection structuredSelection)
	{
		Set changedResources = new HashSet();
		for (Iterator i = structuredSelection.iterator(); i.hasNext();)
		{
			EObject eObject = (EObject)i.next();
			Resource[] resources = EMFUtil.remove(eObject);
			if(resources.length > 0)
				setActionPerformed(true);
			if(isSaveChanges())
				changedResources.addAll(Arrays.asList(resources));
		}
			
		adjustStructuredViewer();	
		getChildren().clear();
		return changedResources;
	}
	
	protected void adjustStructuredViewer()
	{
	}
	
	protected List saveResources(Set changedResources)
	{
		List unsavedResources = new ArrayList();
		for (Iterator i = changedResources.iterator(); i.hasNext();)
		{
			Resource resource = (Resource)i.next();
			try
			{
				EMFUtil.save(resource);
			}
			catch (Exception e)
			{
				TestUIPlugin.logError(e);
				unsavedResources.add(resource);
			}
		}
		return unsavedResources;			
	}
	
	protected void handleUnsavedResources(List unsavedResources)
	{
		if(isShowErrorDialog())
		{
			StringBuffer filePaths = new StringBuffer();
			for (Iterator i = unsavedResources.iterator(); i.hasNext();)
			{
				Resource resource = (Resource)i.next();
				IFile file = EMFUtil.getWorkspaceFile(resource);
				if(file != null)
					filePaths.append("\n").append(file.getFullPath().toString()); //$NON-NLS-1$			
			}
			
			String message = TestUIPlugin.getString("_ERROR_RESS_SAVE", filePaths.toString()); //$NON-NLS-1$
			MessageDialog.openError(Display.getCurrent().getActiveShell(), TestUIPlugin.getString("W_ERROR"), message); //$NON-NLS-1$
		}
		
		if(isUnloadNotSavedResource())
		{
			for (Iterator i = unsavedResources.iterator(); i.hasNext();)
			{
				Resource resource = (Resource)i.next();
				resource.unload();
			}
		}
	}
}
