/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project. All rights reserved. This program
 * and the accompanying materials are made available under the terms of the
 * Common Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/cpl-v10.html Contributors: IBM -
 * Initial API and implementation
 ******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.regex.Pattern;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.local.NodeImpl;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.Common_ConfigurationFactory;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;

/**
 * This class provides an implementation of
 * {@link JavaExecutionDeploymentAdapter}to automatically deploy test assets
 * using file transfer service from RAC.
 * 
 * @author bjiang
 * @since 3.0
 */
public class JavaExecutionDeploymentAdapter implements IExecutionDeploymentAdapter
{
	/**
	 * Slash ("/") will be used in all path names processed by this class.
	 * Reason for this is when this class collects CLASSPATH and gets ready for file
	 * transfer, it has no knowledge of the target machine, whether it is a Windows
	 * or UNIX platform.  Since either slash("/") or back slash("\") works on Windows
	 * and only slash("/") works on UNIX platform, slash is the only choice to ensure
	 * path name gets recognized on both type of platforms.  
	 */
	public static final String PATH_NAME_SEPARATOR = "/";
	
    private HashMap hostFileManagerMap = new HashMap();

    public JavaExecutionDeploymentAdapter()
    {
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.harness.IExecutionDeploymentAdapter#deployTestAssets(org.eclipse.hyades.models.common.testprofile.TPFDeployment)
     */
    public void deployTestAssets(INode node, TPFDeployment deployment, boolean isStandalone)
    {
    	if(node == null || deployment == null)
    		return;       
        
        CFGArtifactLocationPair[] pairs = collectPairsByNode(deployment, node);
        if(pairs == null || pairs.length == 0)
            return;
        
        HashMap deployableFiles = collectDeployableFiles(pairs, isStandalone);
        StringBuffer bufError = new StringBuffer();
        HashMap deployed = deployToNode(node, deployableFiles, bufError);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.harness.IExecutionDeploymentAdapter#cleanUpTestAssets(org.eclipse.hyades.models.common.testprofile.TPFDeployment)
     */
    public void cleanUpTestAssets(INode node, TPFDeployment deployment, boolean isStandalone)
    {
    	if(node == null || deployment == null)
    		return;       
        
        CFGArtifactLocationPair[] pairs = collectPairsByNode(deployment, node);
        if(pairs == null || pairs.length == 0)
            return;
        
        HashMap deployableFiles = collectDeployableFiles(pairs, isStandalone);
        if(deployableFiles != null && deployableFiles.size() > 0)
	    {
	        String[] deleteFiles = (String[])deployableFiles.values().toArray();
	        StringBuffer bufError = new StringBuffer();
	        cleanFromNode(node, deleteFiles, bufError);
	    }
    }
    
    protected CFGArtifactLocationPair[] collectPairsByNode(TPFDeployment deployment, INode node)
    {
    	CFGArtifactLocationPair[] sameNodePairs = new CFGArtifactLocationPair[0];
    	String nodeName = node.getName();
        if(nodeName == null || nodeName.length() < 1)
        	return sameNodePairs;
        EList pairs = deployment.getArtifactLocations();
        if(pairs == null || pairs.isEmpty())
            return sameNodePairs;
        
        CFGArtifactLocationPair pair = null;
        String hostname = null;
        for(Iterator it = pairs.iterator(); it.hasNext(); )
        {
            pair = (CFGArtifactLocationPair)it.next();
			if(!isValidPair(pair))
				continue;
            
            CFGLocation location = pair.getLocation();
			if(location instanceof CFGMachineConstraint)
				hostname = ((CFGMachineConstraint)pair.getLocation()).getHostname();
			else
				hostname = pair.getLocation().getName();
			
			if(hostname == null || hostname.length() < 1)
			    continue;
			
			if(nodeName.equals(hostname))
			{				
			    if(sameNodePairs.length == 0)
			    {
			        sameNodePairs = new CFGArtifactLocationPair[]{pair};
			    }
			    else
			    {
			        CFGArtifactLocationPair[] temp = sameNodePairs;
			        sameNodePairs = new CFGArtifactLocationPair[temp.length + 1];
			        System.arraycopy(temp, 0, sameNodePairs, 0, temp.length);
			        sameNodePairs[temp.length] = pair;
			    }
			}
        }
        
        return sameNodePairs;
    }
    
    protected boolean isValidPair(CFGArtifactLocationPair pair)
    {
    	CFGLocation location = pair.getLocation();
    	CFGArtifact artifact = pair.getArtifact();
    	if(location == null || artifact == null)
    		return false;
    	
    	return true;
    }
    
    protected HashMap collectDeployableFiles(CFGArtifactLocationPair[] pairs, boolean isStandalone)
    {
        HashMap deployableFiles = new HashMap();
        CFGArtifactLocationPair pair = null;
        CFGArtifact artifact = null;
        CFGLocation location = null;
        String deployRootDir = null;
        CFGClass deployable = null;
        String path = null;
        String srcAbsolutePath = null;
        String destAbsolutePath = null;
        for(int i = 0; i < pairs.length; i++)
        {
            pair = pairs[i];
            location = pair.getLocation();
            artifact = pair.getArtifact();
            if(artifact == null || 
			   location == null ||
			   !(location instanceof CFGMachineConstraint))
                continue;
            
            deployRootDir = getDeployRootDir(location, null);            
            CFGPropertyGroup propertyGroup = ConfigurationUtil.searchPropertyGroupById(artifact.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_ARTIFACT);
            if(propertyGroup != null)
            {                
            	// deploy files from classpath
                Vector targetClasspaths = new Vector();
                BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(propertyGroup.getProperties(), ConfigurationUtil.CLASSPATH, false);
                for(int j = 0; j < properties.length; j++)
                {
                	StringTokenizer tokenizer = new StringTokenizer(properties[j].getValue(), File.pathSeparator);
                	while(tokenizer.hasMoreTokens())
                	{
                		srcAbsolutePath = uniformPath(tokenizer.nextToken());
                		destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);
	                    targetClasspaths.add(destAbsolutePath);
	                    
	                    // no deployment of classpath files if the source and destination is the same
	                    if(isSourceDestinationSame((CFGMachineConstraint)location, srcAbsolutePath, destAbsolutePath))
	                    	continue;
	                    
	                    File file = new File(srcAbsolutePath);
	                    if(file != null && file.isDirectory())
	                    {
	                    	String[] files = getClasspathFiles(file);                   
	                        for(int indx = 0; indx < files.length; indx++)
	                        {
	                        	srcAbsolutePath = (String)files[indx];
	                        	destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);
	                            deployableFiles.put(srcAbsolutePath, destAbsolutePath);
	                        }
	                    }                    
	                    else
	                        deployableFiles.put(srcAbsolutePath, destAbsolutePath);
                	}
                }
                
                // save the destination classpath in location property group
				//for(int x = 0; x < targetClasspaths.size(); x++) { System.out.println(targetClasspaths.get(x)); }
                setLocationClasspath(location, targetClasspaths);
            }
            
            // deploy resources from artifact deployables.
            List deployables = artifact.getDeployableInstances();
            for(Iterator it = deployables.iterator(); it.hasNext(); )
            {
                deployable = (CFGClass)it.next();
                if(deployable instanceof TPFTestSuite)
                    continue;
                
                srcAbsolutePath = getDeployableFilePath(deployable, isStandalone);
                if(srcAbsolutePath == null)
                	continue;
                destAbsolutePath = getDeployFilePath(srcAbsolutePath, deployRootDir, isStandalone);  
                if(!isSourceDestinationSame((CFGMachineConstraint)location, srcAbsolutePath, destAbsolutePath))
                	deployableFiles.put(srcAbsolutePath, destAbsolutePath);
            }
        }    
        
        return deployableFiles;
    }
    
    protected String getDeployFilePath(String sourceFile, String deployRootDir, boolean isStandalone)
    {
    	String deployFile;
        String sourceRootDir = getSourceRootDir(new File(sourceFile), isStandalone);
        deployRootDir = (deployRootDir == null || deployRootDir.length() < 1) ? sourceRootDir : deployRootDir;
        if(sourceFile.startsWith(deployRootDir))
        	deployFile = sourceFile;
        else
        	deployFile = (deployRootDir + sourceFile.substring(sourceRootDir.length() - 1, sourceFile.length()));
          
        // bugzilla 67677 fix, use name separator "/" to ensure path name on Windows and UNIX platforms.
        deployFile = uniformPath(deployFile);
        return deployFile;
    }
    
    protected String getDeployableFilePath(CFGClass deployable, boolean isStandalone)
    {
    	Resource resource = deployable.eResource();
        if(resource == null)
            return null;                
        URI uri = resource.getURI();
        
        if(isStandalone)
        	return uri.toFileString();
        else
        {
	        String path = uri.toString();
	        if(path.startsWith("platform:/resource/"))
	            path = getWorkspaceRoot() + path.substring(("platform:/resource/".length()));
	        
	        return path;
        }
    }
    
    protected boolean isSourceDestinationSame(CFGMachineConstraint location, String source, String destination)
    {
    	String hostname = location.getHostname();
    	try
		{
	    	if(!(InetAddress.getLocalHost().equals(InetAddress.getByName(hostname))|| InetAddress.getByName(hostname).isLoopbackAddress())) 
	    		return false;
		}
    	catch(UnknownHostException e){}
    	
    	File src = new File(source);
    	File des = new File(destination);
    	if(src.compareTo(des) == 0)
    		return true;
    	
    	return false;
    }
    
    protected String[] getClasspathFiles(File clspathDirectory)
    {
    	Vector clsFiles = new Vector();
    	collectClassFiles(clspathDirectory, clsFiles);    	
    	
    	String[] files = new String[clsFiles.size()];    	
    	for(int i = 0; i < clsFiles.size(); i++)
    	{
    		files[i] = (String)clsFiles.get(i);
    	}
    	return files;
    }

    protected void collectClassFiles(File dir, Vector clsFilePaths)
    {        
        File[] files = dir.listFiles();
        for(int i = 0; i < files.length; i++)
        {
            if(files[i].isFile())
            {
                if(files[i].getName().endsWith(".class"))
                {
//                	 bugzilla 67677 fix, use name separator "/" to ensure pathname works on Windows and UNIX platforms.
                    clsFilePaths.add(uniformPath(files[i].getPath()));
                }
            }
            else
                collectClassFiles(files[i], clsFilePaths);
        } 
    }
    
    protected HashMap deployToNode(INode node, HashMap deployableFiles, StringBuffer bufError)
    {
        HashMap deployedFiles = new HashMap();
        // TODO remove when file manager services are exposed through INode
        if(!(node instanceof NodeImpl))
            return deployedFiles;
        
        IFileManager fileManager = ((NodeImpl)node).getFileManager();
        if(fileManager == null)
            return deployedFiles;
        
        for(Iterator i = deployableFiles.keySet().iterator(); i.hasNext(); )
        {
            String src = (String)i.next();
            String dest = (String)deployableFiles.get(src);
            
            if(deployFile(fileManager, src, dest, bufError))
                deployedFiles.put(src, dest);
        }
        
        return deployedFiles;
    }
    
    protected boolean deployFile(IFileManager fileManager, String srcAbsolutePath, String destAbsolutePath, StringBuffer bufError)
    {
        try
        {                    
        	//System.out.println("About to PUT file from:");
        	//System.out.println(srcAbsolutePath);
        	//System.out.println("to: " + destAbsolutePath + " on the target machine.");
            fileManager.putFile(srcAbsolutePath, destAbsolutePath);
            return true;            
        }
        catch(IOException e)
        {
            bufError.append(e.getMessage());
        }
        
        return false;
    }
    
    protected void cleanFromNode(INode node, String[] files, StringBuffer bufError)
    {
        // TODO remove when file manager services are exposed through INode
        if(!(node instanceof NodeImpl))
            return;
        
        IFileManager fileManager = ((NodeImpl)node).getFileManager();
        if(fileManager == null)
            return;
        
        Vector deployedFiles = new Vector(); 
        for(int i = 0; i < files.length; i++)
        {
            try
            {
                fileManager.deleteFile(files[i]);
            }
            catch(IOException e)
            {
                bufError.append(e.getMessage());
            }
        }
    }
    
    protected String getSourceRootDir(File source, boolean isStandalone)
    {
    	// bugzilla 67677 fix, use name separator "/" to ensure path name works on Windows and UNIX platforms.
    	String sourcePath = uniformPath(source.getPath());
    	if(!isStandalone)
    	{
    		// get workspace root
	        String workspaceRoot = getWorkspaceRoot();	        
	        if(workspaceRoot != null && workspaceRoot.length() > 0 &&
	        	sourcePath.startsWith(workspaceRoot))
	        {
	        	return workspaceRoot;
	        }
    	}

    	// if it is not from workspace or this is a standalone execution
    	// return the root directory of the source 
    	URI sourceURI = URI.createFileURI(source.getPath());
		if(sourceURI.hasDevice())
			return sourceURI.device() + PATH_NAME_SEPARATOR;
		else 
			return PATH_NAME_SEPARATOR;
    }
    
    protected String getWorkspaceRoot()
    {
    	if(ResourcesPlugin.getWorkspace() != null)
    	{
    		String workspaceRoot = ResourcesPlugin.getWorkspace().getRoot().getLocation().toString();
    		workspaceRoot = uniformPath(workspaceRoot);
    		if(workspaceRoot != null && !workspaceRoot.endsWith(PATH_NAME_SEPARATOR))
    			workspaceRoot += PATH_NAME_SEPARATOR;
	        return workspaceRoot;
    	}
    	
    	return "";
    }
    
    protected String getDeployRootDir(CFGLocation location, String defaultDir)
    {
        String deployRootDir = null;
        
        CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
        if(propGroup == null)
        {
            propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
            propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
            location.getPropertyGroups().add(propGroup);
        }

        BVRProperty[] props = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), ConfigurationUtil.ROOTDIR, false);
        if(props == null || props.length == 0)
        {
            deployRootDir = defaultDir;
            CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
            property.setName(ConfigurationUtil.ROOTDIR);
            property.setOperator("=");
            property.setValue(deployRootDir);
            propGroup.getProperties().add(property);
        }
        else
            deployRootDir = props[0].getValue();
        
        if(deployRootDir != null)
        {
        	// bugzilla 67677 fix, use name separator "/" to ensure path name works on Windows and UNIX platforms.
        	deployRootDir = uniformPath(deployRootDir);
        }
        
        return deployRootDir;
    }
    
    protected void setLocationClasspath(CFGLocation location, Vector classpaths)
    {
        if(classpaths == null || classpaths.size() < 1)
            return;
        
        CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
        if(propGroup == null)
        {
            propGroup = Common_ConfigurationFactory.eINSTANCE.createCFGPropertyGroup();
            propGroup.setPropertyGroupID(ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
            location.getPropertyGroups().add(propGroup);
        }
        
        String classpath = "";
        BVRProperty[] existingProps = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), ConfigurationUtil.CLASSPATH, false);
        for(int i = 0; i < classpaths.size(); i++)
        {
        	classpath = (String)classpaths.get(i);	        
			boolean existed = false;
		    for(int propIndex = 0; propIndex < existingProps.length; propIndex++)
		    {
		        if(existingProps[propIndex].getValue().indexOf(classpath) > -1)
		        {
		            existed = true;
		            break;
		        }
		    }
		    
		    if(existed == false)
		    {
		        CFGComparableProperty property = Common_ConfigurationFactory.eINSTANCE.createCFGComparableProperty();
		        property.setName(ConfigurationUtil.CLASSPATH);
		        property.setOperator("=");
		        property.setValue(classpath);
		        propGroup.getProperties().add(property);
		    }
        }
    }
    
    /**
     * This method is used to make sure name separator used in <i>pathname</i>
     * is slash ("/").
	 * <P>Since either slash("/") or back slash("\") works on Windows
	 * and only slash("/") works on UNIX platform, slash is the choice to ensure
	 * path name gets recognized on both type of platforms.  
	 * 
     * @param pathname the path name to be reformed.
     * @return a pathname separated by slashes("/").
     */
    protected String uniformPath(String pathname)
    {
    	return pathname.replaceAll("\\\\", PATH_NAME_SEPARATOR);
    	//URI pathURI = URI.createFileURI(pathname);
    	//String path = pathURI.devicePath();
    	//if(pathURI.hasAuthority())
    		//path = path.substring(pathURI.authority().length(), path.length());
    	
    	//return path;
    }
}