/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.piclient;

import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.emf.common.util.*;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.common.*;
import org.eclipse.hyades.internal.execution.local.control.*;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.*;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.*;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.*;
import org.eclipse.hyades.trace.ui.launcher.*;


public class PIProcessListener implements ProcessListener, AgentPeerListener {
    //~ Instance fields ----------------------------------------------------------------------------

    protected ArrayList _monitoredAgents = new ArrayList();
    protected ArrayList _processVariableList = new ArrayList();
    protected TRCNode fNode;
    protected TRCProcessProxy fProcess;
    protected boolean _autoMonitoring = true;
    protected boolean _monitor = false;
    protected int _launchMode = 0;
    protected static final String getLauncherHandlersExtPoint = "launcherHandler";
    private static Map launcherHandlers;


    //~ Constructors -------------------------------------------------------------------------------

    /**
     * PIProcessListener constructor comment.
     */
    public PIProcessListener(TRCNode node) {
        super();
        fNode = node;
    }

    public PIProcessListener(TRCNode node, TRCProcessProxy process) {
        this(node);
        fProcess = process;
    }

    //~ Methods ------------------------------------------------------------------------------------

    /**
     * @param b
     */
    public void setAutoMonitoring(boolean b) {
        _autoMonitoring = b;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 5:37:22 PM)
     * @param mode int
     */
    public void setLaunchMode(int mode) {
        _launchMode = mode;
    }

    /**
     *
     * @return HashMap
     */
    public ArrayList getProcessVariableList() {
        return _processVariableList;
    }

    public synchronized void agentActive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        if (_monitoredAgents.contains(ag.getName())) {
            _monitor = false;
        } else {
            _monitoredAgents.add(ag.getName());
        }

        if (!_monitor) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,false);
		
        if (agentProxy == null) {
            return;
        }
        
        agentProxy.setRuntimeId(ag.getUUID());
        LoadersUtils.registerAgentInstance(agentProxy, ag);

        agentProxy.setAttached(true);
        agentProxy.setProfileFile(ag.getProfileFile());

        giveControl(agentProxy, true, true);
        
        if (agentProxy.isActive()) {
            final TRCAgentProxy agentTemp = agentProxy;

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        //update ui
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(agentTemp);
                        event.setType(ProfileEvent.START_MONITOR);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });
            return; //this is an attach scenario
        }

        agentProxy.setActive(true);

        try {
            ag.publishConfiguration();

            CustomCommand command = new CustomCommand();

            command.setData("APPLYFILTERS");
            ag.invokeCustomCommand(command);

            //			XMLTraceDataProcessor processor = (XMLTraceDataProcessor)agent.getDataProcessor();
            XMLTraceDataProcessor processor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(agentProxy);

            if (processor == null) {
                processor = new XMLTraceDataProcessor(agentProxy);

                //				agent.setDataProcessor(processor);
                //				LoadersUtils.loadRootEvent(processor);
                LoadersUtils.registerDataProcessor(agentProxy, processor);
            }

            if (agentProxy.isToProfileFile()) {
                processor.setProfileFileName(agentProxy.getProfileFile());
                processor.createWriter();

                if (processor.isNewProfileFile()) {
                    processor.writeXMLVersion(processor.getWriter());
                    processor.startTrace(processor.getWriter());
                }
            }

            agentProxy.setAttached(true);

            if (_autoMonitoring) {
                ag.startMonitoring(processor);
                agentProxy.setMonitored(true);
            }

            agentProxy.setCollectionData(true);

            command.setData("RESUME");
            ag.invokeCustomCommand(command);
            
            giveControl(agentProxy, false, true);
        } catch (InactiveAgentException exc) {
            exc.printStackTrace();
        }

        final TRCAgentProxy agentTemp = agentProxy;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.START_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }
    


    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 10:36:27 AM)
     * @param command com.ibm.etools.logging.tracing.control.Agent
     */
    public synchronized void agentInactive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,true);
        if (agentProxy == null) {
            return;
        }
  
        
        agentProxy.setActive(false);
        agentProxy.setAttached(false);
        agentProxy.setMonitored(false);
        fProcess.setActive(false);

        giveControl(agentProxy, true, false);
        
        final TRCAgentProxy agentTemp = agentProxy;
        

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.STOP_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }
    /**
     * called during an agentActive or agentInactive scenario
     * @param agentProxy
     * @param pre
     * @param active
     */
    
    protected void giveControl(TRCAgentProxy agentProxy, boolean pre, boolean active)
    {
    	//if the map has not been populated yet, populate it
    	if (launcherHandlers == null) initializeLauncherHandlers();
    	
    	Object[] launcherHandlersArray = launcherHandlers.values().toArray();
    	
    	for (int num = 0; num < launcherHandlersArray.length; num++)
    	{
    		ILauncherHandler type = (ILauncherHandler)launcherHandlersArray[num];
    		
    		try
			{
    			if (active)
    			{
    				if (pre)type.handlePreAgentActive(agentProxy);
    				else type.handlePostAgentActive(agentProxy);
    			}
    			else
    			{
    				//inactive
    				type.handleAgentInactive(agentProxy);
    			}
    			
			}
    		catch (Exception e)
			{
    			if(e.getMessage() != null)
    			{
    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
        			UIPlugin.getDefault().getLog().log(status);    				
    			}
    			//quietly catch any exceptions
    			//and ignore them
			}

    	}
    }
    
	/**
	 * called during a process Lauhced or process Exited scenario
	 * @param processProxy
	 * @param launched
	 */
    protected void giveControl(TRCProcessProxy processProxy, boolean launched)
    {

    	//if the map has not been populated yet, populate it
    	if (launcherHandlers == null) initializeLauncherHandlers();
    	
    	Object[] launcherHandlersArray = launcherHandlers.values().toArray();
    	
    	for (int num = 0; num < launcherHandlersArray.length; num++)
    	{
    		ILauncherHandler type = (ILauncherHandler)launcherHandlersArray[num];
    		
    		try
			{
        		if (launched)
        		{
        			type.handleProcessLaunched(processProxy);
        		}
        		else
        		{
        			type.handleProcessExited(processProxy);
        		}
    			
			}
    		catch (Exception e)
			{
    			if(e.getMessage() != null)
    			{
    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
        			UIPlugin.getDefault().getLog().log(status);    				
    			}
    			//quietly catch any exceptions
    			//and ignore them
			}

    	}
    }
    
    private void initializeLauncherHandlers()
    {
    	//reads the extension points, and creates instances of the GetControlType objects
    	//specified by the class attribute
    	launcherHandlers = new HashMap();
		
		IExtensionPoint point = UIPlugin.getDefault().getDescriptor().getExtensionPoint(getLauncherHandlersExtPoint);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];

				String id = elem.getAttribute("id");								

				ILauncherHandler getControlType = null;
				try
				{
					getControlType = (ILauncherHandler)elem.createExecutableExtension("class");
				}
				catch (Exception e)
				{
	    			if(e.getMessage() != null)
	    			{
	    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
	        			UIPlugin.getDefault().getLog().log(status);    				
	    			}
					
					e.printStackTrace();
				}
				
				if(getControlType != null)			
				{
					//put the instace of the class in a map
					launcherHandlers.put(id, getControlType);					
				}
			}
		}
    }

    public void dispose() {
        _monitoredAgents.clear();
        fNode = null;
        fProcess = null;
        _processVariableList.clear();
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 11:03:54 AM)
     * @return com.ibm.etools.perftrace.TRCAgent
     */
    /**
     * Invoked when an error is recieved from the agent.
     */
    public void error(Agent agent, String errorId, String errStr) {
        final String errorMsg = errStr;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, errorMsg, null);

                    ErrorDialog.openError(new Shell(), UIPlugin.getResourceString("LAUNCH_JAVA_PROBLEM_ERROR_"), UIPlugin.getResourceString("PROBLEM_WHILE_PROCESS_WARN_"), // no special message
                                          err);
                }
            });
    }

    public void handleCommand(Agent agent, CommandElement command) {
    }

    /*
     * return false if the agent should ignore the agentActive event
     */
    public void monitor(boolean monitor) {
        _monitor = monitor;
        _monitoredAgents.clear();
    }

    /**
     * Invoked when an agent requests to be monitored because this client
     * is currently monitoring another agent.
     */
    public void peerWaiting(Agent agent, Agent peer) {
        final TRCMonitor monitor = fNode.getMonitor();

        //get the right node for the peer
        final Process peerProcess = peer.getProcess();
        final Agent _peer = peer;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    TRCNode node = null;

                    try {
                        Node peerNode = peerProcess.getNode();

                        node = PDCoreUtil.createNode(monitor, peerNode.getName(), String.valueOf(peerNode.getConnection().getPort()));
                    } catch (InactiveProcessException exc) {
                        exc.printStackTrace();

                        return;
                    }

                    // Get/Create the proper process and agent for the peer 
                    TRCProcessProxy proxy = PDCoreUtil.createProcess(node, peerProcess);

                    proxy.setActive(true);

                    final TRCAgentProxy ag = PDCoreUtil.createAgent(proxy, _peer);

                    //		PIProcessListener pl = (PIProcessListener)ag.getAgentListener();
                    PIProcessListener pl = (PIProcessListener) LoadersUtils.locateAgentListener(ag);

                    if (pl == null) {
                        pl = new PIProcessListener(node, proxy);

                        //		   ag.setAgentListener(pl);
                        LoadersUtils.registerAgentListener(ag, pl);
                        _peer.addAgentListener(pl);
                    } else {
                        _peer.addAgentListener(pl);
                    }

                    pl.monitor(true);

                    // Set the reference to the agent
                    //		ag.setAgentInstance(_peer);
                    LoadersUtils.registerAgentInstance(ag, _peer);

                    //       XMLTraceDataProcessor aprocessor = (XMLTraceDataProcessor)ag.getDataProcessor();
                    XMLTraceDataProcessor aprocessor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(ag);

                    if (aprocessor == null) {
                        aprocessor = new XMLTraceDataProcessor(ag);

                        LoadersUtils.registerDataProcessor(ag, aprocessor);

                        //					   LoadersUtils.loadRootEvent(aprocessor);
                    }

                    if (_peer.isAttached()) {
                        try {
                            _peer.startMonitoring(aprocessor);
                        } catch (Exception exc) {
                        }

                        ag.setCollectionData(true);
                        ag.setActive(true);
                        ag.setAttached(true);
                        ag.setMonitored(true);
                        
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(ag);
                        event.setType(ProfileEvent.START_MONITOR);
                        UIPlugin.getDefault().notifyProfileEventListener(event);

                        return;
                    }

                    // Attach to the peer and start monitoring 		
                    final XMLTraceDataProcessor processor = aprocessor;

                    _peer.addAgentListener(new AgentListener() {
                            public void agentActive(Agent agent) {
                                try {
                                    if (!agent.isMonitored()) {
                                        ag.setCollectionData(true);
                                        agent.startMonitoring(processor);
                                        
                                    }
                                } catch (Exception e) {
                                    final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, e.toString(), null);

                                    Display.getDefault().syncExec(new Runnable() {
										public void run() {
		                                    ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("LAUNCH_JAVA_PROBLEM_ERROR_"), UIPlugin.getResourceString("PROBLEM_WHILE_PROCESS_WARN_"), err);
										}
                                    });

                                    e.printStackTrace();
                                }
                            }

                            public void agentInactive(Agent agent) {
                            	
                            }

                            public void error(Agent agent, String errorId, String message) {
                            }

                            public void handleCommand(Agent agent, CommandElement command) {
                            }
                        });

                    try {
                        _peer.attach();
                    } catch (InactiveAgentException exc) {
                        exc.printStackTrace();
                    } catch (InactiveProcessException exc) {
                        exc.printStackTrace();
                    }

                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(ag);
                    event.setType(ProfileEvent.UNSPECIFIED);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 9:07:12 AM)
     * @param result com.ibm.etools.logging.tracing.control.Process
     */
    public synchronized void processExited(Process result) {
        if (fProcess != null) {
            fProcess.setActive(false);
            giveControl(fProcess, false);
            
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/9/00 9:07:12 AM)
     * @param result com.ibm.etools.logging.tracing.control.Process
     */
    public synchronized void processLaunched(Process result) {
    	try {
            if ((result.getProcessId() == null) || result.getProcessId().equals("-1")) {
                return;
            }

            fProcess = createProcess(result);

             if (fProcess == null) {
                return;
            }

            //create agents
            Enumeration agents = result.listAgents();

            while (agents.hasMoreElements()) {
                createAgent((Agent) agents.nextElement(), fProcess);
            }
            
            giveControl(fProcess, true);
            
        } catch (InactiveProcessException ex) {
            String text = UIPlugin.getResourceString("RAC_CONFIG_ERROR_");
            final String msg = TString.change(UIPlugin.getResourceString("LAUNCH_ERROR_"), "%1", result.getName());
            final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

            Display.getDefault().syncExec(new Runnable() {
				public void run() {
		            ErrorDialog.openError(UIPlugin.getDefault().getViewer().getShell(), UIPlugin.getResourceString("TRACE_MSG"), msg, err);
				}
            });
            return;
        } catch (Exception exc) {
            exc.printStackTrace();
        }
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 12:50:32 PM)
     * @param agent com.ibm.etools.perftrace.TRCAgent
     */
    public Agent sendConfigurationToAgent(TRCAgentProxy agent, String host, String processId) {
        try {
            Node node = PDCoreUtil.profileConnect(host, String.valueOf(agent.getProcessProxy().getNode().getPort()));

            if (node == null) {
                return null;
            }

            Process p = node.getProcess(processId);

            if (p != null) {
                Agent a = p.getAgent(agent.getName());

                PDCoreUtil.setAgentConfiguration(agent, a);

                if ((a != null) && a.isActive()) {
                    a.publishConfiguration();

                    //			  a.invokeCustomCommand(new ApplyFiltersCommand());
                    CustomCommand command = new CustomCommand();

                    command.setData("APPLYFILTERS");
                    a.invokeCustomCommand(command);
                }

                return a;
            }
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }



    private void addEnvironment(TRCProcessProxy process) {
        for (int idx = 0; idx < _processVariableList.size(); idx++) {
            Variable var = (Variable) _processVariableList.get(idx);

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

            if (!var.getName().equals("UICLASSPATH")) {
                TRCEnvironmentVariable env = factory.createTRCEnvironmentVariable();

                env.setName(var.getName());
                env.setValue(var.getValue());
                process.getEnvironmentVariables().add(env);
            } else {
                process.setClasspath(var.getValue());
            }
        }
    }

    private TRCAgentProxy createAgent(Agent a, TRCProcessProxy process) {
        if (process == null) {
            return null;
        }

        TRCAgentProxy agent = PDCoreUtil.createAgent(process, a);

        //    agent.setAgentListener(this);
        LoadersUtils.registerAgentListener(agent, this);

        return agent;
    }

    private TRCProcessProxy createProcess(Process p, TRCNode pNode) {
        try {
            final TRCNode node = pNode;
            TRCMonitor monitor = node.getMonitor();

            String processId = p.getProcessId();

            if (processId == null) {
                processId = "-1";
            }

            String pName = p.getParameters();

            if (pName == null) {
                pName = "unknown";
            }

            String params = "";
            String vmparam = "";

            if (p.getExecutable().startsWith("java")) {
                String exeName = p.getParameters();

                if (exeName == null) {
                    exeName = "unknown";
                }

                if (!exeName.startsWith("-X")) { //log agents
                    exeName = "-X " + exeName;
                }
                
                exeName = exeName.trim();
                int idx = exeName.indexOf(" ");

                if ((idx != -1) && (idx < (exeName.length() + 1))) {
                    exeName = exeName.substring(idx + 1);
                }

                while (exeName.startsWith("-")) {
                    idx = findVMendIdx(exeName);

                    if (idx != -1) {
                        vmparam += (" " + exeName.substring(0, idx));
                        exeName = exeName.substring(idx + 1);
                    } else {
                        vmparam = exeName;
                        exeName = "";
                    }
                }

                exeName = exeName.trim();
                idx = exeName.indexOf(" ");

                if (idx != -1) {
                    params = exeName.substring(idx);
                    exeName = exeName.substring(0, idx);
                }

                pName = exeName;
            }

            String rId = ((ProcessImpl) p).getUUID();

            if (rId == null) {
                rId = "";
            }

            String fileName = new StringBuffer(monitor.getName()).append("_").append(node.getName()).append("_").append(TString.change(pName, " ", "")).append(processId).append("_").append(rId).append(".").append(TraceConstants.PROCESS_EXT).toString();

            String folderPath = node.eResource().getURI().toString();
            IPath path = new Path(folderPath);

            if (path.segmentCount() > 1) {
                folderPath = path.removeLastSegments(1).toString();
            }

            IPath filePath = new Path(folderPath).append(fileName);

            URI uri = URI.createURI(filePath.toString());

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();
            Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
            pDoc.setModified(true);
            EList pExt = pDoc.getContents();

            UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

//            //*** adding support for multiple files
//            SaveUtil.addDocument(pDoc);

            final TRCProcessProxy process = factory.createTRCProcessProxy();

            process.setPid(Integer.parseInt(processId));
            process.setRuntimeId(rId);
            process.setName(pName);
            process.setLocation(((ProcessImpl) p).getlocation());
            process.setNode(node);

            addEnvironment(process);

            process.setParameters(params.trim());
            process.setVmArguments(vmparam.trim());
            process.setLaunchMode(_launchMode);
            process.setActive(true);
            pExt.add(process);

            node.getProcessProxies().add(process);

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(null);
                        event.setType(ProfileEvent.UNSPECIFIED);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });

            _processVariableList.clear();

            return process;
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }

    /**
     * Insert the method's description here.
     * Creation date: (11/10/2000 11:03:54 AM)
     * @return com.ibm.etools.perftrace.TRCAgent
     */
    private TRCProcessProxy createProcess(Process p) {
        return createProcess(p, fNode);
    }

    private int findVMendIdx(String vmargs) {
        String space = " ";
        String quote = "\"";
        int startIdx = 0;
        int endIdx;
        int qIdx;

        endIdx = vmargs.indexOf(space);

        if (endIdx != -1) {
            qIdx = vmargs.substring(startIdx, endIdx).indexOf(quote);

            if (qIdx != -1) {
                startIdx = endIdx + vmargs.substring(endIdx + 1).indexOf(quote) + 1;
                endIdx = startIdx + findVMendIdx(vmargs.substring(startIdx + 1)) + 1;
            }
        }

        return endIdx;
    }
}
