package org.eclipse.hyades.internal.collection.framework;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import org.eclipse.hyades.internal.execution.file.FileServiceConstants;
import org.eclipse.hyades.execution.security.IClientHandler;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ManageFileCommand;
import org.eclipse.hyades.internal.execution.local.common.Message;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * @author Giridhar.S
 *
 * This is the class which is resposible for actual file transfer. It 
 * contains socket reading and writing methods, and methods for doing
 * server side file transfer operations.
 * 
 */
public class FileClientHandlerImpl extends Thread implements IClientHandler, FileServiceConstants, Constants {

	/* The socket which we receive */
	private Socket _socket;
	
	/*  The associated input and output streams */
	private InputStream _instrm;
	private OutputStream _outstrm;

	/* The data buffer */
	private byte[] _data;
	
	/* The various counters */
	private int byteCount ,count ;
	

	/**
	 *  Constructor
	 * @param s - Socket passed by the connection handler, for performing file transfer.
	 */
	public FileClientHandlerImpl(Socket s) {
		_socket = s;
		
		try {
			_instrm = s.getInputStream();
			_outstrm = s.getOutputStream();
		} catch (IOException e) {
		}
	}
	
	/**
	 *  Method to get the raw IP address of the client machine.
	 */
	public byte[] getRemoteAddress() {
		return _socket.getInetAddress().getAddress();
	}

	/**
	 * Method to read bytes off the socket input stream
	 * @param b - byte buffer to fill up
	 * @param off - starting offset
	 * @param len - length of the data to be read
	 * @return int - number of bytes read 
	 */
	public int read(byte[] b, int off, int len) {
		int bytesRead;
		
		try {
			bytesRead = _instrm.read(b, off,len);
		}
		catch (IOException ioe) {
			bytesRead = -1;
		}
		
		return bytesRead;
	}

	/**
	 * Method to write bytes into the socket output stream
	 * @param b - byte buffer to read from
	 * @param off - starting offset
	 * @param len - length of the data to be written
	 * @return int - 0 - success; -1 - failed 
	 */
	
	public int write(byte[] b, int off, int len) {
		try {
			_outstrm.write(b, off, len);
		}
		catch (IOException ioe) {
			return -1;
		}
		
		return 0;
		
	}
	
	/**
	 * A method to close the connection with the client
	 */
	public void closeConnection() {
		try {
			_instrm.close();
			_outstrm.close();
			
			_socket.close();
		}
		catch (IOException ioe) {
			if (!_socket.isClosed()) {
				try {
					_socket.close();
				} catch (IOException e) {
				}
			}
		}
	}
	
	/**
	 * Method to write bytes into the socket output stream
	 * @param b - byte buffer to read from
	 * @return int - 0 - success; -1 - failed 
	 */
	
	
	public int write (byte[] b)	{
		try {
			_outstrm.write(b);
		}
		catch (IOException ioe) {
			return RA_IO_ERROR;
		}
		
		return 0;
	}
	
	/**
	 * Method to read bytes off the socket input stream
	 * @param b - byte buffer to fill up
	 * @return int - number of bytes read 
	 */
	
	
	public int read (byte[] b)	{
		int bytesRead;
		try {
			bytesRead = _instrm.read(b);
		}
		catch (IOException ioe) {
			bytesRead = -1;
		}
		
		return bytesRead;
	}
	
	/**
	 * The get file method which transfers a file from the RAC machine to the client machine.
	 * @param fileName
	 * @return int - number of bytes transferred / error status
	 */
	
	private int getFile (String fileName){
		
		BufferedInputStream _fis = null;
		
		try {
			_data = new byte[RA_BUFFER_SIZE];
			_fis=new BufferedInputStream(new FileInputStream(fileName));

			/* Read the file bits and send them over the socket */
			while((count=_fis.read(_data)) != -1)	{
				write(_data, 0, count);
				byteCount += count;
			}
			_fis.close();
		
		} catch (Throwable e1) {
			// An Error has been encountered
			// Notify the Client by posting an Error Code
			try {
				_fis.close();
				
			} catch (IOException e2) {
				// Already in an error condition
			}
			
		} finally {
			closeConnection();
		}
		
		return 0;
	}
	
	/**
	 * A method to recieve a file from the client machine and put it on the RAC machine.
	 * @param destFileName - name of the file on the RAC machine.
	 * @return int - number of bytes read / error status
	 */
	private int putFile (String destFileName){
		
		BufferedOutputStream _fos = null;
		try {
			
			File destFile = new File(destFileName);

			// The path must be absolute
			if (!destFile.isAbsolute()) {
				throw new IOException();
			}
		
			// Create Deployment Directories
			String directoryName = destFile.getParent();		
			File dirFile = new File(directoryName);
			if (!dirFile.exists()) {
				boolean successful = dirFile.mkdirs();
				if (!successful) {
					throw new IOException();
				}
			}

			// Write the bits to the File
			File _file = new File(destFileName);
			_file.createNewFile();
			_fos=new BufferedOutputStream(new FileOutputStream(_file));
		
			/* We read the data which the client sends */
			_data = new byte[RA_BUFFER_SIZE];
			_socket.setSoTimeout(3000); /* A fairly large time out */
			while ((count = read(_data)) != -1)	{
				_fos.write(_data, 0, count);
				byteCount += count;
			}
			_fos.flush();
			_fos.close();
			
		} catch (Throwable e) {
			// An Error has been encountered
			// Notify the Client by posting an Error Code
			try {
				_outstrm.write(RA_IO_ERROR);
				_fos.close();
				
			} catch (IOException e2) {
				// Already in an error condition
			}
			
		} finally {
			closeConnection();
		}
		
		return 0;
	}
	
	/**
	 * Method to delete a file. This never gets called since RAC does the file deletion in native code.
	 * @param fileName
	 * @return 0
	 */
	
	private int deleteFile (String fileName){
		return 0;
	}
	
	public void run()	{
		
		_data = new byte[RA_BUFFER_SIZE];
		int offset = 0, result = -1;
		
		/* Read the request information that the client sends */
		/* We first read the bytes from the input stream */
		int count = read(_data);

		/* First 4 bytes will be the type of the operation */
		long operation = Message.readRALongFromBuffer(_data, offset);
		offset += sizeofLong;

		/* The next 4 bytes will be the length of the file name coming next */
		long len = Message.readRALongFromBuffer(_data, offset);
		offset += sizeofLong;

		/* The actual file name */
		String fileName = null;
		
		// bugzilla 65922 z/OS triggered change
		try {
			fileName = new String(_data, offset, (int)len, "UTF-8");
		}
		catch(Throwable e) {
			fileName=new String(_data, offset, (int)len);
		}
		
		
		switch ((int)operation)	{
		
		case (int) ManageFileCommand.GET:		/* Send the file to client workbench */
			result = getFile (fileName);
			break;
			
		case (int) ManageFileCommand.PUT:		/* Recieve a file from client workbench */
			result = putFile (fileName);
			break;
			
		case (int) ManageFileCommand.DELETE:	/* We are never going to come here, since RAC handles this */
			result = deleteFile(fileName);
			break;
		/* We have to just swallow the return status, because we cannot return it */	
		default:
		}
	}
}
