/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.resources.database.internal.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.hyades.resources.database.internal.DBMap;
import org.eclipse.hyades.resources.database.internal.dbmodel.Column;
import org.eclipse.hyades.resources.database.internal.dbmodel.Database;
import org.eclipse.hyades.resources.database.internal.dbmodel.Table;
/**
 * This class implements the DBMap interface.
 */
public class DBMapImpl implements DBMap {
	protected Map map;
	protected Database database;
	protected Map classesToSubclasses;
	protected List classTables;
	protected List attributeTables;
	protected List referenceTables;
	protected Table resourceTable;
	protected Table proxyTable;
	protected Table idTable;
	protected Map tablesToClasses;
	protected Map columnsToReferences;
	protected Map classesToMetadata;

	/**
	 * Constructor for DBMapImpl.
	 */
	public DBMapImpl() {
		super();
		map = new HashMap();
		classTables = new ArrayList();
		attributeTables = new ArrayList();
		referenceTables = new ArrayList();
		tablesToClasses = new HashMap();
		classesToSubclasses = new HashMap();
		columnsToReferences = new HashMap();
		classesToMetadata = new HashMap();
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#add(EModelElement,
	 *      Object)
	 */
	public void add(EModelElement element, Object rdbRepresentation) {
		map.put(element, rdbRepresentation);

		if (rdbRepresentation instanceof Database)
			database = (Database) rdbRepresentation;
		else if (element instanceof EClass) {
			computeSubclasses((EClass) element);
			Table table = ((ClassData) rdbRepresentation).getTable();
			classTables.add(table);
			tablesToClasses.put(table, element);
		} else if (element instanceof EReference) {
			Table table = ((ReferenceData) rdbRepresentation).getTable();

			EReference reference = (EReference) element;
			EReference opposite = reference.getEOpposite();

			if (referenceIsTable(reference, opposite)) {
				if (!referenceTables.contains(table))
					referenceTables.add(table);
			} else if (opposite == null || map.get(opposite) == null) {
				ReferenceData data = (ReferenceData) rdbRepresentation;
				Column column;

				if (reference.isMany())
					column = data.getSourceColumn();
				else
					column = data.getTargetColumn();

				columnsToReferences.put(column, reference);
			}
		} else if (element instanceof EAttribute) {
			Table table = ((AttributeData) rdbRepresentation).getTable();

			if (table != null)
				attributeTables.add(table);
		}
	}

	protected boolean referenceIsTable(EReference reference, EReference opposite) {
		if (reference.isMany() && !reference.isUnique())
			return true;

		if (opposite != null && opposite.isMany() && !opposite.isUnique())
			return true;

		if (opposite == null)
			return reference.isMany();
		else
			return reference.isMany() && opposite.isMany() && !reference.isTransient();
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#getDBRepresentation(EModelElement)
	 */
	public Object getDBRepresentation(EModelElement element) {
		return map.get(element);
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#getRDBDatabase()
	 */
	public Database getDatabase() {
		return database;
	}

	public List getClassTables() {
		return classTables;
	}

	public List getReferenceTables() {
		return referenceTables;
	}

	public List getAttributeTables() {
		return attributeTables;
	}

	/**
	 * Add this class as a subclass for the immediate superclasses.
	 */
	protected void computeSubclasses(EClass eClass) {
		List superClasses = eClass.getESuperTypes();

		for (int i = 0, l = superClasses.size(); i < l; i++)
			addSubclass((EClass) superClasses.get(i), eClass);
	}

	/**
	 * Add the class and subclass to the classesToSubclasses map.
	 */
	protected void addSubclass(EClass superClass, EClass subClass) {
		List subclasses = (List) classesToSubclasses.get(superClass);

		if (subclasses == null) {
			subclasses = new ArrayList();
			classesToSubclasses.put(superClass, subclasses);
		}

		if (!subclasses.contains(subClass))
			subclasses.add(subClass);
	}

	/**
	 * Get all subclasses for the given class.
	 */
	public List getAllSubclasses(EClass eClass) {
		List subclasses = new ArrayList();
		List direct = (List) classesToSubclasses.get(eClass);

		if (direct == null)
			return subclasses;

		for (int i = 0, l = direct.size(); i < l; i++) {
			EClass subclass = (EClass) direct.get(i);
			List indirect = getAllSubclasses(subclass);

			for (int j = 0, l2 = indirect.size(); j < l2; j++)
				if (!subclasses.contains(indirect.get(j)))
					subclasses.add(indirect.get(j));

			if (!subclasses.contains(subclass))
				subclasses.add(subclass);
		}

		return subclasses;
	}

	public Table getResourceTable() {
		return resourceTable;
	}

	public void setResourceTable(Table table) {
		resourceTable = table;
	}

	public EClass getClass(Table table) {
		return (EClass) tablesToClasses.get(table);
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#getReference(com.ibm.etools.rdbschema.RDBColumn)
	 */
	public EReference getReference(Column column) {
		return (EReference) columnsToReferences.get(column);
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#getProxyTable()
	 */
	public Table getProxyTable() {
		return proxyTable;
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#setProxyTable(com.ibm.etools.rdbschema.RDBTable)
	 */
	public void setProxyTable(Table table) {
		this.proxyTable = table;
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#getIdTable()
	 */
	public Table getIdTable() {
		return idTable;
	}

	/**
	 * @see org.eclipse.hyades.resources.database.internal.DBMap#setIdTable(org.eclipse.hyades.resources.database.internal.dbmodel.Table)
	 */
	public void setIdTable(Table table) {
		idTable = table;
	}

	public ClassMetadata getClassMetadata(EClass eClass) {
		if (classesToMetadata == null)
			classesToMetadata = new HashMap();
		ClassMetadata metadata = (ClassMetadata) classesToMetadata.get(eClass);
		if (metadata == null) {
			metadata = new ClassMetadata(eClass, this, new RDBHelper());
			classesToMetadata.put(eClass, metadata);
		}
		return metadata;
	}
} // DBMapImpl
