/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.wizard;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.loaders.util.InvalidXMLException;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.PDProjectViewer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceLocationUI;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.dialogs.WizardDataTransferPage;


/*
 * CONTEXT_ID xmli0000 for XML Trace Import wizard page
 */

/**
 *  Page 1 of the base resource import-from-zip Wizard.
 *
 *  Note that importing from .jar is identical to importing from .zip, so
 *  all references to .zip in this class are equally applicable to .jar
 *  references.
 */
public class ImportTracePage1 extends WizardDataTransferPage implements Listener, ModifyListener, SelectionListener, IRunnableWithProgress {
    //~ Instance fields ----------------------------------------------------------------------------

    protected Button sourceBrowseButton;
    protected ImportTraceRange _option;
    protected String fError;
    protected String fInputLog = "";
    protected TRCMonitor fMonitor;

    // widgets
    protected Text sourceNameField;
    protected TraceLocationUI _location;
    protected double fStartPercentage;
    protected double fStopPercentage;
    private Button _showExecFlow;
    private Button _showExecStatistic;
    private Button _showHeapStatistic;
    private Composite _detailsGroup;
    private Label _description;
    private Shell shell;
    private long logLength;
	private int collectionMode = TRCCollectionMode.HEAP_STATISTICS_ONLY;


    //~ Constructors -------------------------------------------------------------------------------

    /**
     *  Creates an instance of this class
     * @param aWorkbench IWorkbench
     * @param selection IStructuredSelection
     */
    public ImportTracePage1(IWorkbench aWorkbench, IStructuredSelection selection) {
        super("xmlTraceFileImportPage1");
        setTitle(UIPlugin.getResourceString("IMPORT_XML_TRACE_PTITLE"));
        setDescription(UIPlugin.getResourceString("IMPORT_XML_TRACE_PDESC"));
        setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_UI_WZ_IMPORT_PROF));

        shell = aWorkbench.getActiveWorkbenchWindow().getShell();
    }

    //~ Methods ------------------------------------------------------------------------------------

    /** (non-Javadoc)
     * Method declared on IDialogPage.
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NULL);

        composite.setLayout(new GridLayout());
        composite.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_FILL | GridData.HORIZONTAL_ALIGN_FILL));
        composite.setSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

        createSourceGroup(composite);

        createSpacer(composite);

        createBoldLabel(composite, UIPlugin.getResourceString("STR_IMPORT_XML_PSOURCEQ"));

        _location = new TraceLocationUI();
        _location.createControl(composite);
        _location.getLocationLabel().setText(UIPlugin.getResourceString("STR_IMPORT_FOLDER"));

        createSpacer(composite);

        _option = new ImportTraceRange(this);
        _option.createControl(composite);

        addStatisticOption(composite);

        createSpacer(composite);

        _option.getWholeFileBtn().addSelectionListener(this);
        _option.getInRangeBtn().addSelectionListener(this);

        restoreWidgetValues();
        updateWidgetEnablements();
        setPageComplete(determinePageCompletion());

        setControl(composite);

        PDProjectExplorer projectExplorer = null;
        PDProjectViewer projectViewer = null;

        IWorkbenchPage page = UIPlugin.getActiveWorkbenchWindow().getActivePage();

        if ((page != null) && page.getPerspective().getId().equals(PDPerspective.ID_TRACE_PERSPECTIVE)) {
            projectExplorer = UIPlugin.getDefault().getViewer();
        }

        IProject selectionProject = null;
        TRCMonitor selectionMonitor = null;

        if (projectExplorer != null) {
            projectViewer = (PDProjectViewer)projectExplorer.getViewer();
        }

        if (projectViewer != null) {
            selectionProject = projectViewer.getSelectionProject();
            selectionMonitor = projectViewer.getSelectionMonitor();
        }

        _location.getLocation().addModifyListener(this);
        _location.getMonitor().addModifyListener(this);
        sourceNameField.addModifyListener(this);

        if (selectionProject != null) {
            _location.setLocation(selectionProject.getName());
        }

        if (selectionMonitor != null) {
            _location.setMonitor(selectionMonitor.getName());
        }
    }

    /**
     * The Finish button was pressed.  Try to do the required work now and answer
     * a boolean indicating success.  If <code>false</code> is returned then the
     * wizard will not close.
     *
     * @return boolean
     */
    public boolean finish() {
    	fInputLog = sourceNameField.getText().trim();
        File log = new File(sourceNameField.getText().trim());

        if (!log.exists()) {
            fError = UIPlugin.getResourceString("STR_ERROR_XML_NOTFOUND");
            fError = HyadesUtil.change(fError, "%1", fInputLog);

            String errMsg = UIPlugin.getResourceString("STR_ERROR_LOAD_XML");

            Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);

            ErrorDialog.openError(getShell(), UIPlugin.getResourceString("TRACE_MSG"), errMsg, err1);

            return false;
        }

        if (!log.canRead()) {
            fError = UIPlugin.getResourceString("STR_ERROR_XML_READFILE");
            fError = HyadesUtil.change(fError, "%1", fInputLog);

            String errMsg = UIPlugin.getResourceString("STR_ERROR_LOAD_XML");

            Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);

            ErrorDialog.openError(getShell(), UIPlugin.getResourceString("TRACE_MSG"), errMsg, err1);

            return false;
        }

        fError = null;

        if (_option.getInRangeSelection()) {
            fStartPercentage = _option.getStartRange();
            fStopPercentage = _option.getStopRange();
        } else {
            fStartPercentage = 0;
            fStopPercentage = 100;
        }

        return loadXMLTrace();
    }

    /**
     *  Handle all events and enablements for widgets in this dialog
     *
     * @param event Event
     */
    public void handleEvent(Event event) {
        if (event.widget == sourceBrowseButton) {
            handleSourceBrowseButtonPressed();
        }
    }

    /**
     * Sent when the text is modified.
     *
     * @param e an event containing information about the modify
     */
    public void modifyText(ModifyEvent e) {
        setPageComplete(determinePageCompletion());
    }

    public void run(IProgressMonitor mon) {
        Display d = Display.getDefault();

        d.syncExec(new Runnable() {
                public void run() {
                    if (_showHeapStatistic.getSelection()) {
                        collectionMode =  TRCCollectionMode.HEAP_STATISTICS_ONLY;
                    } else if (_showExecStatistic.getSelection()) {
                        collectionMode = TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY;
                    } else if (_showExecFlow.getSelection()) {
                        collectionMode = TRCCollectionMode.HEAP_AND_EXECUTION_FULL;
                    }
                }
            });

		try {
			File log = new File(fInputLog);
			
			if(!PDCoreUtil.isZipFile(fInputLog))
			{
				InputStream readStream = new BufferedInputStream(new FileInputStream(log));
				logLength = log.length();
				XMLLoader processor = new XMLLoader(fMonitor);
				processor.setCollectionMode(collectionMode);
				importFile(mon, processor, log, readStream);
			}
			else
			{
				ZipFile zf = new ZipFile(log);

				Enumeration entries = zf.entries();
				int iter = 0;

				while (entries.hasMoreElements()) {
					ZipEntry zipEntry = (ZipEntry) entries.nextElement();
					logLength = zipEntry.getSize();
					XMLLoader processor = new XMLLoader(fMonitor);
					processor.setCollectionMode(collectionMode);
					importFile(mon, processor, log, zf.getInputStream(zipEntry));
				}
			}
		} catch (IOException e) {
			fError= UIPlugin.getResourceString("STR_IMPORT_IOEXCEPTION_ERROR_");
			fError = HyadesUtil.change(fError, "%1", e.getMessage());
		}
    }

    public void widgetDefaultSelected(SelectionEvent e) {
    }

    public void widgetSelected(SelectionEvent e) {
        if (e.widget == _showExecStatistic) {
            _description.setText(UIPlugin.getResourceString("STR_ANALYZE_EXE_TIME"));
        } else if (e.widget == _showHeapStatistic) {
            _description.setText(UIPlugin.getResourceString("STR_ANALYZE_HEAP"));
        } else if (e.widget == _showExecFlow) {
            _description.setText(UIPlugin.getResourceString("Show Execution flow graphical details."));
        } else {
            setPageComplete(determinePageCompletion());
        }
    }

    protected boolean allowNewContainerName() {
        return true;
    }

    /**
     * Creates a new button with the given id.
     * <p>
     * The <code>Dialog</code> implementation of this framework method
     * creates a standard push button, registers for selection events
     * including button presses and registers
     * default buttons with its shell.
     * The button id is stored as the buttons client data.
     * Note that the parent's layout is assumed to be a GridLayout and
     * the number of columns in this layout is incremented.
     * Subclasses may override.
     * </p>
     *
     * @param parent the parent composite
     * @param id the id of the button (see
     *  <code>IDialogConstants.*_ID</code> constants
     *  for standard dialog button ids)
     * @param label the label from the button
     * @param defaultButton <code>true</code> if the button is to be the
     *   default button, and <code>false</code> otherwise
     */
    protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
        // increment the number of columns in the button bar
        ((GridLayout) parent.getLayout()).numColumns++;

        Button button = new Button(parent, SWT.PUSH);

        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);

        data.heightHint = convertVerticalDLUsToPixels(IDialogConstants.BUTTON_HEIGHT);
        data.widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH + 2);
        button.setLayoutData(data);

        button.setData(new Integer(id));
        button.setText(label);

        if (defaultButton) {
            Shell shell = parent.getShell();

            if (shell != null) {
                shell.setDefaultButton(button);
            }

            button.setFocus();
        }

        button.setFont(parent.getFont());

        return button;
    }

    /**
     *  Create the import source specification widgets
     */
    protected void createSourceGroup(Composite parent) {
        Composite sourceContainerGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();

        layout.numColumns = 3;
        sourceContainerGroup.setLayout(layout);
        sourceContainerGroup.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));

        new Label(sourceContainerGroup, SWT.NONE).setText(UIPlugin.getResourceString("STR_IMPORT_XML_SOURCE"));

        // source name entry field
        sourceNameField = new Text(sourceContainerGroup, SWT.BORDER);

        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);

        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        sourceNameField.setLayoutData(data);

        // source browse button
        sourceBrowseButton = new Button(sourceContainerGroup, SWT.PUSH);
        sourceBrowseButton.setText(UIPlugin.getResourceString("BROWSE"));
        sourceBrowseButton.addListener(SWT.Selection, this);
        sourceBrowseButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));

        sourceNameField.setFocus();

        org.eclipse.ui.help.WorkbenchHelp.setHelp(parent, UIPlugin.getPluginId() + ".xmli0000");
    }

    /**
     * Returns whether this page is complete. This determination is made based upon
     * the current contents of this page's controls.  Subclasses wishing to include
     * their controls in this determination should override the hook methods
     * <code>validateSourceGroup</code> and/or <code>validateOptionsGroup</code>.
     *
     * @return <code>true</code> if this page is complete, and <code>false</code> if
     *   incomplete
     * @see #validateSourceGroup
     * @see #validateOptionsGroup
     */
    protected boolean determinePageCompletion() {
        boolean complete = validateSourceGroup() && validateDestinationGroup() && validateOptionsGroup();

        // Avoid draw flicker by not clearing the error
        // message unless all is valid.
        if (complete) {
            setErrorMessage(null);
        }

        return complete;
    }

    /**
     *  Open an appropriate source browser so that the user can specify a source
     *  to import from
     */
    protected void handleSourceBrowseButtonPressed() {
        String currentSource = this.sourceNameField.getText();

        FileDialog dlg = new FileDialog(sourceNameField.getShell());

        dlg.setFilterPath(currentSource);
        dlg.setFilterExtensions(new String[] { "*.trcxml", "*.*" });
        dlg.open();

        String fileName = dlg.getFileName();

        if ((fileName != null) && !fileName.equals("")) {
            sourceNameField.setText(dlg.getFilterPath() + File.separator + fileName);
        }
    }

    /**
     * Check if widgets are enabled or disabled by a change in the dialog.
     * @param event Event
     */
    protected void updateWidgetEnablements() {
        boolean pageComplete = determinePageCompletion();

        setPageComplete(pageComplete);

        if (pageComplete) {
            setMessage(null);
        }

        super.updateWidgetEnablements();
    }

    /**
     * Returns whether this page's options group's controls currently all contain
     * valid values.
     * <p>
     * The <code>WizardDataTransferPage</code> implementation of this method returns
     * <code>true</code>. Subclasses may reimplement this hook method.
     * </p>
     *
     * @return <code>true</code> indicating validity of all controls in the options
     *   group
     */
    protected boolean validateOptionsGroup() {
        return (!_location.getLocation().getText().trim().equals("") && !_location.getMonitor().getText().trim().equals("") && (_option.getWholeFileSelection() || (_option.getInRangeSelection() && (_option.checkRegionValue() == null))));
    }

    /**
     * Returns whether this page's source specification controls currently all
     * contain valid values.
     * <p>
     * The <code>WizardDataTransferPage</code> implementation of this method returns
     * <code>true</code>. Subclasses may reimplement this hook method.
     * </p>
     *
     * @return <code>true</code> indicating validity of all controls in the
     *   source specification group
     */
    protected boolean validateSourceGroup() {
        return !sourceNameField.getText().trim().equals("");
    }


    //public void run1(IProgressMonitor mon)
    //{
    //	final XMLLoader processor=new XMLLoader(fMonitor);
    //
    //	Display d= Display.getDefault(); 
    //	d.asyncExec(new Runnable() {
    //	public void run()
    //	{
    //		if(_showHeapStatistic.getSelection())
    //			processor.setCollectionMode(TRCCollectionMode.HEAP_STATISTICS_ONLY);
    //		else if(_showExecStatistic.getSelection())
    //			processor.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY);
    //		else if(_showExecFlow.getSelection())
    //			processor.setCollectionMode(TRCCollectionMode.EXECUTION_FULL);
    //	}});
    //    
    //	File log = new File(fInputLog);
    //	FileInputStream readStream = null;
    //	
    //	try {
    //		int logLength = (int)log.length();
    //		
    //		int l=1, lastValidLine=1;
    //		boolean cancelOperation = false;
    //		try {
    //			readStream = new FileInputStream(log);			  
    //			double startLength = logLength/100;
    //			startLength=startLength*fStartPercentage;
    //			
    //			double endLength = logLength;
    //			if(fStopPercentage != 100)
    //			{
    //				endLength = endLength/100;
    //				endLength = endLength *fStopPercentage; 
    //			}
    //	
    //			mon.beginTask(fInputLog, (int)(endLength-startLength));
    //			
    //			long position = 0;
    //			int bytesRead=0;
    //			int bSize = 65536;  //length of each pipe
    //			int bufferSize = bSize; //total size of pipes.
    //			byte[] buffer=new byte[bufferSize];
    //			byte[] forwardBuffer=new byte[bufferSize];
    //			
    //			
    //			int forwardOffset=0;
    //			int offset=0;
    //			while(bytesRead>-1) {   
    //				
    //				if (mon.isCanceled())
    //				{
    //					readStream.close();
    //					cancelOperation = true;
    //					throw new OperationCanceledException();
    //				}
    //					
    //				offset=0;
    //				bytesRead=readStream.read(buffer);
    //	
    //readLine:		while(offset<bytesRead ) {
    //						try{
    //							if(position>)
    //								if(position > endLength && endLength != log.length())
    //								{         
    //									byte end[] = (new String("</TRACE>")).getBytes();
    //									processor.loadEvent(end, 8, true);  
    ////									processor.resetDeferredData();
    //									mon.done();
    //									readStream.close();										                                										
    //									return;
    //                                	
    //								}	
    //								processor.loadEvent(forwardBuffer, forwardOffset, loadToModel);  
    //							}catch(InvalidXMLException e){
    //								mon.worked(forwardOffset);
    ////								processor.resetDeferredData();
    //								readStream.close();								
    //								position-=forwardOffset;
    //							}
    //							catch(java.lang.OutOfMemoryError me) {
    //								readStream.close();								
    ////								processor.resetDeferredData();								
    //								handleOutOfMemoryError();
    //								throw me;
    //							}
    //						}
    //						lastValidLine=l;
    //						mon.worked(forwardOffset);
    //						forwardOffset=0;
    //				fError = null;
    //			}
    //					
    //			if(lastValidLine!=l) {
    //				
    //				readStream.close();				
    //				fError = UIPlugin.getResourceString("STR_EOF_REACHED_ERROR_");
    //							fError = TString.change(fError, "%1", fInputLog);
    //							fError = TString.change(fError, "%2", String.valueOf(lastValidLine));
    //							
    //							return;
    //				
    //			}
    //		}
    //		catch(IOException e)
    //		{
    //			fError= UIPlugin.getResourceString("STR_IMPORT_IOEXCEPTION_ERROR_");
    //			fError = TString.change(fError, "%1", e.getMessage());
    //		}
    //		catch(Exception e) {
    //			//System.out.println(e.getMessage());
    //			//e.printStackTrace();
    //			
    //			if(cancelOperation)
    //				fError = "";
    //			else    
    //				fError = UIPlugin.getResourceString("STR_IMPORT_EXCEPTION_ERROR_");
    //								fError = TString.change(fError, "%1", fInputLog);
    //								fError = TString.change(fError, "%2", String.valueOf(lastValidLine));
    //								fError = TString.change(fError, "%3", e.getMessage());
    //		}
    //		
    //		if(readStream != null)
    //		{
    //			try {
    //				readStream.close();		
    //			}
    //			catch(IOException ioexc)
    //			{
    //			}
    //		}
    //		
    ////		d.asyncExec(new Runnable() {
    ////		public void run()
    ////		{
    ////			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
    ////			event.setSource(null);
    ////			event.setType(ProfileEvent.UNSPECIFIED);
    ////			UIPlugin.getDefault().notifyProfileEventListener(event);	
    ////		}});
    //		
    //		mon.done();
    //	}
    //	finally {
    //		if(readStream != null)
    //		{
    //			try {
    //				readStream.close();		
    //			}
    //			catch(IOException ioexc)
    //			{
    //			}
    //		}
    //		mon.done();
    //	}
    //}


    /**
     *
     */
    private void addStatisticOption(Composite parent) {
        GridLayout layout;
        GridData data;

        Group _showExecGroup = new Group(parent, SWT.NULL);

        data = GridUtil.createHorizontalFill();
        data.horizontalSpan = 2;
        _showExecGroup.setLayoutData(data);
        layout = new GridLayout();
        layout.numColumns = 1;
        _showExecGroup.setLayout(layout);

        _showHeapStatistic = new Button(_showExecGroup, SWT.RADIO);
        _showHeapStatistic.setText(UIPlugin.getResourceString("HEAP_STAT"));
        _showExecStatistic = new Button(_showExecGroup, SWT.RADIO);
        _showExecStatistic.setText(UIPlugin.getResourceString("EXEC_STAT"));
        _showExecFlow = new Button(_showExecGroup, SWT.RADIO);
        _showExecFlow.setText(UIPlugin.getResourceString("FULL_DATA"));

        _description = new Label(parent, SWT.WRAP);
        data = new GridData();
        data.horizontalIndent = 20;
        data.widthHint = 375;
        _description.setLayoutData(data);

        _description.setLayoutData(data);
        _description.setText(UIPlugin.getResourceString("STR_ANALYZE_HEAP"));

        _showHeapStatistic.setSelection(true);

        _showHeapStatistic.addSelectionListener(this);
        _showExecStatistic.addSelectionListener(this);
        _showExecFlow.addSelectionListener(this);
    }

    private String createModel(TRCMonitor monitor, String inputLog) {
        fMonitor = monitor;
        fInputLog = inputLog;

        try {
            getWizard().getContainer().run(true, true, this);
        } catch (InterruptedException e) {
            return "";
        } catch (InvocationTargetException e) {
            return e.getTargetException().getMessage();
        }

        return fError;
    }

    private void handleOutOfMemoryError() {
        /* Inform the user we are running out of memory */
        String text = UIPlugin.getResourceString("IMPORT_OUT_OF_MEMORY_ERROR_");

        ;

        //System.out.println(text);
        if (shell != null) {
            final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

            shell.getDisplay().asyncExec(new Runnable() {
                    public void run() {
                        ErrorDialog.openError(shell, UIPlugin.getResourceString("TRACE_MSG"), UIPlugin.getResourceString("OUT_OF_MEMORY_ERROR_TITLE"), err);
                    }
                });
        }
    }

    private void importFile(IProgressMonitor mon, final XMLLoader processor, File log, InputStream readStream) throws OutOfMemoryError {
        try {
            boolean cancelOperation = false;
            int l = 1;
            int lastValidLine = 1;

            try {
                boolean done = false;
                int bSize = 65536; //length of each pipe
                int bufferSize = bSize; //total size of pipes.
                byte[] buffer = new byte[bufferSize];
                byte[] forwardBuffer = new byte[bufferSize];

                long position = 0;
                int bytesRead = 0;
                double startLength = logLength / 100;

                startLength = startLength * fStartPercentage;

                double endLength = logLength;

                if (fStopPercentage != 100) {
                    endLength = endLength / 100;
                    endLength = endLength * fStopPercentage;
                }

                mon.beginTask(fInputLog, (int) (endLength - startLength));

                int forwardOffset = 0;
                int offset = 0;

                while (bytesRead > -1) {
                    if (mon.isCanceled()) {
                        readStream.close();
                        cancelOperation = true;
                        throw new OperationCanceledException();
                    }

                    offset = 0;
                    bytesRead = readStream.read(buffer);

readLine: 
                    while (offset < bytesRead) {
                        forwardBuffer[forwardOffset++] = buffer[offset];

                        if (forwardOffset == bufferSize) {
                            byte[] tempForwardBuffer = forwardBuffer;

                            bufferSize = bufferSize + bSize;
                            forwardBuffer = new byte[bufferSize];
                            System.arraycopy(tempForwardBuffer, 0, forwardBuffer, 0, forwardOffset);

                            continue readLine;
                        }

                        if (buffer[offset++] == '\n') {
                            l++;

                            if (forwardBuffer[1] != '?') {
                                try {
                                    position += forwardOffset;

                                    if ((position > endLength) && (endLength != log.length())) {
                                        byte[] end = (new String("</TRACE>")).getBytes();

                                        processor.loadEvent(end, 8, true);

                                        // 									processor.resetDeferredData();
                                        mon.done();
                                        readStream.close();

                                        return;
                                    }

                                    boolean loadToModel = loadLine(l, position, startLength, endLength);

                                    processor.loadEvent(forwardBuffer, forwardOffset, loadToModel);

                                    //								if(processor.isProcessingElement()) {
                                    //										processor.reset();
                                    //										mon.worked(forwardOffset);
                                    //										position-=forwardOffset;
                                    //										continue readLine;
                                    //								}
                                } catch (InvalidXMLException e) {
                                    mon.worked(forwardOffset);

                                    //								processor.resetDeferredData();
                                    readStream.close();
                                    position -= forwardOffset;

                                    continue readLine;
                                } catch (java.lang.OutOfMemoryError me) {
                                    readStream.close();

                                    //								processor.resetDeferredData();								
                                    handleOutOfMemoryError();
                                    throw me;
                                }
                            }

                            lastValidLine = l;
                            mon.worked(forwardOffset);
                            forwardOffset = 0;
                        }
                    }

                    fError = null;
                }

                if (lastValidLine != l) {
                    readStream.close();
                    fError = UIPlugin.getResourceString("STR_EOF_REACHED_ERROR_");
                    fError = HyadesUtil.change(fError, "%1", fInputLog);
                    fError = HyadesUtil.change(fError, "%2", String.valueOf(lastValidLine));

                    return;
                }
            } catch (IOException e) {
                fError = UIPlugin.getResourceString("STR_IMPORT_IOEXCEPTION_ERROR_");
                fError = HyadesUtil.change(fError, "%1", e.getMessage());
            } catch (Exception e) {
                //System.out.println(e.getMessage());
                //e.printStackTrace();
                if (cancelOperation) {
                    fError = "";
                } else {
                    fError = UIPlugin.getResourceString("STR_IMPORT_EXCEPTION_ERROR_");
                }

                fError = HyadesUtil.change(fError, "%1", fInputLog);
                fError = HyadesUtil.change(fError, "%2", String.valueOf(lastValidLine));
                fError = HyadesUtil.change(fError, "%3", e.getMessage());
            }

            if (readStream != null) {
                try {
                    readStream.close();
                } catch (IOException ioexc) {
                }
            }

            mon.done();
        } finally {
            if (readStream != null) {
                processor.cleanUp();

                try {
                    readStream.close();
                } catch (IOException ioexc) {
                }
            }

            mon.done();
        }
    }

    private boolean loadLine(int line, double position, double startLength, double endLength) {
        if (line <= 7) {
            return true; //trace start events
        }

        if ((position >= startLength) && (position <= endLength)) {
            return true;
        }

        return false;
    }

    /**
     * Insert the method's description here.
     * Creation date: (8/18/2001 2:50:10 PM)
     * @return boolean
     */
    private boolean loadXMLTrace() {
        IWorkbenchPage page = UIPlugin.getActivePage();

        if (page == null) {
            return false;
        }

        //create destination container
        IContainer res = PDCoreUtil.createContainer(new Path(_location.getLocation().getText().trim()));

        if (res == null) {
            return false;
        }

        //create monitor
        TRCMonitor mon = PDCoreUtil.createMonitor(res, _location.getMonitor().getText().trim());

        fError = createModel(mon, sourceNameField.getText().trim());

        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

        event.setSource(null);
        event.setType(ProfileEvent.UNSPECIFIED);
        UIPlugin.getDefault().notifyProfileEventListener(event);

        if (fError != null) {
            if (!fError.equals("")) {
                String errMsg = UIPlugin.getResourceString("STR_ERROR_LOAD_XML");

                Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);

                ErrorDialog.openError(getShell(), UIPlugin.getResourceString("TRACE_MSG"), errMsg, err1);
            }

            return false;
        }

        return true;
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    static final class FileInfo {
        InputStream inputStream;
        long fileLength;
    }

	public void setTraceFile(String fullPath) {
		sourceNameField.setText(fullPath);
	}
}
