/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.UnsupportedEncodingException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;

/**
 * @author Ernest Jessee
 */
public class TestExecutionHarnessExecutorStub extends ExecutorStub
{

	public void launch(IProgressMonitor progressMonitor) throws ExecutionComponentStateException {
		
		super.launch(progressMonitor);
		
		//we will only allow 60 failed attempts then we will abort.
		int trialCount=60;
		
		/*
		 * Begin the task of this launch method; allocate 2 work units for any potential attempts and
		 * allocate 5 work units for each data processor.
		 */ 
		progressMonitor.beginTask("", (trialCount * 2) + ((this.getDataProcessors().length) * 15)); //$NON-NLS-1$
		
		//get the process ID
		String pid=getPid();
		
		//as we loop through this initialization logic, it is important to track how many agents
		//we have initialized.  this is where we reset teh counter.
		resetAgentInitCount();
			
		try 
		{
			//get our node.
			Node node=((SessionStub)getSessionContext()).getAgent().getProcess().getNode();
			
			//we have a list of valid dataprocessors already in our scope -- placed there by the harness
			IDataProcessor[] dataProcessors = getDataProcessors();
			
			//no dataprocessors is a valid option
			if(dataProcessors!=null)
			{			
				//now we will loop until 1 of 2 conditions is met
				//1.  we have failed "trialcount" times.
				//2.  we have initialized an agent for each valid dataprocessor
				while(trialCount>0 && this.agentInitCount<getDataProcessors().length) 
				{	
					
					// poll for cancel status and track progress
					if (progressMonitor.isCanceled()) {
						return;
					} 
					progressMonitor.worked(2);
					
					//find the our process on the node.
					Process proc = findProcess(pid, node);	
					if(proc!=null)
					{
					 	synchronized(proc) 
						{
							
							for(int j=0; j<getDataProcessors().length; j++)
							{
								
								// poll for cancel status and track progress
								if (progressMonitor.isCanceled()) {
									return;
								} 
								progressMonitor.worked(5);
								
								IExecutionHarnessDataProcessor execDP = (IExecutionHarnessDataProcessor)getDataProcessors()[j];
								
								if(execDP.getControlAgent()!=null)
								    continue;
								
								//find the agent initiated by the runner. The 1st one, is always the XMLExecutionHistoryDataProcessor
								//even if that dataprocessor is not active.  (it's control agent is the bootstrapper by default.
								//this initial dp is started from the runner's main()
								//the remainder of the control agents are instatiated using the information sent by the harness
								//using setAgentData()								 
								Agent agent = findAgent(proc, "tester", execDP.getName()); //$NON-NLS-1$ //$NON-NLS-2$
								
								//for if this dataprocessor has already been initiated with a controlagent or if we didn't find and agent
								//then we will not try to continue for this pass
								if(agent == null)
								{
								    try 
								    {
								        Thread.sleep(100);
                                    } 
								    catch (InterruptedException e1) 
								    {}
								    continue;
								}
								
								// poll for cancel status and track progress
								if (progressMonitor.isCanceled()) {
									return;
								} 
								progressMonitor.worked(5);
								
								
								//now that we have our agent, our proccess and a willing dataprocessor, we can proceed.
								setupDataProcessor(agent,proc,execDP);
								
								// poll for cancel status and track progress
								if (progressMonitor.isCanceled()) {
									return;
								}
								progressMonitor.worked(5);
								
								
								//now we want to attach the appropriate controllistener to the agent's control channel.
								setupControlListener(getAgentListener(),agent);
													
							}
						}	
					}
					else
					{
						//if we did not get a dataprocessor, then we need to wait a second before we continue.
						//I HATE sleeps.  Maybe there's a better way.  If we don't wait, we max out trial count with nothing.
						try{Thread.sleep(1000);}catch (InterruptedException e2){}
					
						trialCount--;
					}
						
				}
				
				
				IExecutionHarnessDataProcessor theControllerDP = (IExecutionHarnessDataProcessor)getDataProcessors()[0]; 
				if(theControllerDP != null) {
				    Agent agent = theControllerDP.getControlAgent();
				    if (agent != null) {
				        agent.invokeCustomCommand(createResumeCommand());
				    }
				}
				
						
			}
			else
			{
				Agent agent=null;
				while(trialCount>0 && agent==null)
				{
					Process proc = findProcess(pid, node);
					if(proc!=null)
					{				
						agent = findAgent(proc,"tester", XMLExecutionDataProcessor.IID); //$NON-NLS-1$
						setupControlListener(getAgentListener(),agent);
										
					}	
					else
					{
						//I explained this mess above...
						try{Thread.sleep(1000);}catch (InterruptedException e2){}
						
						trialCount--;
					}
				
				}
				
				if(agent!=null)
					agent.invokeCustomCommand(createResumeCommand());
				
			}
		}
		catch(InactiveProcessException e) 
		{
			ExecutionHarnessPlugin.getDefault().logError(e);
		}
		catch(NotConnectedException e) 
		{	
			ExecutionHarnessPlugin.getDefault().logError(e);
		} 
		catch (UnsupportedEncodingException e)
		{
			ExecutionHarnessPlugin.getDefault().logError(e);
		} 
		catch (InactiveAgentException e)
		{
			ExecutionHarnessPlugin.getDefault().logError(e);
		}
		finally {
			progressMonitor.done();
		}
		
	}
	
	/**
	 * launches the executor
	 */
	public void launch() throws ExecutionComponentStateException {	
		this.launch(new NullProgressMonitor());
	}

	/**
	 * creates a CustomCommand which is sent to the skeleton to start it.
	 * @return
	 */
	private CustomCommand createResumeCommand()
	{
		BinaryCustomCommand resumeCommand=new BinaryCustomCommand();
		resumeCommand.setData(IControlMessage.START);				
		return resumeCommand;
	}
	
	
	/**
	 *called to acquire the agentlistener which should be registered to listen to the rac control channel
	 * @return
	 */
	protected AgentListener getAgentListener()
	{
		return new AgentListener() 
		{

			public void error(Agent agent, String errorId, String errorMessage) 
			{
			}

			public void handleCommand(Agent agent, CommandElement command) 
			{
				if(command instanceof CustomCommand) {
					String errorMessage=((CustomCommand)command).getData();
					System.out.println("recieving message: "+errorMessage); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}

			public void agentActive(Agent agent) 
			{
			}

			public void agentInactive(Agent agent) 
			{	
			}									
		};

	}
	
	/**
	 * called by launch method to initialize an IExecutiondAtaProcessor
	 * @param agent
	 * @param proc
	 * @param dataProcessor
	 */
	protected void setupDataProcessor(Agent agent, Process proc, IExecutionHarnessDataProcessor dataProcessor)
	{	
		dataProcessor.setControlAgent(agent);
		dataProcessor.setProcess(proc);
		dataProcessor.init();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#performControlEvent(java.lang.String)
	 */
	public String performControlEvent(String controlEvent, String[] params)
	{
		return ""; //$NON-NLS-1$ //$NON-NLS-2$
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#supportsControlEvent(java.lang.String)
	 */
	public boolean supportsControlEvent(String controlEvent)
	{
		return true;
	}

}
