/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.execution.harness.util;

/**
 * A mutable object is mainly used for the accumulation of state from local or
 * anonymously defined inner classes into the surrounding context (to be
 * accessed variables of this type must be defined final at point-of-use). A
 * java.lang.Object is immutable once declared final and therefore cannot be
 * used for this purpose.
 * 
 * This is almost equivalent to a pointer to an Object rather than just the
 * Object itself.
 * 
 * @author Scott E. Schneider
 * @since 3.0.1
 */
public class MutableObject {
    /**
     * The internally stored state of the mutable Object
     */
    private Object value;

    /**
     * Constructs a mutable Object
     */
    public MutableObject() {   
    }

    /**
     * Constructs a mutable Object with the given value.
     * 
     * @param value
     *            the value the internally stored state of the mutable Object
     *            will be initialized to
     */
    public MutableObject(Object value) {
        this.value = value;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {
        if (object instanceof MutableObject) {
            if (object == this) {
                return true;
            } 
            return (((MutableObject) object).value == this.value);
        }
        return false;
    }

    /**
     * Returns the current value of the internally stored mutable Object state.
     * 
     * @return the current value of the mutable Object
     */
    public Object getValue() {
        return this.value;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return this.value.hashCode();
    }

    /**
     * Sets the value of this mutable Object.
     * 
     * @param value
     *            the value to set the mutable Object to
     */
    public void setValue(Object value) {
        this.value = value;
    }
}