/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.harness.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * This class provides a small set of services for standalone execution that
 * are otherwise provided by the Eclipse plugin.  Among these services are
 * locating and retrieving strings from the correct resource bundle for the 
 * current locale, and logging errors (which currently simply writes to 
 * System.out.)
 *  
 * @author jtoomey
 */
public class StandaloneSystemUtility implements ISystemUtility {

	private ResourceBundle bundle;
	private String harnessDir;
	private String localizedBundleJar;
	
	public StandaloneSystemUtility(String harnessDir) {
		super();

		// Make sure the harness dir ends in a '/' character, or the 
		// URLClassLoader will assume it is a jar file. (!!)
		this.harnessDir = harnessDir;
		if ( harnessDir.charAt(harnessDir.length()-1) == '/')
		{
			this.localizedBundleJar = harnessDir.substring(0,harnessDir.length()-2);
		}
		else
		{
			this.harnessDir += '/';
			this.localizedBundleJar = harnessDir;
		}
		
		// Build path to localized bundle jar
		this.localizedBundleJar += ".nl1/nl1.jar"; //$NON-NLS-1$
	}

	/**
	 * This method returns the correct resource bundle for the current locale.
	 * 
	 * @return a resource bundle, or null if one can not be located. 
	 * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#getResourceBundle()
	 */
	public ResourceBundle getResourceBundle() {
		return getResourceBundle(Locale.getDefault());
	}
	
	/**
	 * This method returns the correct resource bundle for the specified locale.
	 * @param targetLocale
	 * @return a resource bundle, or null if one can not be located. 
	 */
	public ResourceBundle getResourceBundle(Locale targetLocale) {
		
		// return the cached bundle, if we've already located it.
		if ( this.bundle != null )
			return this.bundle;
			
		ResourceBundle newBundle = null;

		try {
			URL[] newcp = {
			        new URL("file", "", this.harnessDir), //$NON-NLS-1$ //$NON-NLS-2$
			        new URL("file", "", this.localizedBundleJar) //$NON-NLS-1$ //$NON-NLS-2$
			};  
			ClassLoader resourceLoader = new URLClassLoader(newcp, null);
	
			newBundle = ResourceBundle.getBundle("plugin", targetLocale, resourceLoader); //$NON-NLS-1$
			this.bundle = newBundle;
		}
		// If we throw an exception above, we'll log the error and return null. 
		catch (MalformedURLException exc)
		{
			logError(exc);
			return null;
		}
		catch (MissingResourceException exc)
		{
			logError(exc);
			return null;
		}

		return newBundle;
	}

	/**
	 * This method returns the correct, localized string for the specified key, 
	 * or the key itself if the localized string can not be located.
	 * 
	 * @param key
	 * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#getString(java.lang.String)
	 */
	public String getString(String key) {
		ResourceBundle bundle = getResourceBundle();
		if ( bundle == null )
			return key;

		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * This method logs the message of the specified Throwable to System.out.
	 * @see org.eclipse.hyades.execution.harness.util.ISystemUtility#logError(java.lang.Throwable)
	 */
	public void logError(Throwable exc) {
		System.out.println(exc.getMessage());
	}

}
