/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;

import org.eclipse.hyades.internal.execution.remote.AgentControllerListener;
import org.eclipse.hyades.internal.execution.remote.MonitorListener;
import org.eclipse.hyades.internal.execution.remote.QueueManager;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;

/**
 * This is the profiler skeleton that can be used create profilers in a distributed environment.
 * It has already implemented the ability of queuing messages, and it generates agentCreate and
 * traceStart tags automatically.  Subclasses only need to worry about what to fill in in the body
 * of the traces.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public abstract class ProfilerImpl implements MonitorListener, AgentControllerListener {

    /**
     * the agent that will perform the actual logging
     */
    private RemoteComponentSkeleton _agent = null;

    /**
     * the flag indicating whether the agent is currently being monitored
     */
    private boolean _isMonitored = false;

    /**
     * the flag indicating whether the agent controller is active or not
     */
    private boolean _isAgentControllerAvailable = false;

    /**
     * the record of the agent creation
     */
    private RecordAgentCreate _agentRecord = null;

    /**
     * the message queue acting as a data sink till the agent is active
     */
    private QueueManager _messageQueue = new QueueManager();

    /**
     * the local reference to the UUID of the agent
     */
    private String _agentId = null;

    /**
     * the local reference to the UUID of the agent process
     */
    private String _processId = null;

    /**
     * the local reference to the UUID of the agent node
     */
    private String _nodeId = null;


    protected ProfilerImpl(String name, String type) {

        /* Due to a race condition inside the RAC, this must be synchronized to give the initializer
           enough time to get initialize all fields properly so that agentControllerActive() does not
           get called too early. */
        synchronized (this) {
            try {
                _agent = new RemoteComponentSkeleton(name, type);
                _agent.addMonitorListener(this);
                _agent.addAgentControllerListener(this);
                _agent.initializeFast();
            } catch (Throwable e) {
                /* The exception here is assumed to be OK. */
            }
        }
    }


    /**
     * This is called by the Logging Agent mechanism to indicate that the 
     * agent controller is present and running on the current node.
     */
    public void agentControllerActive() {

        try {

            /* Due to a race condition inside the RAC, this synchronization will delay the creation
               of the record to ensure all fields are initialized. */
            synchronized (this) {

                /* Obtain the UUIDs and create the agent record for later traces. */
                _agentId = _agent.getAgentUUID();
                _processId = _agent.getJVMUUID();
                _nodeId = _agent.getNodeUUID();
                _agentRecord = new RecordAgentCreate(_agentId, _processId, _nodeId,
                    _agent.getName(), Utilities.getCurrentTimeStamp());
            }

        } catch (Throwable e) {
            /* Exception gets thrown only if not registered.  It will be registered at this point,
               so we are OK. */
        }
        _isAgentControllerAvailable = true;
    }


    /**
     * This is called by the Logging Agent mechanism to indicate that the 
     * agent controller is not present and not running on the current node.
     */
    public void agentControllerInactive() {

        _isAgentControllerAvailable = false;
        _isMonitored = false;
    }


    /**
     * Invoked by the agent mechainism to indicate a remote client is collecting the data.
     * This method will flush any data held in the queue.
     */
    public void monitorActive() {

        /* Enable the monitor.  Create the agentCreate and traceStart for the new trace.  There may already data
           generated for the trace saved in the message queue.  Thus, the queue should be flushed as necessary. */
        _isMonitored = true;
        write(_agentRecord.toString());
        write(new RecordTraceStart(_agent.getAgentUUID()).toString());
        flushMessageQueue();
    }


    /**
     * Invoked by the logging agent mechanism to indicate a remote client is discontinuing
     * data collection.
     */
    public void monitorInactive() {

        _isMonitored = false;
    }


    /**
     * Writes the message to the appropriate data sink, depending on the status of the agent
     * and the agent controller.
     *
     * @param message   the message to be written
     */
    public void write(String message) {

        if (_isAgentControllerAvailable && _isMonitored) {
            _agent.logMessageUTF8(message);
        } else {
            _messageQueue.insertInQueue(message);
        }
    }

    /**
     * Writes any pending messages to the agent.
     */
    public void flushMessageQueue() {

        if (_messageQueue.isFlushNeeded() && _isAgentControllerAvailable && _isMonitored) {
            _messageQueue.flushCurrentQueue(_agent);
        }
    }

    /**
     * Deregister this logging agent instance from the agent controller.
     */
    public void finalize() {

        _agent.deregister();
    }


    /**
     * Retrieves the UUID of the agent.
     * 
     * @return a reference to the agent UUID.
     */
    public String getAgentId() {
        return _agentId;
    }


    /**
     * Retrieves the UUID of the process.
     * 
     * @return a reference to the UUID of the process.
     */
    public String getProcessId() {

        return _processId;
    }


    /**
     * Retrieves the UUID of the node.
     * 
     * @return a reference to the UUID of the node.
     */
    public String getNodeId() {

        return _nodeId;
    }
}
