package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/** 
 * <code>AbstractErrorLogParser</code> is the abstract superclass for the <code>ApacheErrorLogParser</code>.
 * <p>
 * For each error log record, this class parses the time stamp, message category,  
 * client IP address and message.  This method also has a simplistic algorithm for 
 * finding a file name in the message field.  This parsed information and information 
 * about the local machine (machine where the parse is performed, but not 
 * necessarily the machine that produced the error log) is used to produce a
 * <code>org.eclipse.hyades.logging.events.cbe.CommonBaseEvent<code> object for each record.
 *  
 * 
 * @author  Paul Slauenwhite
 * @author  Gary Dudley
 * @version	September 30, 2003
 * @see		org.eclipse.hyades.logging.parsers.Parser
 * @see		org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 */
public abstract class AbstractErrorLogParser extends Parser {

    //Variables to hold values parsed from the error log record:  

    /**
     * Parsed value of the category field. 
     */
    protected String category = null;

    /**
     * Severity associated mapped from the category.
     */
    protected short severity = 0;

    /**
     * Parsed value of the client's IP address field. 
     */
    protected String clientIP = null;

    /**
     * Parsed value of the message field. 
     */
    protected String message = null;

    /**
     * File name found in the message field. 
     */
    protected String fileName = null;

    /**
     * CGI debugging output. 
     * NOTE:  Must be initialized to an empty string buffer.
     */
    protected StringBuffer cgiDebuggingOutput = new StringBuffer();

    /**
     * Parsed value of the time stamp field converted to the XML dateTime format. 
     * NOTE:  Must be initialized to null.
     */
    protected StringBuffer currentTimeStamp = null;

    //User-supplied variables:

    /**
     * User-supplied value of the version of the web server product.  Instantiated in 
     * the setUserInput method of the subclass.
     */
    protected String productVersion = null;

    /**
     * Web server product name and version supplied by the subclass.   
     */
    protected String sourceID = null;

    //Private variables:

    /**
     * Parsed value of the time stamp field converted to the XML dateTime format from previous error log record. 
     * NOTE:  Must be initialized to a zero-based time stamp (e.g. 1970-01-01 00:00:00.000000-00:00).
     * ASSUMPTION:  All error log records appear in chronological order within the log file.
     */
    private String previousTimeStamp = EventHelpers.longToDate(0);

    /**
     * Count of records with identical time stamp values. 
     * NOTE:  Must be initialized to 0 since no log records have been parsed.
     * ASSUMPTION:  All error log records appear in chronological order within the log file.
     */
    private long duplicateTimeStampCounter = 0;

    /**
     * Initialize this parser.
     */
    public void preParse() throws LogParserException {
        super.preParse();
    }

	/**
	 * De-constructs this parser (e.g. closes the log file handle).
	 */
	public void postParse() throws LogParserException {
		super.postParse();
	}
	
    /**
     * Parses each error log record and produces a Common Base Event (CBE) object that is 
     * returned as a member of a CBE array.
     *
     * @return CommonBaseEvent[] array of CBE's representing parsed records.
     * @exception LogParserException thrown if the parser is unable to parse the error log
     */
    public CommonBaseEvent[] parseNext() throws LogParserException {
        CommonBaseEvent[] temp = null;

        curLine = readLine();
        arrayIndex = 0;

        try {

            //Iterate all the lines (e.g. until EOF/null line) within the file:
            while (curLine != null) {

                //Trim leading/trailing whitespace from the current line:
                curLine = curLine.trim();

                //Only parse non-empty (e.g. lines with one or more non-whitespace characters) lines:
                if (curLine.length() > 0) {

                    //ASSUMPTION:  At least one valid log record (e.g. containing at least a time stamp, category and message) appears in each log file, otherwise the log file is deemed invalid. 

                    //Parse the log record:
                    if (parseLogRecord()) {

                        if (cgiDebuggingOutput.length() > 0) {

                            createCGIDebuggingOutputCBE();

                            arrayIndex++;

                            if (arrayIndex == MessageArraySize) {
                                // Increase the message array size by one
                                increaseMsgArraySize();
                            }

                            recordCount++;
                        }

                        //Reinitialize the CBE:
                        reinitializeCBE();

                        //Increment the CBE's sequence number and running counter when duplicate time stamp values are detected. 
                        //ASSUMPTION:  All error log records appear in chronological order within the log file.
                        if (previousTimeStamp.equals(currentTimeStamp.toString()))
                            messages[arrayIndex].setSequenceNumber(++duplicateTimeStampCounter);

                        //Otherwise persist the current time stamp as the new previous time stamp and reset the running duplicate time stamp counter:
                        else {
                            previousTimeStamp = currentTimeStamp.toString();
                            duplicateTimeStampCounter = 0;
                        }

                        //Add information from the error log record to the CBE:
                        messages[arrayIndex].setCreationTime(currentTimeStamp.toString());
                        messages[arrayIndex].setMsg(message);
                        messages[arrayIndex].setSeverity(severity);

                        //Add remaining information from the error log record as extended data elements to the CBE:
                        if (category != null)
                            messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.CATEGORY, category));
                        if (clientIP != null)
                            messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.CLIENT, clientIP));
                        if (fileName != null)
                            messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.FILE, fileName));

                        arrayIndex++;

                        if (arrayIndex == MessageArraySize) {
                            arrayIndex = 0;
                            recordCount++;
                            reset();
                            return messages;
                        }

                        recordCount++;

                        //Reset the local properties of a log record:
                        reset();
                    }
                }

                //Read the next line in the file:
                curLine = readLine();
            }

            if (cgiDebuggingOutput.length() > 0) {

                //Invalid log record:
                if (recordCount == 0)
                    throw new LogParserException(ParserUtilities.getResourceString("INVALID_ERROR_LOG_ERROR_", file_path));

                else {

                    createCGIDebuggingOutputCBE();

                    arrayIndex++;

                    if (arrayIndex == MessageArraySize) {
                        arrayIndex = 0;
                        recordCount++;
                        reset();
                        return messages;
                    }
                    else {
                    	/* Bugzilla 69409 - Reset the cgi debugging output string buffer */
                    	cgiDebuggingOutput = new StringBuffer();
                    }

                    recordCount++;
                }
            }

            // If we are not logging the message then null the array elements that weren't set on this call

            if (arrayIndex == 0) {
                temp = null;
            }
            else {
                for (int i = arrayIndex; i < MessageArraySize; i++) {
                    messages[i] = null;
                }
                temp = messages;
            }

            //Throw an exception if no valid access log records are parsed/logged:
            if (recordCount == 0)
                throw new LogParserException(ParserUtilities.getResourceString("NO_ERROR_LOG_RECORDS_ERROR_", file_path));
        }
        catch (LogParserException l) {
            throw l;
        }
        catch (Throwable t) {
            ParserUtilities.exceptionHandler(t, curLineNumber, curLine, ParserUtilities.getResourceString("ERROR_LOG_PARSER_ERROR_"));
        }
        /*        
        		finally {
        			//Close the log file:
        			closeFiles();
        		}
        */
        return temp;
    }

    private void createCGIDebuggingOutputCBE() {

        //Reinitialize the CBE:
        reinitializeCBE();

        //Add information from the error log record to the CBE:
        messages[arrayIndex].setSequenceNumber(++duplicateTimeStampCounter);
        messages[arrayIndex].setCreationTime(previousTimeStamp);
        messages[arrayIndex].setMsg(cgiDebuggingOutput.toString().trim());
        messages[arrayIndex].setSeverity(ParserConstants.CBE_SEVERITY_3);
        messages[arrayIndex].addExtendedDataElement(createStringEDE(ParserConstants.CGI, cgiDebuggingOutput.toString().trim()));
    }

    private void reinitializeCBE() {

        //(Re)initialize the CBE:
        messages[arrayIndex].init();

        //Set the event's globalInstanceId property with a new GUID:
        messages[arrayIndex].setGlobalInstanceId(Guid.generate());

        //(Re)initialize the CBE's source component ID, set various properties and add to the CBE:
        ComponentIdentification sourceComponentID = eventFactory.createComponentIdentification();
        sourceComponentID.setLocation(localHostId);
        sourceComponentID.setLocationType(localHostIdFormat);
        sourceComponentID.setComponent(sourceID);
        sourceComponentID.setSubComponent(ParserConstants.UNKNOWN);
        //New for CBE 1.0.1:
        sourceComponentID.setComponentType(ParserConstants.APACHE_COMPONENT_TYPE);
        sourceComponentID.setComponentIdType(ParserConstants.APACHE_COMPONENT_ID_TYPE);

        messages[arrayIndex].setSourceComponentId(sourceComponentID);

        messages[arrayIndex].setSituation(createSituation());
    }

    /**
     * Generates a generic situation for a CBE. 
     * 
     * @return ISituation The generic situation for a CBE. 
     * @since CBE 1.0.1 
     */
    private Situation createSituation() {

        //Initialize the CBE's situation and set various properties:
        Situation cbeSituation = eventFactory.createSituation();

        //Unknown situation therefore use the generic:
        //Initialize the CBE's situation type, set various properties and add to the situation:
        ReportSituation cbeReportSituation = eventFactory.createReportSituation();
        cbeReportSituation.setReasoningScope(ParserConstants.INTERNAL_REASONING_SCOPE);
        cbeReportSituation.setReportCategory(ParserConstants.LOG_REPORT_CATEGORY);

        cbeSituation.setCategoryName(ParserConstants.REPORT_SITUATION_CATEGORY_NAME);
        cbeSituation.setSituationType(cbeReportSituation);

        return cbeSituation;
    }

    /**
     * Parses a time stamp from an access log record, produces a Date object from the
     * parsed information, and converts the Date object into a XML DateTime String.
     *
     * @param startIndex Starting index of time stamp in curLine. 
     * @param endIndex Ending index of time stamp in curLine. 
     * @return true if time stamp is valid; otherwise false. 
     */
    protected boolean parseDate(int startIndex, int endIndex) {

        //Running example: [28/May/2003:16:36:39 -0400] --> 2003-05-28T16:36:39.000000-04:00

        //Verify that the time stamp is enclosed with square brackets:
        if ((curLine.charAt(startIndex) == '[') && (curLine.charAt(endIndex) == ']')) {

            //Parse the access log's time stamp excluding the time zone offset and square brackets (e.g. 28/May/2003:16:36:39) to a java.util.Date object:
            SimpleDateFormat formatter = new SimpleDateFormat(ParserConstants.APACHE_ERROR_TIME_STAMP_FORMAT, Locale.US);
            Date creationDate = formatter.parse(curLine.substring((startIndex + 1), endIndex).trim(), new ParsePosition(0));

            //If the access log's time stamp is valid (e.g. non-null java.util.Date object), convert to its XML dateTime format:
            if (creationDate != null) {

                //Format the java.util.Date object to its XML dateTime format (e.g. "yyyy-MM-dd HH:mm:ss"):
                formatter = new SimpleDateFormat(ParserConstants.XML_DATETIME_FORMAT);
                currentTimeStamp = new StringBuffer(formatter.format(creationDate).trim());

                //Replace the first space with "T":
                currentTimeStamp.replace(10, 11, "T");

                //Add the fractional second value (e.g. .000000):
                currentTimeStamp.append(ParserConstants.SIX_ZERO);

                //Generate the local UTC offset using the parsed date:
                Calendar localCalendar = Calendar.getInstance();
                localCalendar.setTime(creationDate);
                int utcOffset = ((localCalendar.get(Calendar.ZONE_OFFSET) + localCalendar.get(Calendar.DST_OFFSET)) / 60000);

                //Add the sign of the UTC offset:
                if (utcOffset < 0)
                    currentTimeStamp.append("-");
                else
                    currentTimeStamp.append("+");

                utcOffset = Math.abs(utcOffset);

                // Time Zone (+/-hh:mm)

                String numberHours = String.valueOf(utcOffset / 60);

                if (numberHours.length() == 1)
                    currentTimeStamp.append("0");

                currentTimeStamp.append(numberHours);
                currentTimeStamp.append(":");

                String numberMinutes = String.valueOf(utcOffset % 60);

                if (numberMinutes.length() == 1)
                    currentTimeStamp.append("0");

                currentTimeStamp.append(numberMinutes);

                return true;
            }
        }

        return false;
    }

    /**
     * Main parsing routine for an error log record.
     *
     * @return   true if the error log record is successfully parsed; otherwise false. 
     */
    protected boolean parseLogRecord() {

        //Running example: [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        //Running example notation: 
        //1) Previous index is enclosed with asterisks (e.g. *p*).
        //2) Current index is enclosed with pipes (e.g. |c|).
        //3) The previous and current indices are the same(e.g. *|c|*).
        //4) Second quote index is enclosed with uppercase Q (e.g. Q"Q).

        //Index of the previous substring:
        //Example:  *[*Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        int previousIndex = 0;

        //Find the first '[' character in the log record:
        //Example:  *|[|*Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        int currentIndex = curLine.indexOf("[", previousIndex);

        //Return false if the '[' character cannot be found since it is an invalid log record: 
        if (currentIndex == -1) {

            cgiDebuggingOutput.append(curLine);
            cgiDebuggingOutput.append(ParserConstants.LINE_SEPARATOR);

            return false;
        }

        //Advance the previous pointer to the current pointer: 
        //Example:  *|[|*Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        previousIndex = currentIndex;

        //Find the first ']' character in the log record:
        //Example:  *[*Wed May 28 16:44:10 2003|]| [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        currentIndex = curLine.indexOf("]", previousIndex);

        //Return false if the ']' character cannot be found since it is an invalid log record: 
        if ((currentIndex == -1) || !parseDate(previousIndex, currentIndex)) {

            cgiDebuggingOutput.append(curLine);
            cgiDebuggingOutput.append(ParserConstants.LINE_SEPARATOR);

            return false;
        }

        //Advance the previous pointer to the current pointer: 
        //Example:  [Wed May 28 16:44:10 2003*|]|* [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        previousIndex = currentIndex;

        //Find the second '[' character in the log record:
        //Example:  [Wed May 28 16:44:10 2003*]* |[|debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        currentIndex = curLine.indexOf("[", previousIndex);

        //Return false if the '[' character cannot be found since it is an invalid log record: 
        if (currentIndex == -1)
            return false;

        //Advance the previous pointer to the current pointer: 
        //Example:  [Wed May 28 16:44:10 2003] *|[|*debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        previousIndex = currentIndex;

        //Find the second ']' character in the log record:
        //Example:  [Wed May 28 16:44:10 2003] *[*debug|]| D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        currentIndex = curLine.indexOf("]", previousIndex);

        //Return false if the ']' character cannot be found since it is an invalid log record: 
        if (currentIndex == -1)
            return false;

        //Parse the category (see http://httpd.apache.org/docs-2.0/mod/core.html#loglevel):
        //Example:  [Wed May 28 16:44:10 2003] *[*debug|]| D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        category = curLine.substring((previousIndex + 1), currentIndex).trim(); //1 is the length of "[".

        //Map the category to a severity:
        if ((category.equals(ParserConstants.EMERG_STRING)) || (category.equals(ParserConstants.ALERT_STRING)) || (category.equals(ParserConstants.CRIT_STRING)) || (category.equals(ParserConstants.ERROR_STRING))) {
            severity = ParserConstants.CBE_SEVERITY_1;
        }
        else if ((category.equals(ParserConstants.WARN_STRING))) {
            severity = ParserConstants.CBE_SEVERITY_2;
        }
        //[notice]/[info]/[debug] Apache severities .
        else {
            severity = ParserConstants.CBE_SEVERITY_3;
        }

        //Advance the previous pointer to the current pointer: 
        //Example:  [Wed May 28 16:44:10 2003] [debug*|]|* D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        previousIndex = currentIndex;

        //Find the "[client" string in the log record:
        //Example:  [Wed May 28 16:44:10 2003] [debug*]* D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): |[client| 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        currentIndex = curLine.indexOf("[client", previousIndex);

        //Return false if the "[client" string cannot be found since it is an invalid log record: 
        if (currentIndex != -1) {

            //Advance the previous pointer to the current pointer: 
            //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): *|[client|* 9.26.157.51] afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
            previousIndex = currentIndex;

            //Find the third ']' character in the log record:
            //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): *[client* 9.26.157.51|]| afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
            currentIndex = curLine.indexOf("]", previousIndex);

            //Return false if the ']' character cannot be found since it is an invalid log record: 
            if (currentIndex == -1)
                return false;

            //Parse the client's IP address (e.g. 9.26.157.51):
            //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): *[client* 9.26.157.51|]| afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
            clientIP = curLine.substring((previousIndex + 7), currentIndex).trim(); //7 is the length of "[client".

            //Advance the previous pointer to the current pointer: 
            //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51*|]|* afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
            previousIndex = currentIndex;
        }

        //Parse the message (e.g. afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy):
        //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51*|]|* afpa_cache: AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        message = curLine.substring((previousIndex + 1)).trim(); //1 is the length of "]".

        //Find the ": " string in the log record:
        //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51*]* afpa_cache|: |AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        currentIndex = curLine.indexOf(": ", previousIndex);

        //Return false if the ": " string cannot be found since it is an invalid log record: 
        //Parse the file name (e.g. AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy):
        //Example:  [Wed May 28 16:44:10 2003] [debug] D:\Build\WWWIHS1326\apache\ibm\modules\afpa\mod_ibm_afpa.c(1261): [client 9.26.157.51*]* afpa_cache|: |AfpaCmLoadFile status: 0, file: d:/ibm/ibmhttpserver/htdocs/en_us/manual/ibm/navtext.txt, uri: /manual/ibm/navText.txt, hostname: pansy, virtual host: pansy
        if (currentIndex != -1)
            fileName = curLine.substring(currentIndex + 2).trim(); //2 is the length of ": ".

        return true;
    }

    /**
     * Resets the parsed values of a log record before next record is parsed.
     */
    protected void reset() {
        severity = 0;
        message = null;
        clientIP = null;
        fileName = null;
        cgiDebuggingOutput = new StringBuffer();
        category = null;
        currentTimeStamp = null;
    }
}