/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.util;

import java.util.*;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;

/**
 * @author marcelop
 * @since 0.3.0
 */
public class ExecutionUtil
{
	/**
	 * Returns a list with the execution results of a given 
	 * {@link TPFTest test}.  The execution results are queried from the 
	 * workspace.
	 * @param test
	 * @return a not <code>null</code> list
	 */
	public static List findExecutionResults(final TPFTest test)
	{
		if((test != null) && (test.eResource() != null))
		{
			URI uri = test.eResource().getURI().trimFileExtension();
			final ResourceSet resourceSet = test.eResource().getResourceSet(); 
			final String executionPrefix = uri.lastSegment().toString();
			ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					if(file.getName().startsWith(executionPrefix))
					{
						EObject[] eObjects = ResourceUtil.load(resourceSet, file);
						if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
						{
							TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
							if(executionResult.getTest() == test)
								return executionResult;
						}
					}
					return null;
				}
			};
			
			return (List)ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), new String[]{ICommonConstants.EXECUTION_FILE_EXTENSION}, visitor).get(ICommonConstants.EXECUTION_FILE_EXTENSION);
		}
		
		return Collections.EMPTY_LIST;
	}
	
	/**
	 * Returns a map with the list of execution results for each 
	 * {@link TPFTest test} in the argument.  The execution results are queried 
	 * from the workspace.
	 * 
	 * <p>The tests are supposed to be located in the same <code>ResourceSet</code>. 
	 * 
	 * @param test
	 * @return a not <code>null</code> list
	 */
	public static Map findExecutionResults(List tests)
	{
		if((tests != null) && (!tests.isEmpty()))
		{
			ResourceSet resourceSet = null;
			Set resourcePrefixes = new HashSet(tests.size()+1);
			for (Iterator i = tests.iterator(); i.hasNext();)
			{
				Object object = i.next();
				if(object instanceof TPFTest)
				{
					TPFTest test = (TPFTest)object;
					if(test.eResource() != null)
					{
						if(resourceSet == null)
							resourceSet = test.eResource().getResourceSet();
						
						URI uri = test.eResource().getURI().trimFileExtension();
						resourcePrefixes.add(uri.lastSegment().toString());
					}					
				}
			}
			
			if(!resourcePrefixes.isEmpty())
			{
				final ResourceSet finalResourceSet = resourceSet;
				final String[] prefixes = (String[])resourcePrefixes.toArray(new String[resourcePrefixes.size()]); 
				final Map ret = new HashMap(resourcePrefixes.size()+1,1);
				ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
				{
					public Object visit(IFile file)
					{
						for (int i = 0, maxi = prefixes.length; i < maxi; i++)
						{
							if(file.getName().startsWith(prefixes[i]))
							{
								EObject[] eObjects = ResourceUtil.load(finalResourceSet, file);
								if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
								{
									TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
									List executionResults = (List)ret.get(executionResult.getTest());
									if(executionResults == null)
									{
										executionResults = new ArrayList();
										ret.put(executionResult.getTest(), executionResults);
									}
									executionResults.add(executionResult);
								}
							}
						}
						return null;
					}
				};
			
				ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), new String[]{ICommonConstants.EXECUTION_FILE_EXTENSION}, visitor);
				ret.keySet().retainAll(tests);
				return ret;
			}
		}
		
		return Collections.EMPTY_MAP;
	}
}
