/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Rational - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import java.io.UnsupportedEncodingException;
import java.util.Enumeration;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport;
import org.eclipse.hyades.execution.core.IProcessConsole;
import org.eclipse.hyades.execution.invocation.IRemoteObject;
import org.eclipse.hyades.execution.invocation.RemoteInvocationException;
import org.eclipse.hyades.execution.invocation.ReturnData;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;

/**
 * @author ddreakfo
 * @author ejessee
 */
public abstract class ExecutorStub extends ExecutionComponentStub implements IExecutorWithProgressMonitorSupport {
	
	private String executionResultLocation = null;
	private String executionResultName = null;
	private String communicationPort = null;
	public IDataProcessor[] dataProcessors = null;
	protected int agentInitCount;
	
	public ExecutorStub() {
		super();
	}
	
	public ExecutorStub(IExecutionComponent delegate){
		super(delegate);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport#launch(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void launch(IProgressMonitor progressMonitor) {
		ReturnData invokeRtn = delegateRemoteCall(new Class[]{}, new Object[]{}, "launch");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
		fireStateChangeEvent(new ExecutionComponentStateChangeEvent(this, IExecutionComponent.SUSPENDED));
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor#launch()
	 */
	public void launch() throws ExecutionComponentStateException {	
		this.launch(new NullProgressMonitor());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport#kill(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void kill(IProgressMonitor progressMonitor) throws ExecutionComponentStateException {
		ReturnData invokeRtn = delegateRemoteCall(new Class[]{}, new Object[]{}, "kill");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
		fireStateChangeEvent(new ExecutionComponentStateChangeEvent(this, IExecutionComponent.DEAD));
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor#kill()
	 */
	public void kill() throws ExecutionComponentStateException {
		this.kill(new NullProgressMonitor());
	}

	/**
	 * Associate an executable object with this executor.
	 * There is a local and a remote portion of this operation.
	 *  
	 * @see org.eclipse.hyades.execution.core.IExecutor#setExecutableObject(org.eclipse.hyades.execution.core.IExecutableObject)
	 */
	public void setExecutableObject(IExecutableObject theExecutableObject) throws ExecutionComponentStateException {
		// Set the local values in case we decide to add a getExecutableObject().
		((IExecutor)delegate).setExecutableObject(theExecutableObject);
		
		ReturnData invokeRtn = delegateRemoteCall(
			new Class[]{IExecutableObject.class},
			new Object[]{theExecutableObject},
			"setExecutableObject");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}

		// Do this after both (the local and the remote) portions of
		// setExecutableObject() have been completed.
		theExecutableObject.setExecutor(this);
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.IExecutor#getExecutableObject()
	 */
	public IExecutableObject getExecutableObject()
	{
		return ((IExecutor)delegate).getExecutableObject();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.IExecutor#getProcessConsole()
	 */
	public IProcessConsole getProcessConsole() throws ExecutionComponentStateException {
		// TODO Auto-generated method stub
		return null;
	}
		
	/**
	 * Create an <code>IExecutableObject</code> instance that is compatible
	 * with this executor.
	 * 
	 * The implementation of this method differs from the
	 * norm, as no responsibilities are delegated to the delegate object.
	 * Instead, the creation of the executable object is handled completely
	 * by the stub/skeleton combination.
	 * 
	 * @return a <code>JavaTaskExcutableObjectStub</code> 
	 */
	public IExecutableObject getCompatibleExecutableObject(String classname) throws ClassNotFoundException {
		
		// We could configure the factory to return the expected type of
		// executable object here, instead of in the Test Tool initialization
		// code. I'm not sure where it fits best.
		ExecutionComponentFactoryImpl factory = (ExecutionComponentFactoryImpl)ExecutionComponentFactoryImpl.getInstance(
			getSessionContext());
		IRemoteObject remoteObj = factory.createRemoteObjectByType(classname);
		if ( !(remoteObj instanceof IExecutableObject) )
			throw new ClassCastException("Component factory returned an incorrect type; expected an IExecutableObject");
		return (IExecutableObject)remoteObj;
	}
	
	/**
	 * @return
	 */
	public String getExecutionResultLocation() {
		return executionResultLocation;
	}

	/**
	 * @return
	 */
	public String getExecutionResultName() {
		return executionResultName;
	}

	/**
	 * @param string
	 */
	public void setExecutionResultLocation(String string) {
		executionResultLocation = string;
	}

	/**
	 * @param string
	 */
	public void setExecutionResultName(String string) {
		executionResultName = string;
	}

	
	/**
	 * returns the array of IDataProcessors which are to monitor this executor
	 */
	public IDataProcessor[] getDataProcessors()
	{
		return dataProcessors;
	}

	/**
	 * called to set the IDataProcessor collection.
	 * @param processors
	 */
	public void setDataProcessors(IDataProcessor[] processors)
	{
		dataProcessors = processors;
	}

	/**
	 * returns the process ID of this executor.
	 * @return String
	 */
	public String getPid()
	{
		ReturnData invokeRtn = delegateRemoteCall(new Class[]{}, new Object[]{}, "getPid");
		Object rtnValue = invokeRtn.getReturnValue();
		if(invokeRtn.isError()) {
			throw new RemoteInvocationException((Throwable)rtnValue);
		}
		return (String)rtnValue;
	}

	/**
	 * returns the Process parented by the passed Node with the passed pid
	 * @param pid
	 * @param node
	 * @return
	 * @throws NotConnectedException
	 * @throws InactiveProcessException
	 */protected Process findProcess(String pid, Node node) throws NotConnectedException, InactiveProcessException
	{
		Process proc = null;
		Enumeration processes=node.listProcesses();
		
		while(processes.hasMoreElements()) 
		{
			Process procBuf=(Process)processes.nextElement();
			if(procBuf.getProcessId().equals(pid)) 
			{
				return procBuf;
			}	
		}	
		return null;
	}

	/**
	 * return the Agent whose name is a superset of agentName
	 * @param proc
	 * @param agentType
	 * @param agentName
	 * @return Agent
	 */
	protected Agent findAgent(Process proc, String agentType, String agentName)
	{
		Enumeration agents=proc.getAgentsByType(agentType);
		while(agents.hasMoreElements()) 
		{
			Agent agent = (Agent)agents.nextElement();
			if(agent.getName().indexOf(agentName)!=-1)
			{
				return agent;
			}
		}
		return null;
	}

	/**
	 * called to register an agentListener as the listener on the control channel for the passed Agent
	 * @param agentListener
	 * @param agent
	 */
	protected void setupControlListener(AgentListener agentListener, Agent agent) throws InactiveAgentException, InactiveProcessException, UnsupportedEncodingException
	{
		
		
			agent.addAgentListener(agentListener);
		
			agent.attach();
		
			String controlAgentName = agent.getName();
		
			IExecutionComponent[] children=getChildren();
		
			agentInitCount++;
			
			((JavaTaskRemoteHyadesComponentStub)children[agentInitCount-1]).setAgent(agent);
						
			
			IExecutableObject executableObject = getExecutableObject();
			if(executableObject instanceof JavaProcessExecutableObjectStub)
			{
				String script = ((JavaProcessExecutableObjectStub)executableObject).getAgentData();
				if(script != null)
				{
					BinaryCustomCommand agentData = new BinaryCustomCommand();
					agentData.setData(script.getBytes("UTF-8"));
					agent.invokeCustomCommand(agentData);
				}
			}
											
	}

	protected void resetAgentInitCount()
	{
		agentInitCount = 0;
	}

	
	/**
	 * @return Returns the communicationPort.
	 */
	public String getCommunicationPort() {
		return communicationPort;
	}
	/**
	 * @param communicationPort The communicationPort to set.
	 */
	public void setCommunicationPort(String communicationPort) {
		this.communicationPort = communicationPort;
	}
}
