/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.execution.local;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.Principal;

import org.eclipse.hyades.execution.core.DaemonConnectException;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.file.IFileManager;
import org.eclipse.hyades.execution.invocation.Marshaller;
import org.eclipse.hyades.execution.local.file.FileManagerImpl;
import org.eclipse.hyades.internal.execution.local.common.Console;
import org.eclipse.hyades.internal.execution.local.common.DataProcessor;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.ConnectionListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.NoSuchApplicationException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessActiveException;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.security.DuplicateUserException;
import org.eclipse.hyades.internal.execution.security.LoginFailedException;
import org.eclipse.hyades.internal.execution.security.SecureConnectionRequiredException;

/**
 * @author rduggan
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class NodeImpl implements INode {

	/* The name of the node */
	private String name;
	
	/* The objects for synchronizing our session handshake */
	private Object sessionLock = new Object();
	private boolean connected = false;
	
	/* whether we are building in debug mode */
	private static final boolean DEBUG = true;
	
	/* The FileManagerused by this Node */
	IFileManager fileManager;
	
	public NodeImpl(String name) {
		this.name=name;
	}
	
	/**
	 * @see org.eclipse.hyades.execution.core.INode#getName()
	 */
	public String getName() {
		return name;
	}

	/**
	 * 
	 * @see org.eclipse.hyades.execution.core.INode#connect(java.lang.String, java.security.Principal)
	 */
	public ISession connect(String daemonType, Principal principal) throws UnknownDaemonException, 
																			DaemonConnectException,
																			UnknownHostException {
		
		//
		// Check the Daemon type here and throw an UnknownDaemonException if it isnt recognised
		//
		
		/* create the proxy to the node */
		
		Node node=null;
		
		try {
			if(principal==null) {
				node = NodeFactory.createNode(name);
			}
			else {
				node = NodeFactory.createNode(name, principal);
			}
		}
		catch(DuplicateUserException e) {
			/* The prinical already exists */
			throw e;
		}
		catch(UnknownHostException e) {
			/* We could not resolve the remote host */
			throw e;
		}
	
		/* Connect on the proper port. */
		try {
			node.connect(Integer.parseInt(daemonType));
		}
		catch(SecureConnectionRequiredException e) {
			/* We need s secure connection to the node.  Collect the user information 
			 * and add a user and keystoremanager to the node.
			 */
			 throw e;
		}
		catch(LoginFailedException e) {
			/* We are trying to connect with the wrong authentication credentials.  Update
			 * the username and password and try again
			 */
			 throw e;
		}
		catch(AgentControllerUnavailableException e) {
			/* there are several subclasses of this exception we need to handle
			 * properly.
			 */
			 throw e;
		}
		
	
		/* Add connection listener for connection events. */
		node.getConnection().addConnectionListener(new ConnectionListener() {

			/**
			 * @see org.eclipse.hyades.internal.execution.local.control.ConnectionListener#connectionClosed(org.eclipse.hyades.internal.execution.local.control.Connection)
			 */
			public void connectionClosed(Connection connection) {
				// TODO Auto-generated method stub
			}
				
		});
		
		/* Create a process.  This will be our session hosting process */
		Process rprocess = ProcessFactory.createProcess(node, "HyadesTestingSession"); 
		
		/* if we are running in debug mode redirect the stdout/stderr of the remote process */
		if(DEBUG) {
			captureConsoleOutput(rprocess);
		}
		
		/* Create our agent for communication with the remote session. We add the agent 
		 * here with the understanding that it will be instantiated on the remote process
		 * by the RemoteSession.  The autoattach means we will grab an exclusive lock on
		 * it at registration time in the Daemon process.
		 */
		Agent ragent = AgentFactory.createAgent(rprocess,"RemoteSession", "HyadesSession");
		ragent.setAutoAttach(true);
		
		/* Create our ISession and give it the agent it will use to communicate with */ 
		final ISession sessionInstance = new SessionStub(new SessionImpl(this, ragent)); 
		
		Marshaller.addInstanceToMap(((SessionStub)sessionInstance).getUniqueId(), sessionInstance);
		
		/* As a parameter to the process we will provide the uniqueId of the session
		 * stub so that the proper associatione between the session objects can occur
		 */
		 try {
		 	 rprocess.setParameters("org.eclipse.hyades.execution.remote.NodeImpl "+((SessionStub)sessionInstance).getUniqueId().toString());
		 }
		 catch(ProcessActiveException e) {
		 	/* We can ignore this as we have not launched the process yet */
		 }
		
		/* Add a process listener */
		rprocess.addProcessListener(new ProcessListener() {

			public void processLaunched(Process process) {
			}

			public void processExited(Process process) {
				/* If the process is exiting and we are still locked below then there were problems
				 * instantiating the agent in the remote process.  Allow the method to return.
				 */
				releaseSessionLock();
			}
	
		});
		
		/* Add a state change listener to the ISession so we know when the session is active and good to be used */
		sessionInstance.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
			public void stateChanged(ExecutionComponentStateChangeEvent event) {
				/* When we go to active state make sure we signal the session lock */
				if(event.getState()==IExecutionComponent.READY) {
					releaseSessionLock();
				}	
			}
		
		});

		
		/* We are all setup now.  Launch the process. */
		try {
			rprocess.launch();
			
			/* Confirm our process is running correctly */
			rprocess.getProcessId();
		}
		catch(InactiveProcessException e) {
			/* Our process failed to launch */
			// Cascade the exception
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		}
		catch(NoSuchApplicationException e) {
			/* There is no alias in the config file */
			// Cascade the exception
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		}
		catch(ProcessActiveException e) {
			/* We just created the process above and this should not be posible. */
			// Cascade the exception
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		}
		catch(NotConnectedException e) {
			/* This will not occur */
			// Cascade the exception
			DaemonConnectException exc = new DaemonConnectException();
			exc.initCause(e);
			throw exc;
		}
		
		/* Block until the session is up and running completely */
		while(!connected) {
			synchronized(sessionLock) {
				try {
					sessionLock.wait();
				}
				catch(InterruptedException e) {
					/* We can ignore this */
				}
			}
		}
		
		fileManager=new FileManagerImpl(node.getConnection());

		/* Return the ISession for use */ 		
		return sessionInstance;	
	}
	
	private void releaseSessionLock() {
		synchronized(sessionLock) {
			connected=true;
			sessionLock.notify();
		}
	}
	
	
	/**
	 * This method is provided strictly to simplify debugging the stdout and stderr of the remote process
	 * hosting the session.  This will take the remote console output and dump it to the local processes 
	 * stdout stream.
	 * 
	 * @param process - the remote process to redirect its stdout/stderr to the local stdout.
	 */
	private void captureConsoleOutput(Process process) {
		Console console=process.getConsole();
		console.setDataProcessor(new DataProcessor() {

			public void incommingData(byte[] buffer, int length, InetAddress peer) {
				System.out.print("====>");
				System.out.print(new String(buffer, 0, length));
				System.out.flush();
				
			}
			public void incommingData(char[] buffer, int length, InetAddress peer) {
				System.out.print("====>");
				System.out.print(new String(buffer, 0, length));
				System.out.flush();
				
			}
			public void invalidDataType(byte[] data, int length, InetAddress peer) {
				
			}
			public void waitingForData() {
				
			}
			
		});
		
	}
	
	public IFileManager getFileManager() {
		return fileManager;
	}


}
