/**********************************************************************
 * Copyright (c) 2003,2004 Scapa Technologies Limited and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.statistical.ui.editor.internal;

import org.eclipse.hyades.statistical.ui.widgets.alert.internal.Alert;
import org.eclipse.hyades.statistical.ui.widgets.alert.internal.AlertAction;
import org.eclipse.hyades.statistical.ui.widgets.grapher.internal.*;
import org.eclipse.hyades.statistical.ui.widgets.table.internal.InvalidTableValueException;
import org.eclipse.hyades.statistical.ui.widgets.table.internal.*;

import org.eclipse.hyades.statistical.ui.EditorPlugin;
import org.eclipse.hyades.statistical.ui.ImageManager;

import java.text.DecimalFormat;
import java.util.*;

import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;

import org.eclipse.swt.widgets.*;

import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

public class AlertTable extends TableAdapter implements DisposeListener, SelectionListener, FocusListener, MouseListener
{
	DecimalFormat df = new DecimalFormat("###########################################.########################################");

	public static final int VAL_MONITOR = 0;
	public static final int VAL_NAME = 1;
	public static final int VAL_PERIOD = 2;
	public static final int VAL_CALCTYPE = 3;
	public static final int VAL_LOWER = 4;
	public static final int VAL_UPPER = 5;
	public static final int VAL_MTYPE = 6;
	public static final int VAL_ONTRIGGER = 7;
	public static final int VAL_ONRESET = 8;

	private static final int COL_NAME = 0;
	private static final int COL_PERIOD = 1;
	private static final int COL_CALCTYPE = 2;
	private static final int COL_LOWER = 3;
	private static final int COL_UPPER = 4;
	private static final int COL_MTYPE = 5;
	private static final int COL_ONTRIGGER = 6;
	private static final int COL_ONRESET = 7;
	
	private Vector listeners = new Vector();
	private Vector datas = new Vector();
	private boolean editable = true;
	
	Menu alert_menu;
	MenuItem alert_delete;
	MenuItem alert_restart;

	
	private String[] typeChoices = new String[] {
			"TYPE_MAX_TRIGGER",
			"TYPE_MIN_TRIGGER",
		};
	private String[] typeChoicesString = new String[] {
			EditorPlugin.getString("TYPE_MAX_TRIGGER"),
			EditorPlugin.getString("TYPE_MIN_TRIGGER"),
		};

	private String[] calcTypeChoices = new String[] {
			"PLOTTING_TYPE_AVERAGE",
			"PLOTTING_TYPE_MAX",
			"PLOTTING_TYPE_MIN",
			"PLOTTING_TYPE_SUM",
			"PLOTTING_TYPE_COUNT",
			"PLOTTING_TYPE_NEAREST",
			"PLOTTING_TYPE_STDDEV",
		};
	private String[] calcTypeChoicesString = new String[] {
			EditorPlugin.getString("PLOTTING_TYPE_AVERAGE"),
			EditorPlugin.getString("PLOTTING_TYPE_MAX"),
			EditorPlugin.getString("PLOTTING_TYPE_MIN"),
			EditorPlugin.getString("PLOTTING_TYPE_SUM"),
			EditorPlugin.getString("PLOTTING_TYPE_COUNT"),
			EditorPlugin.getString("PLOTTING_TYPE_NEAREST"),
			EditorPlugin.getString("PLOTTING_TYPE_STDDEV"),
		};

	public void setEditable(boolean newValue)
	{
		editable = newValue;
	}

	public void addListener(AlertTableListener newListener) {
		listeners.add(newListener);
	}
	public void notifyListenersSelect(Alert a) {
		for (int v = 0; v < listeners.size(); v++)
		{
			AlertTableListener t = (AlertTableListener) (listeners.elementAt(v));
			t.alertSelected(a);
		}
	}
	public void notifyListenersDelete(Alert a) {
		for (int v = 0; v < listeners.size(); v++)
		{
			AlertTableListener t = (AlertTableListener) (listeners.elementAt(v));
			t.alertDeleted(a);
		}
	}
	
	public void widgetDisposed(DisposeEvent e) {
		dispose();
	}
	
	public Object[] getElements(Object inputObject) {
		return datas.toArray();
	}
	
	public void setSelectedIndex(int i) {
		getViewer().getTable().setFocus();
		getViewer().getTable().setSelection(i);
		getViewer().getTable().showSelection();
		getViewer().getTable().redraw();
	}
	
	public void update() {
		getViewer().refresh();
	}
	
	public AlertTable(Composite parent, Object[][] values)
	{
		super(parent);

		parent.addDisposeListener(this);

		String[] names = new String[] {
			EditorPlugin.getString("TABLE_COLUMN_NAME"),
			EditorPlugin.getString("TABLE_COLUMN_PERIOD"),
			EditorPlugin.getString("TABLE_COLUMN_CALCTYPE"),
			EditorPlugin.getString("TABLE_COLUMN_LOWER"),
			EditorPlugin.getString("TABLE_COLUMN_UPPER"),
			EditorPlugin.getString("TABLE_COLUMN_MTYPE"),
			EditorPlugin.getString("TABLE_COLUMN_ONTRIGGER"),
			EditorPlugin.getString("TABLE_COLUMN_ONRESET"),
		};		
		
		int[] types = new int[] {
			TEXT,
			INTEGER,
			CHOICE,
			DOUBLE,
			DOUBLE,
			CHOICE,
			CUSTOM,
			CUSTOM,
		};
		
		setColumnNames(names);	
		setColumnTypes(types);
		
		setColumnEditor(new AlertTableDialogCellEditor(getViewer().getTable()),COL_ONTRIGGER);
		setColumnEditor(new AlertTableDialogCellEditor(getViewer().getTable()),COL_ONRESET);
		
		autoSizeOnContents = new boolean[]{true, true, true, true, true, true, true, true};
//		columnExpands = new boolean[]{true, false, false, false, false, false, false, false};
				
		configure();

		for (int i = 0; i < datas.size(); i++) {
			try {
				MonitorElement ge = (MonitorElement)datas.get(i);
			} catch (Throwable e) {}
		}

		datas = new Vector();
		for (int i = 0; i < values.length; i++) {
			datas.add(new MonitorElement((Alert)values[i][0],
										(String)values[i][1], 
										(Long)values[i][2], 
										(String)values[i][3], 
										(Double)values[i][4], 
										(Double)values[i][5], 
										(Boolean)values[i][6], 
										(AlertAction[])values[i][7], 
										(AlertAction[])values[i][8] ));
		}
		
		
		alert_menu = new Menu(getViewer().getTable());
		alert_delete = new MenuItem(alert_menu,0);
		alert_restart = new MenuItem(alert_menu,0);

		alert_delete.setText(EditorPlugin.getString("ALERT_REMOVE"));
		alert_delete.setImage(EditorPlugin.img.getImage(ImageManager.IMG_SMODEL_REMOVE));
		
		alert_restart.setText(EditorPlugin.getString("ALERT_RESTART"));
		alert_restart.setImage(EditorPlugin.img.getImage(ImageManager.IMG_RUN_EXEC));
		
		getViewer().getTable().addFocusListener(this);
		getViewer().getTable().addSelectionListener(this);
		getViewer().getTable().addMouseListener(this);
		alert_delete.addSelectionListener(this);
		alert_restart.addSelectionListener(this);
	}

	public String [] getComboChoices(Object element, int columnIndex)
	{
		if (columnIndex == COL_MTYPE) {
			return typeChoicesString;
		} else if (columnIndex == COL_CALCTYPE) {
			return calcTypeChoicesString;
		}
			
		return new String[0];
	}
	
	public void setValues(Object[][] values)
	{

		for (int i = 0; i < datas.size(); i++) {
			try {
				MonitorElement ge = (MonitorElement)datas.get(i);
			} catch (Throwable e) {}
		}

		datas.clear();
		for (int i = 0; i < values.length; i++) {
			datas.add(new MonitorElement((Alert)values[i][0],
										(String)values[i][1], 
										(Long)values[i][2], 
										(String)values[i][3], 
										(Double)values[i][4], 
										(Double)values[i][5], 
										(Boolean)values[i][6], 
										(AlertAction[])values[i][7], 
										(AlertAction[])values[i][8] ));
		}
		configure();
	}


	public Object[][] getValues()
	{
		Object[][] result = new Object[datas.size()][8];
		for (int v = 0; v < datas.size(); v++)
		{
			result[v][0] = ((MonitorElement) datas.get(v)).monitor;
			result[v][1] = ((MonitorElement) datas.get(v)).name;
			result[v][2] = ((MonitorElement) datas.get(v)).period;
			result[v][3] = ((MonitorElement) datas.get(v)).calc_type;
			result[v][4] = ((MonitorElement) datas.get(v)).lower;
			result[v][5] = ((MonitorElement) datas.get(v)).upper;
			result[v][6] = ((MonitorElement) datas.get(v)).max_trigger;
			result[v][7] = ((MonitorElement) datas.get(v)).trigger_action;
			result[v][8] = ((MonitorElement) datas.get(v)).reset_action;
		}
		return result;
	}

	public void dispose() {
		super.dispose();
		
		disposables.addAll(datas);
		
		for (int i = 0; i < disposables.size(); i++) {
			try {
			Object o = disposables.get(i);
			if (o != null) {
				if (o instanceof Widget) {
					((Widget)o).dispose();	
				} else if (o instanceof Color) {
					((Color)o).dispose();	
				} else if (o instanceof Image) {
					((Image)o).dispose();	
				} else {
					try {
						EditorPlugin.disposeObject(o);
					} catch (Throwable e) {
						EditorPlugin.DBG.warning("Class not found when disposing of "+o.getClass()+"/"+o+" ("+e+")");
					}
				}
			}
			} catch (Throwable e) {
				EditorPlugin.DBG.error("Problem disposing of objects",e);
			}
		}		
	}
	
	public void valueChanged(Object element, int column, String newValue) throws InvalidTableValueException {
		MonitorElement e = (MonitorElement)element;
		
		boolean changed = false;
		
		if (column == COL_NAME) {
			changed = true;
			e.name = newValue;
		}

		if (changed) {
			e.writeToMonitor();
			getViewer().refresh(e);
		}
	}
	
	public void valueChanged(Object element, int column, Float newValue) throws InvalidTableValueException {
		valueChanged(element,column,new Double(newValue.doubleValue()));
	}
	
	public void valueChanged(Object element, int column, Boolean newValue) throws InvalidTableValueException {
	}
		
	public void valueChanged(Object element, int column, Double newValue) throws InvalidTableValueException {
		double val = newValue.doubleValue();
		
		MonitorElement e = (MonitorElement)element;
		
		boolean changed = false;
		
		if (column == COL_LOWER) {
			changed = true;
			e.lower = newValue;
			
			if (val > e.upper.doubleValue()) {
				e.upper = new Double(val);
			}
			
		} else if (column == COL_UPPER) {
			changed = true;
			e.upper = newValue;
			
			if (val < e.lower.doubleValue()) {
				e.lower = new Double(val);
			}
		}
		
		if (changed) {
			e.writeToMonitor();
			getViewer().refresh(e);
		}
	}
	
	public void valueChanged(Object element, int column, Integer newValue) throws InvalidTableValueException {
		int val = newValue.intValue();
		
		MonitorElement e = (MonitorElement)element;
		
		boolean changed = false;
		
		if (column == COL_PERIOD) {
			changed = true;
			e.period = new Long(val);
			
		} else if (column == COL_CALCTYPE) {
			changed = true;
			e.calc_type = calcTypeChoices[val];
			
		} else if (column == COL_MTYPE) {
			changed = true;
			e.max_trigger = new Boolean(val == 0);
		}

		if (changed) {
			e.writeToMonitor();
			getViewer().refresh(e);
		}
	
	}

	public void valueChanged(Object element, int column) throws InvalidTableValueException {
		//trigger actions changed inside Monitor - read them from the monitor into our element
		
		((MonitorElement)element).readFromMonitor();
		configure();
		getViewer().refresh(element);
//		getViewer().update(element, null);
	}
ArrayList disposables = new ArrayList();
	public class MonitorElement
	{
		public Alert monitor;
		public String name;
		public Long period;
		public String calc_type;
		public Double lower;
		public Double upper;
		public Boolean max_trigger;
		public AlertAction[] trigger_action;
		public AlertAction[] reset_action;
		
		public void writeToMonitor() {
			monitor.name = name;
			monitor.period = period.longValue();
			
			if (calc_type.equals("PLOTTING_TYPE_AVERAGE")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_AVERAGE;
			} else if (calc_type.equals("PLOTTING_TYPE_MAX")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_MAX;
			} else if (calc_type.equals("PLOTTING_TYPE_MIN")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_MIN;
			} else if (calc_type.equals("PLOTTING_TYPE_SUM")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_SUM;
			} else if (calc_type.equals("PLOTTING_TYPE_COUNT")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_COUNT;
			} else if (calc_type.equals("PLOTTING_TYPE_NEAREST")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_NEAREST;
			} else if (calc_type.equals("PLOTTING_TYPE_STDDEV")) {
				monitor.calc_type = Graph.PLOTTING_TYPE_STDDEV;
			} else {
				monitor.calc_type = Graph.PLOTTING_TYPE_AVERAGE;
			}
			
			monitor.lower = lower.doubleValue();
			monitor.upper = upper.doubleValue();
			monitor.max_trigger = max_trigger.booleanValue();
			monitor.trigger_actions = trigger_action;
			monitor.reset_actions = reset_action;
		}
		
		public void readFromMonitor() {
			name = monitor.name;
			period = new Long(monitor.period);
			if (monitor.calc_type == Graph.PLOTTING_TYPE_AVERAGE) {
				calc_type = "PLOTTING_TYPE_AVERAGE";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_MAX) {
				calc_type = "PLOTTING_TYPE_MAX";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_MIN) {
				calc_type = "PLOTTING_TYPE_MIN";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_SUM) {
				calc_type = "PLOTTING_TYPE_SUM";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_COUNT) {
				calc_type = "PLOTTING_TYPE_COUNT";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_NEAREST) {
				calc_type = "PLOTTING_TYPE_NEAREST";
			} else if (monitor.calc_type == Graph.PLOTTING_TYPE_STDDEV) {
				calc_type = "PLOTTING_TYPE_STDDEV";
			} else {
				calc_type = "PLOTTING_TYPE_AVERAGE";
			}
			lower = new Double(monitor.lower);
			upper = new Double(monitor.upper);
			max_trigger = new Boolean(monitor.max_trigger);
			trigger_action = monitor.trigger_actions;
			reset_action = monitor.reset_actions;
		}
		
		public MonitorElement(Alert monitor, 
								String name,
								Long period,
								String calc_type,
								Double lower,
								Double upper,
								Boolean max_trigger,
								AlertAction[] trigger_action,
								AlertAction[] reset_action
								)
		{
			this.monitor = monitor;
			this.name = name;
			this.period = period;
			this.calc_type = calc_type;
			this.lower = lower;
			this.upper = upper;
			this.max_trigger = max_trigger;
			this.trigger_action = trigger_action;
			this.reset_action = reset_action;
		}
	}

	private Object lookup(Object[] indexer, Object[] result, Object o) {
//		int index = Arrays.binarySearch(indexer,o);
		for (int i = 0; i < indexer.length; i++) {
			if (indexer[i].equals(o)) {
				return result[i];
			}
		}
		return result[-1];
	}
	
	public Image getColumnImage(Object element, int columnIndex)
	{
		MonitorElement el = (MonitorElement) element;
		if (el.monitor.isTriggered() && columnIndex == COL_NAME) {
			return EditorPlugin.img.getImage(ImageManager.IMG_ALERT);
		}
		
		return null;
	}
	
	public String getColumnText(Object element, int columnIndex)
	{
		MonitorElement el = (MonitorElement) element;
		
		if (columnIndex == COL_NAME) {
			return el.name;	
		} else if (columnIndex == COL_PERIOD) {
			return el.period+" ms";
		} else if (columnIndex == COL_CALCTYPE) {
			return ""+lookup(calcTypeChoices,calcTypeChoicesString,el.calc_type);
		} else if (columnIndex == COL_LOWER) {
			String ret = df.format(el.lower);
			if (el.max_trigger.booleanValue()) {
				ret = ret + " ("+EditorPlugin.getString("VALUE_RESET")+")";
			} else {
				ret = ret + " ("+EditorPlugin.getString("VALUE_TRIGGER")+")";
			}
			return ret;
		} else if (columnIndex == COL_UPPER) {
			String ret = df.format(el.upper);
			if (el.max_trigger.booleanValue()) {
				ret = ret + " ("+EditorPlugin.getString("VALUE_TRIGGER")+")";
			} else {
				ret = ret + " ("+EditorPlugin.getString("VALUE_RESET")+")";
			}
			return ret;
		} else if (columnIndex == COL_MTYPE) {
			if (el.max_trigger.booleanValue()) {
				return EditorPlugin.getString("TYPE_MAX_TRIGGER");
			} else {
				return EditorPlugin.getString("TYPE_MIN_TRIGGER");
			}
		} else if (columnIndex == COL_ONTRIGGER) {
			if (el.trigger_action.length > 0) {
				return ""+el.trigger_action[0].getDescription();
			} else {
				return EditorPlugin.getString("ALERT_ACTION_NAME_DONOTHING");
			}
		} else if (columnIndex == COL_ONRESET) {
			if (el.reset_action.length > 0) {
				return ""+el.reset_action[0].getDescription();
			} else {
				return EditorPlugin.getString("ALERT_ACTION_NAME_DONOTHING");
			}
		}
		
		return "";
	}

boolean has_focus = false;		
	public Alert getSelectedAlert() {
		if (!has_focus) return null;
		int n = getViewer().getTable().getSelectionIndex();
		if (n == -1) return null;
		MonitorElement element = (MonitorElement)datas.get(n);
		return element.monitor;
	}	
	
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	public void widgetSelected(SelectionEvent e) {
		if (e.getSource() == alert_delete) {
			if (last_selected != null) {
				notifyListenersDelete(last_selected);
			}
		} else if (e.getSource() == alert_restart) {
			if (last_selected != null) {
				last_selected.restart();
			}
		} else {
			Alert a = getSelectedAlert();
			if (a != null) {
				notifyListenersSelect(a);
			}
		}
	}
	public void focusLost(FocusEvent e) {
		has_focus = false;
	}
	public void focusGained(FocusEvent e) {
		has_focus = true;
	}
	
	
	//////////////////////////////
	// popup menu listener
	//////////////////////////////
Alert last_selected;
	public void mouseDoubleClick(MouseEvent e) {}
	public void mouseDown(MouseEvent e) {}
	public void mouseUp(MouseEvent e) {
//		pt = new Point(e.x,e.y);
//		TableItem item = getViewer().getTable().getItem();
//		int index = getViewer().getTable().getSelectionIndex();
		
		last_selected = getSelectedAlert();
		
		if (e.button > 1) {
			alert_menu.setVisible(true);
		}
//		if (index != -1) {
//			notifyListenersDelete(getSelectedAlert());
//		}
	}
	
}