/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;
import java.nio.charset.Charset;
import java.util.Locale;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.IContext;
import org.eclipse.hyades.logging.adapter.IContextListener;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.w3c.dom.Element;
/**
 * Context is a component. A Context instance corresponds to one thread of processing - 
 * sensor, event extractor, parser, outputters etc. 
 * @version 0.10
 */
public abstract class Context extends Component implements IContext
{
	private boolean continuousOperation = false;
	private long maximumIdleTime = 0;
	private long pauseInterval = 0;
	private long pauseCount = 0;
	
	/* Additions for Internationalization support.  As the adapter runs many contexts each coontext
	 * may be targeting a different file format.  This local and charset info allows us to define
	 * the parameters for this context.
	 */
	private Locale locale;
	private String charset;
	
	private boolean hardStop=false;
	
	protected IContextListener listener;
	
	public void stop() {
		super.stop();
		listener=null;
		/* Log the fact we have stopped the contextt */
		CommonBaseEvent stopEvent = getEventFactory().createCommonBaseEvent();

		stopEvent.setMsg(Messages.getString("HyadesGAContext_Stopped_INFO_", getName()));
		stopEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
		Situation stopSituation = getEventFactory().createSituation();
		stopSituation.setStopSituation("INTERNAL","STOP COMPLETED","SUCCESSFUL");
		stopEvent.setSituation(stopSituation);
	    log(stopEvent);
	}
	
	public void resetControledStop()
	{
		resetPauseCount();
	}
	public void resetPauseCount()
	{
		pauseCount = 0;
	}
	public void incrementPauseCount()
	{
		pauseCount++;
	}
	public long getPauseCount()
	{
		return pauseCount;
	}
	public void setPauseCount(long newPauseCount)
	{
		pauseCount = newPauseCount;
	}
	public void setContinuousOperation(boolean newContinuousOperation)
	{
		continuousOperation = newContinuousOperation;
	}
	public boolean getContinuousOperation()
	{
		return continuousOperation;
	}
	public void setMaximumIdleTime(long newMaximumIdleTime)
	{
		maximumIdleTime = newMaximumIdleTime;
	}
	public long getMaximumIdleTime()
	{
		return maximumIdleTime;
	}
	public void setPauseInterval(long newPauseInterval)
	{
		pauseInterval = newPauseInterval;
	}
	public long getPauseInterval()
	{
		return pauseInterval;
	}
	public void update() throws AdapterInvalidConfig
	{
		/* RKD:  As the context is a parent component we want to run its update method
		 * before having our children be updated.  Moving the super.update() to the end of the method
		 * super.update();
		 */
		
		Element element = getConfiguration();
		if (element.hasAttribute(Messages.getString("HyadesGAcontinuousOperationAttributeName")))
			setContinuousOperation(
				Boolean
					.valueOf(element.getAttribute(Messages.getString("HyadesGAcontinuousOperationAttributeName")))
					.booleanValue());
		if (element.hasAttribute(Messages.getString("HyadesGApauseIntervalAttributeName")))
			setPauseInterval(
				Long
					.valueOf(element.getAttribute(Messages.getString("HyadesGApauseIntervalAttributeName")))
					.longValue());
		if (element.hasAttribute(Messages.getString("HyadesGAmaximumIdleTimeAttributeName")))
			setMaximumIdleTime(
				Long
					.valueOf(element.getAttribute(Messages.getString("HyadesGAmaximumIdleTimeAttributeName")))
					.longValue());	
		
		/* Additions for Internationalization support.  As the adapter runs many contexts each coontext
		 * may be targeting a different file format.  This local and charset info allows us to define
		 * the parameters for this context.
		 */
		
		
		String isoLanguageCode=element.getAttribute(Messages.getString("HyadesGAISOLanguageCodeAttributeName"));
		String isoCountryCode=element.getAttribute(Messages.getString("HyadesGAISOCountryCodeAttributeName"));
		charset=element.getAttribute(Messages.getString("HyadesGACharsetAttributeName"));
		
		/* if neither the language nor the country have been specified then we are good to run with the default locale */
		if((isoLanguageCode==null || isoLanguageCode.equals("")) && (isoCountryCode==null || isoCountryCode.equals(""))) {
			locale=Locale.getDefault();
		}
		else {
			/* We have a language and/or a country.  We need to find a locale that supports these */
		
			Locale[] supportedLocals=Locale.getAvailableLocales();
			for(int i=0; i<supportedLocals.length; i++) {
				/* If we have a language lets test this locale is a match */
				if(isoLanguageCode!=null && !isoLanguageCode.equals("")) {
					if(isoLanguageCode.equals(supportedLocals[i].getLanguage())) {
						/* We have a language match, do we need to do a country match now? */
						if(isoCountryCode!=null && !isoCountryCode.equals("")) {
							if(supportedLocals[i].getCountry().equals(isoCountryCode)) {
								/* We have a matching country and langauge, we are golden */
								locale=supportedLocals[i];
								break;
							}
						}
						else {
							/* we were only given the language, this is our only criteria for a match */
							locale=supportedLocals[i];
							
						}
					}
				}
				else {
					/* We don't have a language, we will use the country and take our first hit if we have one */
					if(supportedLocals[i].getCountry().equals(isoCountryCode)) {
						/* We have a matching country, we are golden */
						locale=supportedLocals[i];
						break;
					}
				}
			}
		}
		
		/* Do we still not have a valid locale?  If so we have an invalid configuration */
		if(locale==null) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextInstanceConfiguration_LocaleNotFound_ERROR_", isoLanguageCode, isoCountryCode));
		}
		
		setLocale(locale);
		
		/* Log the fact we are running this context in a specific locale */
		CommonBaseEvent localeEvent = getEventFactory().createCommonBaseEvent();

		localeEvent.setMsg(Messages.getString("HyadesGAContextInstanceConfiguration_ContextLocaleSetting_INFO_",locale.getDisplayName()));
		localeEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
	    log(localeEvent);
		
	    /* If we are not given a specific charset to run with we will default to UTF-8 */
		if(charset==null || charset.equals("")) {
			if (System.getProperty("os.name", "Windows").equals("z/OS") || System.getProperty("os.name", "Windows").equals("OS/390")) {
				/* Default to EBCDIC charset on z/OS */
				charset="IBM-1047";
			}
			else {
				charset="UTF-8";
			}
		}
		/* is the specified charset supported */
		if(!Charset.isSupported(charset)) {
			throw new AdapterInvalidConfig(Messages.getString("HyadesGAContextInstanceConfiguration_CharsetNotSupported_ERROR_", charset));	
		}
		
		/* Log the fact we are running this context with a specified Charset */
		CommonBaseEvent charsetEvent = getEventFactory().createCommonBaseEvent();

		charsetEvent.setMsg(Messages.getString("HyadesGAContextInstanceConfiguration_ContextCharsetSetting_INFO_",charset));
		charsetEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
	    log(charsetEvent);
		
	    /* Update all our children */
	    super.update();
	    
		/* Log the fact we have finished configuring this context */
		CommonBaseEvent configureEvent = getEventFactory().createCommonBaseEvent();

		configureEvent.setMsg(Messages.getString("HyadesGAContextInstanceConfiguration_ConfigurationComplete_INFO_",getName()));
		configureEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
		Situation configSituation = getEventFactory().createSituation();
		configSituation.setConfigureSituation("INTERNAL","SUCCESSFUL");
		configureEvent.setSituation(configSituation);
	    log(configureEvent);
			    
		return;
	}
	
	public void handleControledStop()
	{
		//if this is not a continuous run then stop right now
		if (!getContinuousOperation())
		{
			hardStop=false;
			stop();
		}
		else
		{
			//for a continuous operation, pause and keep track of the total paused time
			if (getMaximumIdleTime() / getPauseInterval() >= getPauseCount())
			{
				try
				{
					synchronized (this)
					{
						Thread.sleep(getPauseInterval());
					}
				}
				catch (Exception e)
				{
				}
				incrementPauseCount();
			}
			else
			{
				hardStop=true;
				stop();
			}
		}
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#setContextListener(IContextListener)
	 */
	public void setContextListener(IContextListener listener) {
		this.listener=listener;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#getContextListener()
	 */
	public IContextListener getContextListener() {
		return listener;
	}
	
	/**
	 * 
	 */
	protected boolean isHardStop() {
		return hardStop;
	}
	/**
	 * Set the hardStop flag.
	 */
	protected void setHardStop() {
		hardStop = true;
		return; 
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#getLocale()
	 */
	public Locale getLocale() {
		return locale;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#setLocale(java.util.Locale)
	 */
	public void setLocale(Locale locale) {
		this.locale=locale;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#getCharset()
	 */
	public String getCharset() {
		return charset;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IContext#setCharset(java.nio.charset.Charset)
	 */
	public void setCharset(String charset) {
		this.charset=charset;
		
	}
}
