/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.parsers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author rduggan
 *
 * This class represents a group of attribute statements
 */
public class GroupStatement implements IStatement {
	
	private List ruleAttributes;
	private String name;
	private IParser parser;
	
	/* Is there only one child allowed for thie group attribute? */
	private boolean isChildChoice=false;
	
	/* Is this statement required for our parent to be valid */
	private boolean isRequiredByParent=false;
	
	/**
	 * 
	 */
	public GroupStatement() {
		super();
	}

	/**
	 * 
	 */
	public GroupStatement(IParser parser) {
		this.parser=parser;
	}
	
	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#prepare(org.w3c.dom.Element)
	 */
	public void prepare(Element node, String path) throws PreparationException {
		
		if(path==null) {
			this.name=node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		else {
			this.name=path+"."+node.getAttribute(Messages.getString("HyadesGANameAttributeName"));
		}
		
		/* Retrieve the index if there is one specified and embed this into the name */
		String index=node.getAttribute(Messages.getString("HyadesGAIndexAttributeName"));
		if(index!=null && !index.equals("")) {
			this.name=this.name+"["+index+"]";
		}
		
		/* retrieve whether this groupStatement has only one child IStatement (ie. a choice) */
		String isChildChoice=node.getAttribute(Messages.getString("HyadesGAisChildChoiceAttributeName"));
		if(isChildChoice!=null && isChildChoice.equals("true")) {
			this.isChildChoice=true;
		}
		else {
			this.isChildChoice=false;
		}
		
		/* retrieve whether this groupStatement has only one child IStatement (ie. a choice) */
		String isRequiredByParent=node.getAttribute(Messages.getString("HyadesGAisRequiredByParentAttributeName"));
		if(isRequiredByParent!=null && isRequiredByParent.equals("true")) {
			this.isRequiredByParent=true;
		}
		else {
			this.isRequiredByParent=false;
		}
		
		
		/* Retrieve all the children and look for Group and RuleAtt elements. */ 
		NodeList children=node.getChildNodes();
		
		ruleAttributes=new ArrayList();
		
		for(int i=0; i<children.getLength(); i++) {
			Element child=null;
			try {
				child=(Element)children.item(i);
			}
			catch(ClassCastException e) {
				/* We can ignore this child as it is not a Element */	
				continue; 	
			}
			if(child.getNodeName().endsWith(Messages.getString("HyadesGARuleElementTagName"))) {
				IStatement statement=new GroupStatement(parser);
				statement.prepare(child, name);
				ruleAttributes.add(statement);
			}
			else if(child.getNodeName().endsWith(Messages.getString("HyadesGARuleAttributeTagName"))) {
				IStatement statement=new AttributeStatement(parser);
				statement.prepare(child, name);
				ruleAttributes.add(statement);	
			}
		}
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IStatement#run(java.lang.String, null)
	 */
	public List run(String line, HashMap inputs) throws AdapterException, StatementException {
		if (ruleAttributes!=null) {
			/* create a new List to put our results in.  We can likely create this only once
			 * and maintain the list to save object creation times.
			 */
			List list=new ArrayList();
			
			ListIterator iterator=ruleAttributes.listIterator();
			while(iterator.hasNext()) {
				IStatement statement=(IStatement)iterator.next();
				
				try {
					list.addAll(statement.run(line, inputs));
					/* If we can have only one child element prune the rest of the processing */
					if(isChildChoice) {
						return list;
					}
				}
				catch(StatementException e) {
					/* If this child was required for us to be valid then we can also prune */
					if(statement.isRequiredByParent()) {
						throw e;
					}
				}
			}
			if(list.isEmpty()) {
				throw StatementException.instance();
			}
			return list;
		}
		throw StatementException.instance();
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isRequiredByParent()
	 */
	public boolean isRequiredByParent() {
		return isRequiredByParent;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.parsers.IStatement#isChildChoice()
	 */
	public boolean isChildChoice() {
		return isChildChoice;
	}
}
