/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.actions.internal.OpenSourceAction;
import org.eclipse.hyades.trace.views.adapter.internal.ClassInstanceStatisticPage;
import org.eclipse.hyades.trace.views.internal.PackageStatistic.PackageTreeViewer;
import org.eclipse.hyades.trace.views.internal.StatisticView.StatisticSorter;
import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.*;

public class ClassInstanceStatistic extends StatisticView
{
	private final TRCClassImpl.TRCClassSnapshot class1Delta = new TRCClassImpl.TRCClassSnapshot();
	private final TRCClassImpl.TRCClassSnapshot class2Delta = new TRCClassImpl.TRCClassSnapshot();				

	public class ClassStatisticFilter extends StatisticFilter {

		public ClassStatisticFilter() {
			super();
		}


		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			if(!(element instanceof TRCClass))
			  return true;
						
			String compareText = "";
			
			if (_noPattern)
				return true;
			
			if(element instanceof TRCClass)   
			   compareText = ((TRCClass)element).getName();
			   
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	public class ClassStatisticSorter extends StatisticSorter {

		public ClassStatisticSorter() {
			super();
		}

		public int compare(Viewer viewer, Object e1, Object e2) {
			
			double d = 0;
			
 			if(e1 instanceof TRCClass)	
			{
				TRCClass c1 = (TRCClass)e1;
				TRCClass c2 = (TRCClass)e2;
				TRCClassImpl.TRCClassSnapshot cs1 = (TRCClassImpl.TRCClassSnapshot)c1.retrieveSnapshot();
				TRCClassImpl.TRCClassSnapshot cs2 = (TRCClassImpl.TRCClassSnapshot)c2.retrieveSnapshot();
				
				
				
				switch(_pos)
				{
					case 0://class name
					  return (int) _sortSequence * (int)c1.getName().compareToIgnoreCase(c2.getName());
					  
					case 1:
					  return (int) _sortSequence * (int)c1.getPackage().getName().compareToIgnoreCase(c2.getPackage().getName());
	
					case 2: //total inst
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getTotalInstances() - cs2.getTotalInstances());
					  }
					  else
					  {
						 c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						 c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						 return (int) _sortSequence * (class1Delta.getTotalInstances() - class2Delta.getTotalInstances());					  	
					  }
					  					  					 
					case 3: //live inst
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getTotalInstances() - cs1.getCollectedInstances()- cs2.getTotalInstances() + cs2.getCollectedInstances());
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * ((class1Delta.getTotalInstances() - class1Delta.getCollectedInstances())
						       - (class2Delta.getTotalInstances() - class2Delta.getCollectedInstances()));					  	
					  	
					  }
					  
					case 4://collected
					  if(!_info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  int inst = cs1.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  double size = (double)cs1.getCollectedInstances()/inst;
						  	  
						  	  inst = cs2.getTotalInstances();
						  	  if(inst == 0) inst = 1;					  	  
						  	  size -= (double)cs2.getCollectedInstances()/inst;
						  	  
							  return (int) _sortSequence * (int)(size*10000);					  	  					  	  
						  }
						  else
						  {										
						  	return (int) _sortSequence * (cs1.getCollectedInstances() - cs2.getCollectedInstances());
						  }
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						return (int) _sortSequence * (class1Delta.getCollectedInstances() - class2Delta.getCollectedInstances());					  	
					  }
					  
					case 5: //total size
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getTotalSize() - cs2.getTotalSize());
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						return (int) _sortSequence * (class1Delta.getTotalSize() - class2Delta.getTotalSize());					  	
					  }
					  
					case 6: //active size
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getTotalSize() - cs1.getCollectedSize()- ((TRCClass)e2).getTotalSize() + ((TRCClass)e2).getCollectedSize());
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						return (int) _sortSequence * ((class1Delta.getTotalSize() - class1Delta.getCollectedSize())
							   - (class2Delta.getTotalSize() - class2Delta.getCollectedSize()));					  	
					  }
					  					  
					case 7: //base time
					  if(!_info.isDeltaColumn())
					  {																				
						  d = cs1.getBaseTime() - cs2.getBaseTime();
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__BASE_TIME);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__BASE_TIME);
						d = class1Delta.getBaseTime() - class2Delta.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 8: //inherited base time
					  if(!_info.isDeltaColumn())
					  {																				
						  d = cs1.getInheritedBaseTime() - cs2.getInheritedBaseTime();
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						d = class1Delta.getInheritedBaseTime() - class2Delta.getInheritedBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 9: //cumulative time
					  if(!_info.isDeltaColumn())
					  {																				
						  d = cs1.getCumulativeTime() - cs2.getCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						d = class1Delta.getCumulativeTime() - class2Delta.getCumulativeTime();					  	
					  }					  
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 10: //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {																				
						  d = cs1.getInheritedCumulativeTime() - cs2.getInheritedCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						d = class1Delta.getInheritedCumulativeTime() - class2Delta.getInheritedCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 11: //calls
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getCalls() - cs2.getCalls());
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__CALLS);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__CALLS);
						return (int) _sortSequence * (class1Delta.getCalls() - class2Delta.getCalls());					  	
					  }

					case 12: //inherited calls
					  if(!_info.isDeltaColumn())
					  {																				
						  return (int) _sortSequence * (cs1.getInheritedCalls() - cs2.getInheritedCalls());
					  }
					  else
					  {
						c1.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CALLS);
						c2.computeDelta(class2Delta,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return (int) _sortSequence * (class1Delta.getInheritedCalls() - class2Delta.getInheritedCalls());					  	
					  }
					  
				}				  				  				 				   				  
			}
			else
			{				 
				TRCObject obj1 = (TRCObject) e1;
				TRCObject obj2 = (TRCObject) e2;  
				
				switch(_pos)
				{
					case 0://instance name
					  return (int) _sortSequence * (int)(obj1.getId() - obj2.getId());
					  
					case 1: //package name
					  return 0;
	
					case 2: //total inst
					  return 0;
					  					 
					case 3: //live inst
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{

						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						int live1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 1 : 0;
						int live2 = class2.getCollectedSize() == 0 ? 1 : 0;
						return (int) _sortSequence * (live1-live2);			   

					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{

						TRCHeapObject object1 = (TRCHeapObject) e1;		
						TRCClass class1 = PerftraceUtil.getClass(object1);
						int live2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 1 : 0;
						int live1 = class1.getCollectedSize() == 0 ? 1 : 0;
						return (int) _sortSequence * (live1-live2);
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						int live1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 1 : 0;
						int live2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 1 : 0;  
						return (int) _sortSequence * (live1-live2);

					}
					return 0;  

					  
					case 4://collected
					
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{

						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						int collected1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 0 : 1;
						int collected2 = class2.getCollectedSize() == 0 ? 0 : 1;
						return (int) _sortSequence * (collected1-collected2);			   

					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;		
						TRCClass class1 = PerftraceUtil.getClass(object1);
						int collected2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 0 : 1;
						int collected1 = class1.getCollectedSize() == 0 ? 0 : 1;
						return (int) _sortSequence * (collected1-collected2);
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						int collected1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? 0 : 1;
						int collected2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? 0 : 1;  
						return (int) _sortSequence * (collected1-collected2);

					}
					return 0;  

					case 5: //total size
					  return (int) _sortSequence * (obj1.getSize() - obj2.getSize());
					  
					case 6: //active size
						
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{
		
						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						int activeSize1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize2 = class2.getCollectedSize() == 0 ? class2.getSize() : 0;
						return (int) _sortSequence * (activeSize1-activeSize2);			   
			
					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;		
						TRCClass class1 = PerftraceUtil.getClass(object1);
						int activeSize2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize1 = class1.getCollectedSize() == 0 ? class1.getSize() : 0;
						return (int) _sortSequence * (activeSize1-activeSize2);
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						int activeSize1 = ((TRCFullTraceObject)obj1).getCollectTime() == 0 ? obj1.getSize() : 0;
						int activeSize2 = ((TRCFullTraceObject)obj2).getCollectTime() == 0 ? obj2.getSize() : 0;  
						return (int) _sortSequence * (activeSize1-activeSize2);

					}
					return 0;  
	
						
					  					  
					case 7: //base time
					
					if (true)
					{
						d = 0;
	
						if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
						{

							TRCHeapObject object2 = (TRCHeapObject) e2;				
							TRCClass class2 = PerftraceUtil.getClass(object2);
							double baseTime1 = ((TRCFullTraceObject)obj1).getBaseTime();
							double baseTime2 = class2.getBaseTime();
							d = baseTime1 - baseTime2;
						}
						else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
						{

							TRCHeapObject object1 = (TRCHeapObject) e1;				
							TRCClass class1 = PerftraceUtil.getClass(object1);
							double baseTime2 = ((TRCFullTraceObject)obj2).getBaseTime();
							double baseTime1 = class1.getBaseTime();
							d = baseTime1 - baseTime2;
						}
						else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
						{
							d = ((TRCFullTraceObject)obj1).getBaseTime() -((TRCFullTraceObject)obj2).getBaseTime();	
						}
	
						if(d < 0)
							return  -1 * _sortSequence;
						if(d > 0) 
							return _sortSequence;
						return 0;	
					} 
					
					case 9: //cumulative time

					if (true)
					{
						d = 0;
						
						if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
						{
		
								TRCHeapObject object2 = (TRCHeapObject) e2;				
								TRCClass class2 = PerftraceUtil.getClass(object2);
								double time1 = ((TRCFullTraceObject)obj1).getCumulativeTime();
								double time2 = class2.getCumulativeTime();
								d = time1 - time2;
						}
						else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
						{
		
							TRCHeapObject object1 = (TRCHeapObject) e1;				
							TRCClass class1 = PerftraceUtil.getClass(object1);
							double time2 = ((TRCFullTraceObject)obj2).getCumulativeTime();
							double time1 = class1.getCumulativeTime();
							d = time1 - time2;
						}
						else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
						{
							d = ((TRCFullTraceObject)obj1).getCumulativeTime()-((TRCFullTraceObject)obj2).getCumulativeTime();	
						}
						
						if(d < 0)
							return  -1 * _sortSequence;
						if(d > 0) 
							return _sortSequence;
						return 0;	
						
					} 
					  
					case 11: //calls
					
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{
						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						return (int) _sortSequence * (((TRCFullTraceObject)obj1).getCalls()-class2.getCalls());
							
					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;				
						TRCClass class1 = PerftraceUtil.getClass(object1);
						return (int) _sortSequence * (class1.getCalls()-((TRCFullTraceObject)obj2).getCalls());
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						return (int) _sortSequence * (((TRCFullTraceObject)obj1).getCalls()-((TRCFullTraceObject)obj2).getCalls());
					}
					return 0;
					
					
					  
				}			  				  				 				   				  
			}
			
			return 0;
		}
		
	}
	
	public class ClassStatisticContentProvider implements ITreeContentProvider {
		
		public void dispose() {
		}
		
		public Object getParent(Object element) {
			if(element instanceof TRCObject)
			   PerftraceUtil.getClass((TRCObject)element);
			return null;
		}
		
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
		
		public Object[] getElements(Object inputElement) {
			return PerftraceUtil.getAllClasses(_page.getMOFObject());
		}
				
		public Object[] getChildren(Object element)	{
			tmpList.clear();
			if(element instanceof TRCClass) {
				TRCClass cls = (TRCClass)element;
				List list = cls.getObjects();

				/*
				 * #48680: Use only TRCFullTraceObjects if they
				 * are present. 
				 */
				Iterator iter = cls.getObjects().iterator();
				while (iter.hasNext()) {
					Object obj = iter.next();
					if (obj instanceof TRCFullTraceObject) {
						tmpList.add(obj);
					}
				}
				
				/*
				 * #48680: If there are no TRCFullTraceObjects,
				 * then include the TRCHeapObjects.
				 */
				if (tmpList.isEmpty()) {
					tmpList.addAll(list);
				}
   			}
			return tmpList.toArray(); 	
		}
	
		public boolean hasChildren(Object element) {
			if(element instanceof TRCClass) {
				TRCClass cls = (TRCClass)element;
			    return (cls.getObjects().size() > 0);
			}
			return false;
		}
	}
	
	public class ClassStatisticLabelProvider extends LabelProvider implements ITableLabelProvider
	{
		protected StatisticView _viewer;
		
		public ClassStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}
		
		public Image getColumnImage(Object obj, int col) {
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			   
			int state = 0;
			double d = 0;
			if(obj instanceof TRCClass)	
			{
				TRCClass clas = (TRCClass)obj;

				if(((TRCClass)obj).computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES)<0)
				{
					if(col == 1)
						return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					return null;	
				}
								 
				switch(pos)
				{
					case 2: //total inst
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
					  state = class1Delta.getTotalInstances();
					  break;
					  					 
					case 3: //live inst	
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
					  state = class1Delta.getTotalInstances() - class1Delta.getCollectedInstances();
					  break;
					  
					case 4://collected
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);					
					  state = class1Delta.getCollectedInstances();
					  break;
					  
					case 5: //total size
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);					
					  state = class1Delta.getTotalSize();
					  break;
					  
					case 6: //active size
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_SIZE);
					
					  state = class1Delta.getTotalSize() - class1Delta.getCollectedSize();
					  break;
									  					  
					case 7: //base time		
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__BASE_TIME);
					  d = class1Delta.getBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;

					case 8: //inherited base time
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);					
					  d = class1Delta.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  break;
					  					  
					case 9: //cumulative time
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__CUMULATIVE_TIME);										
					  d = class1Delta.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 10: //inherited cumulative time
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);										
					  d = class1Delta.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 11: //calls
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__CALLS);
					  state = class1Delta.getCalls();
					  break;
					  
					case 12: //inherited calls
					  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CALLS);
					  state = class1Delta.getInheritedCalls();
					  break;
					
				}				  				  				 				   				  
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
			
		}

		public String getColumnText(Object obj, int col) {
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCClass)	
			{
				TRCClass clas = (TRCClass) obj;
				TRCClassImpl.TRCClassSnapshot cs = (TRCClassImpl.TRCClassSnapshot)clas.retrieveSnapshot();

				switch(pos)
				{
					case 0://class name
					  return clas.getName();
					  
					case 1: //package name
 						return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();
					
					case 2: //total inst
					
					  if(!info.isDeltaColumn())
					  {					
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getTotalInstances()/_totalInst);
						  }
						  else
						  {					
							  return String.valueOf(cs.getTotalInstances());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						  return String.valueOf(class1Delta.getTotalInstances());					  	
					  }
					  					 
					case 3: //live inst
					
					  if(!info.isDeltaColumn())
					  {										
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getTotalInstances() - cs.getCollectedInstances())/_totalInst);
						  }
						  else
						  {					
							  return String.valueOf(cs.getTotalInstances() - cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_INSTANCES);
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						  return String.valueOf(class1Delta.getTotalInstances() - class1Delta.getCollectedInstances());					  	
					  }
					  
					case 4://collected
					  if(!info.isDeltaColumn())
					  {															
						  if(isShowPercent())
						  {
						  	  int inst = clas.getTotalInstances();
						  	  if(inst == 0) inst = 1;
						  	  
						  	  return TString.formatAsPercentage((double)cs.getCollectedInstances()/inst);
						  }
						  else
						  {									
							  return String.valueOf(cs.getCollectedInstances());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_INSTANCES);
						  return String.valueOf(class1Delta.getCollectedInstances());					  	
					  }
					  
					case 5: //total size
					
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getTotalSize()/_totalSize);
						  }
						  else
						  {									
							  return String.valueOf(cs.getTotalSize());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						  return String.valueOf(class1Delta.getTotalSize());					  	
					  }
					  
					case 6: //active size
					
					  if(!info.isDeltaColumn())
					  {																									
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getTotalSize() - cs.getCollectedSize())/_activeSize);
						  }
						  else
						  {														
							  return String.valueOf(cs.getTotalSize() - cs.getCollectedSize());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__TOTAL_SIZE);
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__COLLECTED_SIZE);
						  return String.valueOf(class1Delta.getTotalSize() - class1Delta.getCollectedSize());					  	
					  }
					  					  
					case 7: //base time
					
					  if(!info.isDeltaColumn())
					  {																														
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getBaseTime()/_maxTime);
						  }
						  else
						  {										
							  return TString.formatTimeValue(cs.getBaseTime());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__BASE_TIME);
						  return TString.formatTimeValue(class1Delta.getBaseTime());					  	
					  }

					case 8: //inherited base time
					
					  if(!info.isDeltaColumn())
					  {																																			
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {										
							  return TString.formatTimeValue(cs.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						  return TString.formatTimeValue(class1Delta.getInheritedBaseTime());					  	
					  }
					  					  
					case 9: //cumulative time
					
					  if(!info.isDeltaColumn())
					  {																																								
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getCumulativeTime()/_maxTime);
						  }
						  else
						  {															
							  return TString.formatTimeValue(cs.getCumulativeTime());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						  return TString.formatTimeValue(class1Delta.getCumulativeTime());					  	
					  }

					case 10: //inherited cumulative time
					
					  if(!info.isDeltaColumn())
					  {																																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getInheritedCumulativeTime()/_maxTime);
						  }
						  else
						  {															
							  return TString.formatTimeValue(cs.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						  return TString.formatTimeValue(class1Delta.getInheritedCumulativeTime());					  	
					  }
					  
					case 11: //calls
					
					  if(!info.isDeltaColumn())
					  {																																																		
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getCalls()/_totalCalls);
						  }
						  else
						  {															
							  return String.valueOf(cs.getCalls());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__CALLS);
						  return String.valueOf(class1Delta.getCalls());					  	
					  }

					case 12: //inherited calls
					
					  if(!info.isDeltaColumn())
					  {																																																							
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)cs.getInheritedCalls()/_totalCalls);
						  }
						  else
						  {															
							  return String.valueOf(cs.getInheritedCalls());
						  }
					  }
					  else
					  {
						  clas.computeDelta(class1Delta,TracePackage.TRC_CLASS__INHERITED_CALLS);
						  return String.valueOf(class1Delta.getInheritedCalls());					  	
					  }
					  
				}				  				  				 				   				  
			}
			else
			{
//				if(!(obj instanceof TRCFullTraceObject))
//				   return "";
				   
				if(info.isDeltaColumn())
				   return "";

				TRCObject object = (TRCObject)obj;				
	            //TRCObject
				switch(pos)
				{
					case 0://class name
					{
						TRCClass clas = PerftraceUtil.getClass(object);
						if(clas ==object.getProcess().getClassClass())
						    return clas.getName();					
										
						return clas.getName() + "." + object.getId();
					}
					  
					case 1: //package name
					{
					    return "";
					}
	
					case 2: //total inst
					  return "";
					  					 
					case 3: //live inst
					  return "";
					  
					case 4://collected
					{
					  if(object instanceof TRCFullTraceObject)
					  {
						if(((TRCFullTraceObject)object).getCollectTime() == 0)
						   return "0";
						    
						 return "1";

					  }
					  
					  return "";
				    } 
					  
					case 5: //total size
					{
					   return String.valueOf(((TRCObject)obj).getSize());
				    } 
					  
					case 6: //active size
					{
						if(object instanceof TRCFullTraceObject)
						{
						
						  if(((TRCFullTraceObject)object).getCollectTime() != 0)
						    return "0";
						    
						  return String.valueOf(object.getSize());
						}
						
						return "0";
				    } 
					  					  
					case 7: //base time
					
					if(object instanceof TRCFullTraceObject)
					{
					  
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getBaseTime()/_maxTime);
					  }
					  else
					  {													
						  return TString.formatTimeValue(((TRCFullTraceObject)object).getBaseTime());
					  }
					}
					return "";
					  					  
					case 9: //cumulative time
					
					if(object instanceof TRCFullTraceObject)
					{
					
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getCumulativeTime()/_maxTime);
					  }
					  else
					  {																	
						  return TString.formatTimeValue(((TRCFullTraceObject)object).getCumulativeTime());
					  }
					}
					return "";
					  
					case 11: //calls
					
					if(object instanceof TRCFullTraceObject)
					{
					
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getCalls()/_totalCalls);
					  }
					  else
					  {																	
						  return String.valueOf(((TRCFullTraceObject)object).getCalls());
					  }
					}
					
					return "";
						  
				}
			}
            							
			return "";
		}
	}

	class ClassTreeViewer extends TableTreeViewer {
		public ClassTreeViewer(Composite parent) {
			super(parent);
		}

		public ClassTreeViewer(TableTree table) {
			super(table);
		}

		public void expandItem(TableTreeItem item) {
			item.setExpanded(true);
			createChildren(item);

		}

	}
	protected Composite createTable(Composite parent, int options) {
		return new TableTree(parent, options);
	}

	public ClassInstanceStatistic(Composite parent, TraceViewerPage page) {
		super(parent, page);
		_viewerFilter = new ClassStatisticFilter();				
	}

	/**
	 * getClassColumnsPerferencesKey returns the string key used to find the
	 * column preferences information in the preferences store.
	 * @return java.lang.String key name
	 */
	public String getColumnsPreferencesKey() {
		// Note this string is not to be translated and must be changed whenever the 
		// default columns template is changed by adding, removing or renaming columns.
		// Changing the version will result in the default preferences being used.
		return "ClassI60"; 
		
	}
	
	public String getDefaultColumnsTemplate() {
		//Class Columns Data
		String classColumn =
			TraceUIPlugin.getString("STR_ST_CLASS_NAME") 	+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
		   	+ TraceUIPlugin.getString("STR_ST_PACKAGE") 		+ ":1:"
		   		+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
			+ TraceUIPlugin.getString("STR_ST_INSTANCES")		+ ":2:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_LIVE_INSTANCES")+ ":3:"  
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_COLLECTED")		+ ":4:"
			    + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"				
			+ TraceUIPlugin.getString("STR_ST_SIZE") 			+ ":5:"  
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"				
			+ TraceUIPlugin.getString("STR_ST_LIVE_STORAGE") 	+ ":6:"  
				+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":7:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_CPU_INHTIME")		+ ":8:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_STACK_TIME") 	+ ":9:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_STACK_INHTIME") 	+ ":10:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_CALLS") 		+ ":11:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:50,"
			+ TraceUIPlugin.getString("STR_ST_INHCALLS") 		+ ":12:"
			+ String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100"			
		;
		return classColumn;
	}

	protected IContentProvider getContentProvider() {
		return new ClassStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new ClassStatisticLabelProvider(this);
	}
	

	public Table getTable() {
		return ((TableTree) getTableViewer().getControl()).getTable();
	}

	protected TableTreeViewer getTableTreeViewer(TableTree table)
	{
		return new ClassTreeViewer(table);
	}
	
	protected void handleSelectionEvent() {
		((ClassInstanceStatisticPage) getTraceViewerPage()).openSource().setEnabled(false);

		ITraceSelection model = UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0 )
		{
			((ClassInstanceStatisticPage) getTraceViewerPage()).openSource().setEnabled(true);

			Object sel = model.getFirstElement();
			// If the selection is an message then turn the selection into the class of the object
			if (sel instanceof TRCMethod) {
				sel = ((TRCMethod) sel).getDefiningClass();
			}
			else if (sel instanceof TRCObject) {
				select((TRCObject)sel);
			}
			else if (sel instanceof TRCClass) {
				select((TRCClass)sel);
			}
			else if(sel instanceof TRCMethodInvocation)
			{
				select(PerftraceUtil.getClass((TRCMethodInvocation)sel));
			}
			
			
		}
	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		menu.add(fSeparator);

		menu.add(getUpdateAction());		
		menu.add(fSeparator);
		menu.add(getChooseColumnsAction(getColumnDataList(), getColumnsPreferencesKey()));
	}

	private void select(TRCObject obj) {
		
		boolean isClassObject = true;
		TRCClass cls = obj.getIsA();
		
		TableTree table = ((TableTreeViewer)getTableViewer()).getTableTree();
		
		TableTreeItem[] items = table.getItems();
		for (int idx = 0; idx < items.length; idx++) {
			TableTreeItem item = items[idx];
			if(item.getData() == cls)
			{
				((ClassTreeViewer) getTableViewer()).expandItem(item);
				TableTreeItem[] childItems = item.getItems();
	
				for (int i = 0; i < childItems.length; i++) {
					TableTreeItem childItem = childItems[i];
					if (obj == childItem.getData()) {
							((ClassTreeViewer) getTableViewer()).expandItem(item);
							table.setSelection(new TableTreeItem[] { childItem });
							return;
					}
				}
			}
		}
		
		table.deselectAll();				
	}
	
	private void select(TRCClass obj) {
		
		TableTree table = ((TableTreeViewer)getTableViewer()).getTableTree();
        
		TableTreeItem[] items = table.getItems();
		for (int idx = 0; idx < items.length; idx++) {
			Object data = items[idx].getData();
			if (data != null && data == obj) {
				table.setSelection(new TableTreeItem[] {items[idx]});
				
				return;
			}
		}
		
		table.deselectAll();		
	}
	

	public void updateButtons() {
		((ClassInstanceStatisticPage) getTraceViewerPage()).openSource().setEnabled(canOpenSourceOnThisObject());
		((ClassInstanceStatisticPage) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());		
		((ClassInstanceStatisticPage) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
	}

	private boolean canOpenSourceOnThisObject() {
		ITraceSelection model =	UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() == 0 || model.getFirstElement() == null) {
			return false;
		}
		Object sel = model.getFirstElement();
		return (sel instanceof TRCClass ||
				sel instanceof TRCHeapObject ||
				sel instanceof TRCFullTraceObject);
	}

	protected void updateDetailsPane() {
		int selIndex = getTable().getSelectionIndex();
		if (selIndex < 0) {
			return;
		}

		Item item = getTable().getItem(selIndex);
		Object itemData = item.getData();

		if (itemData == null) {
			return;
		}
		if (itemData != null && itemData instanceof TableTreeItem) {
			itemData = ((TableTreeItem) itemData).getData();
		}
	}

	public void updateModelSelection() {
		
		ISelection selection = getTableViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			
			notifyViewSelectionChanged(this,sel);
		}
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new ClassStatisticSorter();
	}
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());	
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());	
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());	
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());						
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
	   }			
	}
	
	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event)
	{
		Object  source = event.getSource();
		if(source!=this){	
			handleSelectionEvent();
		}
	}
	
	public boolean isEmpty()
	{
		if (PerftraceUtil.getAllClasses(_page.getMOFObject()).length > 0)
			return false;
		return true;
	}
	
    protected String getViewTypeStr()
    {
    	return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.CLASS_INSTANCE_STATS_VIEW;
    }	
}
