package org.eclipse.hyades.logging.java;

import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import org.eclipse.hyades.internal.logging.core.XmlGenerator;
import org.eclipse.hyades.logging.core.IExternalizableToXml;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Extension of the Java Logging <code>java.util.logging.Formatter</code>
 * class used by <code>java.util.logging.Handler</code> instances to format
 * <code>java.util.logging.LogRecord</code> instances to XML.
 * <p>
 * If the <code>java.util.logging.LogRecord</code> instance is an
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code> (e.g.
 * instance of the
 * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
 * interface), the encapsulated Common Base Event is serialized to XML based on
 * the Common Base Event v.1.0.1 specification. Otherwise, the
 * <code>java.util.logging.LogRecord</code> is serialized to XML based on a
 * generic serialization algorithm as implemented in
 * <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
 * <p>
 * An instance of this class will be returned from the
 * <code>getFormatter()</code> API on <code>java.util.logging.Handler</code>
 * instances.
 * <p>
 * The default <code>java.util.logging.LogManager</code> implementation uses
 * the following configuration variable in the default
 * &lt;JRE&gt;/lib/logging.properties configuration file to load configuration
 * for <code>java.util.logging.Logger</code> instances:
 * <p>
 * <handler name>.formatter = org.eclipse.hyades.logging.java.XmlFormatter
 * <p>
 * Alternatively, an instantiation of this formatter class may be set directly
 * on <code>java.util.logging.Handler</code> instances by using the
 * <code>setFormatter()</code> API.
 * <p>
 * NOTE: The Java Logging classes must be on the CLASSPATH at run-time to
 * utilize this formatter class.
 * <p>
 * 
 * @author   Paul E. Slauenwhite
 * @version  July 22, 2004
 * @since    April 1, 2003
 * @see      java.util.logging.Formatter
 * @see      java.util.logging.LogRecord
 * @see      org.eclipse.hyades.internal.logging.core.XmlGenerator
 * @see      org.eclipse.hyades.logging.core.IExternalizableToXml
 */
public class XmlFormatter extends Formatter {

    /**
     * Private reference to the generic XML serialization class.
     */
    private XmlGenerator xmlGenerator = new XmlGenerator();
    
    /**
     * Private static reference to the platform-dependent line separator character.
     */
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    
    /**
     * Generates the XML representation of the parameter
     * <code>java.util.logging.LogRecord</code> instance.
     * <p>
     * If the parameter <code>java.util.logging.LogRecord</code> instance is
     * an <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
     * (e.g. instance of the
     * <code>org.eclipse.hyades.logging.core.IExternalizableToXml</code>
     * interface), the encapsulated Common Base Event is serialized to XML based
     * on the Common Base Event v.1.0.1 specification. Otherwise, the
     * <code>java.util.logging.Handler</code> is serialized to XML based on a
     * generic serialization algorithm as implemented in
     * <code>org.eclipse.hyades.internal.logging.core.XmlGenerator</code>.
     * <p>
     * 
     * @param logRecord
     *            The <code>java.util.logging.LogRecord</code> instance to be
     *            formatted to XML.
     * @return The XML representation of the parameter
     *         <code>java.util.logging.LogRecord</code> instance.
     * @see java.util.logging.Formatter#format(LogRecord)
     */
    public String format(LogRecord record) {

        if (record instanceof IExternalizableToXml) { 
            return (((IExternalizableToXml) record).externalizeCanonicalXmlString().concat(LINE_SEPARATOR)); 
        }
        
        if(xmlGenerator == null){
            xmlGenerator = new XmlGenerator();
        }
        
        xmlGenerator.reset(null, true, 4);
        
        return (xmlGenerator.objectToXML(record).concat(LINE_SEPARATOR));
    }
}