/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.recorder;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.hyades.internal.execution.recorder.ui.actions.IRecorderAction;

/**
 * Factory which manages recorders 
 * This object is a singleton which should be accessed via getInstance().
 * @author Ernest Jessee
 *
 */
public class RecorderFactory
{
	/**
	 * stores default instance
	 */
	private static RecorderFactory instance = null;
	
	
	/**
	 * HashMap for storing recorder reference with ID key
	 */
	private HashMap recorderMap = new HashMap();
	
	/**
	 * Hashmap for storing actions to listen to the active state of the recorder 
	 */
	private HashMap listeningActionMap = new HashMap();
	
	/**
	 * String array of registered recorders
	 */
	private String[] recorderIDs = null;
		
		
	/**
	 * This is the method called to acquire a reference to the RecorderFactory
	 * 
	 */
	public static RecorderFactory getInstance()
	{
		
		if(instance == null)
		{
			instance = new RecorderFactory();
		}	
		
		return instance;
	}
	
	
	/**
	 * called to add a recorder to the factory.  This should not be called directly
	 * by any object other than the recorder object itself.  When a recorder is 
	 * constructed, it automatically adds itself to the factory
	 * 
	 * @param Recorder recorder
	 */
	protected void addRecorder(Recorder recorder)
	{
		recorderMap.put(recorder.getId(), recorder);
	}

	
	/**
	 * private constructor
	 * @see getInstance
	 */
	private RecorderFactory()
	{	
	}
	
	/**
	 * Called to get a String Array of the Id's of the recorders managed
	 * by this factory.
	 * @return String[]
	 */
	public String[] getRecorderIDs()
	{
		if(recorderIDs == null)
		{
			Object keys[] = recorderMap.keySet().toArray();
			recorderIDs = new String[keys.length];
			for(int i=0; i<keys.length; i++)
			{
				recorderIDs[i]=(String)keys[i];
			}
		}
		
		
		return recorderIDs;
	}

		
	
	/**
	 * called to get a reference to a recorder with a specific ID.
	 * @param String ID
	 * @return Recorder
	 */
	public Recorder getRecorderWithID(String ID)
	{
		return (Recorder)recorderMap.get(ID);
	}
	
	
	/**
	 * returns the currently active recorder
	 * @return Recorder
	 */
	public Recorder getActiveRecorder()
	{
		Collection recorders = recorderMap.values();
		Iterator recorderIterator = recorders.iterator();
		while(recorderIterator.hasNext())
		{
			Recorder recorder = (Recorder)recorderIterator.next();
			if(recorder.isActive())
				return recorder;
		}
		
		return null;
		
	}	

	/**
	 * called to add a listening action to the factory.  This should not be called directly
	 * by any object other than the action object itself.  When an action is 
	 * constructed, it automatically adds itself to the factory
	 * 
	 * @param IRecorderAction action
	 */
	public void addListener(IRecorderAction action)
	{
		listeningActionMap.put(action.toString(), action);
		if (getActiveRecorder() == null)
			action.updateRecorderStatus(false);
		else 
			action.updateRecorderStatus(getActiveRecorder().isActive());
	}
	
	public void updateActiveState(Recorder recorder, boolean bActive)
	{
		//if (recorder == getActiveRecorder())
		//{
			Object[] actions = listeningActionMap.values().toArray();
			for (int i=0; i < actions.length; i++)
			{
				if (actions[i] instanceof IRecorderAction)
				{
					((IRecorderAction)actions[i]).updateRecorderStatus(bActive);
				}
			}
		//}
	}
}
