/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.launch.processes;

import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamsProxy;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.harness.IExecutionHarnessDataProcessor;
import org.eclipse.hyades.execution.harness.TestExecutionHarness;
import org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.loaders.common.ExecutionContext;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.ui.TestUIPlugin;
import org.eclipse.hyades.test.ui.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.ui.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.launch.extensions.IRunHandler;

/**
 * Fake process that represents a Hyades Execution Session. The lifecycle of this process
 * is mapped to the one of an IExecutor Execution Component.
 * @author jcanches
 */
public class TestExecutionProcess implements IProcess {
	
	private IExecutor executor;
	private ILaunch launch;
	private boolean terminated = false;
	private TPFTest test;
	private TPFExecutionResult executionResult;
	
	public TestExecutionProcess(IExecutor executor, ILaunch launch, ResourceSet resourceSet) throws CoreException {
		this.executor = executor; 
		this.launch = launch;
		this.test = TestLaunchConfigurationFacade.getTest(launch.getLaunchConfiguration(), resourceSet);
		synchronized(executor) {
			if (executor.getState() == IExecutionComponent.DEAD) {
				onTerminate();
			} else {
				this.executor.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
					public void stateChanged(ExecutionComponentStateChangeEvent newState) {
						onTerminate();
					}
				});
			}
			// Work-around: the Executor is not signaling its state change. If available,
			// we monitor the control agent instead.
			this.setupExecutorStateMonitoring(executor);
		}
		if (executor instanceof ExecutorStub) {
			ExecutorStub exStub = (ExecutorStub) executor;
			if (exStub.getDataProcessors().length > 0) {
				IDataProcessor dataProcessor = exStub.getDataProcessors()[0];
				this.executionResult = getExecutionResult(dataProcessor);
			}
		}
	}
	
	private void setupExecutorStateMonitoring(IExecutor executor) {
		
		class ControlAgentListener implements AgentListener {
			public void agentActive(Agent agent) {
				// NOP
			}
			public void agentInactive(Agent agent) {
				TestExecutionProcess.this.onTerminate();
			}
			public void error(Agent agent, String errorId, String errorMessage) {
				System.out.println("Error from tester agent: id=" + errorId + ", " + errorMessage);
			}
			public void handleCommand(Agent agent, CommandElement command) {
				// NOP
			}
		}
		
		if (executor != null && executor instanceof ExecutorStub) {
			ExecutorStub es = (ExecutorStub)executor;
			IDataProcessor[] dataProcessors = es.getDataProcessors();
			if (dataProcessors.length > 0 && dataProcessors[0] instanceof IExecutionHarnessDataProcessor) {
				IExecutionHarnessDataProcessor dp = (IExecutionHarnessDataProcessor)dataProcessors[0];
				Agent controlAgent = dp.getControlAgent();
				if (controlAgent != null) {
					controlAgent.addAgentListener(new ControlAgentListener());
				}
			}
		}

	}
	public String getLabel() {
		return TestUIPlugin.getString("TestExecutionProcess.testProcess") + " [" + this.test.getName() + "]"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

	public ILaunch getLaunch() {
		return launch;
	}

	public IStreamsProxy getStreamsProxy() {
		// TODO Consider supporting this feature ?
		// null means not supported
		return null;
	}

	public void setAttribute(String key, String value) {
		// We don't support attributes. We don't need them anyway.
	}

	public String getAttribute(String key) {
		return null;
	}
	
	public int getExitValue() /*throws DebugException*/ {
		// TODO Consider providing something else
		return 0;
	}

	public Object getAdapter(Class adapter) {
		return null;
	}
	
	public boolean canTerminate() {
		return executionResult != null;
	}

	public synchronized boolean isTerminated() {
		return this.terminated || executor.getState() == IExecutionComponent.DEAD;
	}
	
	public void terminate() throws DebugException {
		TestExecutionHarness teh = new TestExecutionHarness();
		String errorMessage = teh.stopTest(this.executor, this.executionResult);
		if (errorMessage != null) {
			IStatus status = new Status(IStatus.ERROR, TestUIPlugin.getID(), 0, errorMessage, null);
			throw new DebugException(status);
		}
	}

	protected synchronized void onTerminate() {
		if (!terminated) {
			// Notify the run handler (if any)
			IRunHandler runHandler = LaunchConfigurationExtensionsManager.getInstance().getRunHandler(test);
			if (runHandler != null) {
				runHandler.postRun(launch);
			}
			this.terminated = true;
			// Notify the event
			DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[]{new DebugEvent(this, DebugEvent.TERMINATE)});
		}
	}
	
	/**
	 * Returns the TPFExecutionResult object that the execution of the test
	 * is producing. This method may return null in non-standard cases.
	 * @return
	 */
	private TPFExecutionResult getExecutionResult(IDataProcessor dataProcessor) {
		if (dataProcessor instanceof XMLExecutionDataProcessor) {
			XMLExecutionDataProcessor dp = (XMLExecutionDataProcessor) dataProcessor;
			ExecutionContext ec = (ExecutionContext)dp.getContext().getCustomData().get(ExecutionContext.root);
			if (ec != null) {
				Resource res = ec.getResource();
				Iterator it = res.getContents().iterator();
				while (it.hasNext()) {	
					EObject obj = (EObject) it.next();
					if (obj instanceof TPFExecutionResult) {
						return (TPFExecutionResult)obj;
					}
				}
			}
		}
		return null;
	}
	
	public TPFExecutionResult getExecutionResult() {
		return this.executionResult;
	}
	
	public IExecutor getExecutor() {
		return this.executor;
	}
}
