package org.eclipse.hyades.logging.java.tests;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;
import org.eclipse.hyades.logging.java.CommonBaseEventLevel;
import org.eclipse.hyades.logging.java.CommonBaseEventLogRecord;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * All JUnit test cases for the
 * <code>org.eclipse.hyades.logging.java.CommonBaseEventLogRecord</code>
 * class.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version September 30, 2004
 * @since August 17, 2004
 */
public class CommonBaseEventLogRecordTest extends TestCase {

    private EventFactory eventFactory = null;

    /**
     * Constructor for CommonBaseEventLogRecordTest.
     * <p>
     * 
     * @param name
     *            The name of the test case.
     */
    public CommonBaseEventLogRecordTest(String name) {
        super(name);
    }

    /**
     * @see junit.framework.TestCase#setUp()
     */
    protected void setUp() throws Exception {

        super.setUp();

        eventFactory = EventFactoryContext.getInstance().getSimpleEventFactoryHome().getAnonymousEventFactory();
    }

    /**
     * Test an empty logger name.
     */
    public void testEmptyStringDataInExtendedDataElementValues() {

        try {

            CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);
            commonBaseEventLogRecord.setLoggerName("");

            assertEquals(commonBaseEventLogRecord.getLoggerName(), "");

            assertTrue(commonBaseEvent.getExtendedDataElements().size() > 1);

            assertEquals(((ExtendedDataElement) (commonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getName(), CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);

            assertEquals(((ExtendedDataElement) (commonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);

            assertEquals(((ExtendedDataElement) (commonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValues().size(), 1);

            assertEquals(((String) (((ExtendedDataElement) (commonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValues().get(0))), "");

            String commonBaseEventString = EventFormatter.toCanonicalXMLDocString(commonBaseEvent);

            CommonBaseEvent serializedCommonBaseEvent = EventFormatter.eventsFromCanonicalXMLDoc(commonBaseEventString)[0];

            CommonBaseEventLogRecord serializedCommonBaseEventLogRecord = new CommonBaseEventLogRecord(serializedCommonBaseEvent);

            assertEquals(serializedCommonBaseEventLogRecord.getLoggerName(), "");

            assertTrue(serializedCommonBaseEvent.getExtendedDataElements().size() > 1);

            assertEquals(((ExtendedDataElement) (serializedCommonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getName(), CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME);

            assertEquals(((ExtendedDataElement) (serializedCommonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);

            assertEquals(((ExtendedDataElement) (serializedCommonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValues().size(), 1);

            assertEquals(((String) (((ExtendedDataElement) (serializedCommonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValues().get(0))), "");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test single argument constructor with no Common Base Event severity and
     * creationTime.
     */
    public void testSingleArgumentConstructorNoCBESeverityCreationTime() {

        try {

            LogRecord logRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (logRecord.getSequenceNumber() + 1);

            final int THREAD_ID = logRecord.getThreadID();

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            final long START_TIME = System.currentTimeMillis();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            final long END_TIME = System.currentTimeMillis();

            assertNull(commonBaseEventLogRecord.getMessage());

            assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getLevel(), Level.OFF);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 0);

            assertFalse(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.OFF));

            assertTrue(commonBaseEventLogRecord.getMillis() >= START_TIME);

            assertTrue(commonBaseEventLogRecord.getMillis() <= END_TIME);

            assertTrue((END_TIME - START_TIME) <= 1000);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), commonBaseEventLogRecord.getMillis());

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), SEQUENCE_NUMBER);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), SEQUENCE_NUMBER);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            assertNull(commonBaseEventLogRecord.getThrown());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test single argument constructor with the Common Base Event severity and
     * creationTime set.
     */
    public void testSingleArgumentConstructorCBESeverityCreationTime() {

        try {

            final long CURRENT_TIME = System.currentTimeMillis();

            //Spin for 1 second:
            while (System.currentTimeMillis() < (CURRENT_TIME + 1000)) {
                ;
            }

            LogRecord logRecord = new LogRecord(Level.OFF, null);

            long sequenceNumber = (logRecord.getSequenceNumber() + 1);

            final int THREAD_ID = logRecord.getThreadID();

            //severity == 50:
            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();
            commonBaseEvent.setCreationTimeAsLong(CURRENT_TIME);
            commonBaseEvent.setSeverity((short) 50);

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getLevel().getName(), Level.SEVERE.getName());

            assertEquals(commonBaseEventLogRecord.getLevel().intValue(), Level.SEVERE.intValue());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

            assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.SEVERE));

            assertEquals(commonBaseEventLogRecord.getMillis(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            assertNull(commonBaseEventLogRecord.getThrown());

            //severity == 49:
            sequenceNumber++;

            commonBaseEvent = eventFactory.createCommonBaseEvent();
            commonBaseEvent.setCreationTimeAsLong(CURRENT_TIME);
            commonBaseEvent.setSeverity((short) 49);

            commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getLevel().getName(), Level.WARNING.getName());

            assertEquals(commonBaseEventLogRecord.getLevel().intValue(), Level.WARNING.intValue());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

            assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.WARNING));

            assertEquals(commonBaseEventLogRecord.getMillis(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            assertNull(commonBaseEventLogRecord.getThrown());

            //severity == 30:
            sequenceNumber++;

            commonBaseEvent = eventFactory.createCommonBaseEvent();
            commonBaseEvent.setCreationTimeAsLong(CURRENT_TIME);
            commonBaseEvent.setSeverity((short) 30);

            commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getLevel().getName(), Level.WARNING.getName());

            assertEquals(commonBaseEventLogRecord.getLevel().intValue(), Level.WARNING.intValue());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

            assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.WARNING));

            assertEquals(commonBaseEventLogRecord.getMillis(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            assertNull(commonBaseEventLogRecord.getThrown());

            //severity == 29:
            sequenceNumber++;

            commonBaseEvent = eventFactory.createCommonBaseEvent();
            commonBaseEvent.setCreationTimeAsLong(CURRENT_TIME);
            commonBaseEvent.setSeverity((short) 29);

            commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getLevel().getName(), Level.INFO.getName());

            assertEquals(commonBaseEventLogRecord.getLevel().intValue(), Level.INFO.intValue());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

            assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.INFO));

            assertEquals(commonBaseEventLogRecord.getMillis(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), sequenceNumber);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            assertNull(commonBaseEventLogRecord.getThrown());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test double argument constructor with the Common Base Event creationTime
     * set.
     */
    public void testDoubleArgumentConstructorCBESeverityCreationTime() {

        try {

            final long CURRENT_TIME = System.currentTimeMillis();

            //Spin for 1 second:
            while (System.currentTimeMillis() < (CURRENT_TIME + 1000)) {
                ;
            }

            LogRecord logRecord = new LogRecord(Level.OFF, null);

            long sequenceNumber = logRecord.getSequenceNumber();

            final int THREAD_ID = logRecord.getThreadID();

            CommonBaseEvent commonBaseEvent = null;
            CommonBaseEventLogRecord commonBaseEventLogRecord = null;
            Level[] levels = new Level[] { Level.ALL, Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.INFO, Level.OFF, Level.SEVERE, Level.WARNING, new CommonBaseEventLevel("TestLevel", 123456789)};

            for (int counter = 0; counter < levels.length; counter++) {

                sequenceNumber++;

                commonBaseEvent = eventFactory.createCommonBaseEvent();
                commonBaseEvent.setCreationTimeAsLong(CURRENT_TIME);

                commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent, levels[counter]);

                assertNull(commonBaseEventLogRecord.getMessage());

                assertTrue(commonBaseEventLogRecord.getCommonBaseEvent() == commonBaseEvent);

                assertEquals(commonBaseEventLogRecord.getLevel().getName(), levels[counter].getName());

                assertEquals(commonBaseEventLogRecord.getLevel().intValue(), levels[counter].intValue());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

                assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), levels[counter]));

                assertEquals(commonBaseEventLogRecord.getMillis(), CURRENT_TIME);

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

                assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

                assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), sequenceNumber);

                assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

                assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

                assertNull(commonBaseEvent.getMsgDataElement());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

                assertNull(commonBaseEventLogRecord.getLoggerName());

                assertNull(commonBaseEventLogRecord.getParameters());

                assertNull(commonBaseEventLogRecord.getResourceBundle());

                assertNull(commonBaseEventLogRecord.getResourceBundleName());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

                assertNull(commonBaseEventLogRecord.getSourceClassName());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

                assertNull(commonBaseEventLogRecord.getSourceMethodName());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

                assertNull(commonBaseEventLogRecord.getThrown());
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the level property.
     */
    public void testLevelAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = null;
            CommonBaseEventLogRecord commonBaseEventLogRecord = null;
            Level[] levels = new Level[] { Level.ALL, Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.INFO, Level.OFF, Level.SEVERE, Level.WARNING, new CommonBaseEventLevel("TestLevel", 123456789)};

            for (int counter = 0; counter < levels.length; counter++) {

                commonBaseEvent = eventFactory.createCommonBaseEvent();

                commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

                assertEquals(commonBaseEventLogRecord.getLevel(), Level.OFF);

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 0);

                assertFalse(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), Level.OFF));

                commonBaseEventLogRecord.setLevel(levels[counter]);

                assertEquals(commonBaseEventLogRecord.getLevel().getName(), levels[counter].getName());

                assertEquals(commonBaseEventLogRecord.getLevel().intValue(), levels[counter].intValue());

                assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

                assertTrue(containsLevelExtendedDataElement(commonBaseEventLogRecord.getCommonBaseEvent(), levels[counter]));
            }
            try {

                commonBaseEventLogRecord.setLevel(null);

                fail();

            } catch (NullPointerException n) {
                //Ignore since null levels are not permitted.
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the loggerName property.
     */
    public void testLoggerNameAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 0);

            final String LOGGER_NAME = "Logger Name";

            commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

            assertEquals(commonBaseEventLogRecord.getLoggerName(), LOGGER_NAME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValuesAsString(), LOGGER_NAME);

            commonBaseEventLogRecord.setLoggerName(null);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 1);

            assertNull(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValuesAsString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the message property.
     */
    public void testMessageAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertNull(commonBaseEvent.getMsgDataElement());

            final String MESSAGE = "Message";

            commonBaseEventLogRecord.setMessage(MESSAGE);

            assertEquals(commonBaseEventLogRecord.getMessage(), MESSAGE);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogId(), MESSAGE);

            commonBaseEventLogRecord.setMessage(null);

            assertNull(commonBaseEventLogRecord.getMessage());

            assertNull(commonBaseEvent.getMsgDataElement().getMsgCatalogId());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the millis property.
     */
    public void testMillisAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            final long START_TIME = System.currentTimeMillis();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            final long END_TIME = System.currentTimeMillis();

            assertTrue(commonBaseEventLogRecord.getMillis() >= START_TIME);

            assertTrue(commonBaseEventLogRecord.getMillis() <= END_TIME);

            assertTrue((END_TIME - START_TIME) <= 500);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), commonBaseEventLogRecord.getMillis());

            commonBaseEventLogRecord.setMillis(START_TIME);

            assertEquals(commonBaseEventLogRecord.getMillis(), START_TIME);

            assertEquals(commonBaseEvent.getCreationTimeAsLong(), START_TIME);

            commonBaseEventLogRecord.setMillis(0);

            assertEquals(commonBaseEventLogRecord.getMillis(), 0);

            assertEquals(commonBaseEvent.getCreationTimeAsLong(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the parameters property.
     */
    public void testParametersAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEvent.getMsgDataElement());

            //Different objects and different state:
            String[] parameters = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), parameters.length);

            assertEquals(Arrays.asList(commonBaseEvent.getMsgDataElement().getMsgCatalogTokensAsStrings()), Arrays.asList(parameters));

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), 0);

            //Different objects and same state:
            parameters = new String[] { "Parameter", "Parameter", "Parameter"};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), parameters.length);

            assertEquals(Arrays.asList(commonBaseEvent.getMsgDataElement().getMsgCatalogTokensAsStrings()), Arrays.asList(parameters));

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), 0);

            //Same objects and same state:
            String parameter = "Parameter";
            parameters = new String[] { parameter, parameter, parameter};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), parameters.length);

            assertEquals(Arrays.asList(commonBaseEvent.getMsgDataElement().getMsgCatalogTokensAsStrings()), Arrays.asList(parameters));

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogTokens().size(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the resourceBundleName property.
     */
    public void testResourceBundleNameAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertNull(commonBaseEvent.getMsgDataElement());

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            assertEquals(commonBaseEventLogRecord.getResourceBundleName(), RESOURCE_BUNDLE_NAME);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalog(), RESOURCE_BUNDLE_NAME);

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogType(), "Java");

            commonBaseEventLogRecord.setResourceBundleName(null);

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertNull(commonBaseEvent.getMsgDataElement().getMsgCatalog());

            assertEquals(commonBaseEvent.getMsgDataElement().getMsgCatalogType(), "Java");
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the resourceBundle property.
     */
    public void testResourceBundleAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEvent.getMsgDataElement());

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

            assertNull(commonBaseEvent.getMsgDataElement());

            assertEquals(commonBaseEventLogRecord.getResourceBundle(), RESOURCE_BUNDLE);

            commonBaseEventLogRecord.setResourceBundle(null);

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEvent.getMsgDataElement());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the sequenceNumber property.
     */
    public void testSequenceNumberAPIs() {

        try {

            final long SEQUENCE_NUMBER = (new LogRecord(Level.OFF, null).getSequenceNumber() + 1);

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), SEQUENCE_NUMBER);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), SEQUENCE_NUMBER);

            commonBaseEventLogRecord.setSequenceNumber(Long.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), Long.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), Long.MAX_VALUE);

            commonBaseEventLogRecord.setSequenceNumber(0);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), 0);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the sourceClassName property.
     */
    public void testSourceClassNameAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 0);

            final String SOURCE_CLASS_NAME = "Source Class Name";

            commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

            assertEquals(commonBaseEventLogRecord.getSourceClassName(), SOURCE_CLASS_NAME);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).get(0))).getValuesAsString(), SOURCE_CLASS_NAME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 1);

            commonBaseEventLogRecord.setSourceClassName(null);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 1);

            assertNull(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).get(0))).getValuesAsString());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the sourceMethodName property.
     */
    public void testSourceMethodNameAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 0);

            final String SOURCE_METHOD_NAME = "Source Method Name";

            commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            assertEquals(commonBaseEventLogRecord.getSourceMethodName(), SOURCE_METHOD_NAME);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).get(0))).getValuesAsString(), SOURCE_METHOD_NAME);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 1);

            commonBaseEventLogRecord.setSourceMethodName(null);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertNull(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).get(0))).getValuesAsString());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 1);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the threadID property.
     */
    public void testThreadIDAPIs() {

        try {

            final int THREAD_ID = new LogRecord(Level.OFF, null).getThreadID();

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);

            commonBaseEventLogRecord.setThreadID(Integer.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getThreadID(), Integer.MAX_VALUE);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), Integer.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            commonBaseEventLogRecord.setThreadID(0);

            assertEquals(commonBaseEventLogRecord.getThreadID(), 0);

            assertEquals(((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), 0);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test getter and setter for the thrown property.
     */
    public void testThrownAPIs() {

        try {

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            assertNull(commonBaseEventLogRecord.getThrown());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 0);

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowableMessage" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowableMessage" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("ThrowableMessage", child);

            commonBaseEventLogRecord.setThrown(throwable);

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 1);

            assertEquals(commonBaseEventLogRecord.getThrown(), throwable);

            ExtendedDataElement extendedDataElement = ((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).get(0)));

            assertEquals(extendedDataElement.getName(), CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN);

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            StackTraceElement stackTraceElements[] = throwable.getStackTrace();

            assertEquals(extendedDataElement.getValues().size(), (stackTraceElements.length + 1));

            assertEquals(((String) (extendedDataElement.getValues().get(0))), throwable.toString());

            for (int counter = 0; counter < stackTraceElements.length; counter++) {
                assertEquals(((String) (extendedDataElement.getValues().get(counter + 1))), stackTraceElements[counter].toString());
            }

            assertEquals(extendedDataElement.getChildren().size(), 1);

            Throwable causeThrowable = throwable;
            ExtendedDataElement causeExtendedDataElement = extendedDataElement;

            for (int counter = 0; counter < THROWABLES_LEVELS; counter++) {

                causeThrowable = causeThrowable.getCause();

                assertNotNull(causeThrowable);

                assertEquals(causeExtendedDataElement.getChildren().size(), 1);

                causeExtendedDataElement = ((ExtendedDataElement) (causeExtendedDataElement.getChildren().get(0)));

                assertEquals(causeExtendedDataElement.getName(), "Cause");

                assertEquals(causeExtendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

                stackTraceElements = causeThrowable.getStackTrace();

                assertEquals(causeExtendedDataElement.getValues().size(), (stackTraceElements.length + 1));

                assertEquals(((String) (causeExtendedDataElement.getValues().get(0))), causeThrowable.toString());

                for (int index = 0; index < stackTraceElements.length; index++) {
                    assertEquals(((String) (causeExtendedDataElement.getValues().get(index + 1))), stackTraceElements[index].toString());
                }
            }

            assertNull(causeThrowable.getCause());

            assertEquals(causeExtendedDataElement.getChildren().size(), 0);

            //Test a 'null' Throwable:
            commonBaseEventLogRecord.setThrown(null);

            assertNull(commonBaseEventLogRecord.getThrown());

            assertEquals(commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).size(), 1);

            extendedDataElement = ((ExtendedDataElement) (commonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN).get(0)));

            assertEquals(extendedDataElement.getName(), CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THROWN);

            assertEquals(extendedDataElement.getTypeAsInt(), ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            assertEquals(extendedDataElement.getValues().size(), 1);

            assertEquals(((String) (extendedDataElement.getValues().get(0))), null);

            assertEquals(extendedDataElement.getChildren().size(), 0);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test all the APIs in the super class (e.g. LogRecord) if a Common Base
     * Event is not used at construction (no Common Base Event capabilities)
     * with the single argument constructor.
     */
    public void testLogRecordAPIsSingleArgumentConstructor() {

        try {

            LogRecord logRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (logRecord.getSequenceNumber() + 1);

            final int THREAD_ID = logRecord.getThreadID();

            final long START_TIME = System.currentTimeMillis();

            CommonBaseEvent commonBaseEvent = null;

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            final long END_TIME = System.currentTimeMillis();

            assertNull(commonBaseEventLogRecord.getMessage());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            assertEquals(commonBaseEventLogRecord.getLevel(), Level.OFF);

            assertTrue(commonBaseEventLogRecord.getMillis() >= START_TIME);

            assertTrue(commonBaseEventLogRecord.getMillis() <= END_TIME);

            assertTrue((END_TIME - START_TIME) <= 1000);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), SEQUENCE_NUMBER);

            assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            assertNull(commonBaseEventLogRecord.getThrown());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            Level[] levels = new Level[] { Level.ALL, Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.INFO, Level.OFF, Level.SEVERE, Level.WARNING, new CommonBaseEventLevel("TestLevel", 123456789)};

            for (int counter = 0; counter < levels.length; counter++) {

                commonBaseEventLogRecord.setLevel(levels[counter]);

                assertEquals(commonBaseEventLogRecord.getLevel().getName(), levels[counter].getName());

                assertEquals(commonBaseEventLogRecord.getLevel().intValue(), levels[counter].intValue());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());
            }

            try {

                commonBaseEventLogRecord.setLevel(null);

                fail();

            } catch (NullPointerException n) {
                //Ignore since null levels are not permitted.
            }

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final String LOGGER_NAME = "Logger Name";

            commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

            assertEquals(commonBaseEventLogRecord.getLoggerName(), LOGGER_NAME);

            commonBaseEventLogRecord.setLoggerName(null);

            assertNull(commonBaseEventLogRecord.getLoggerName());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final String MESSAGE = "Message";

            commonBaseEventLogRecord.setMessage(MESSAGE);

            assertEquals(commonBaseEventLogRecord.getMessage(), MESSAGE);

            commonBaseEventLogRecord.setMessage(null);

            assertNull(commonBaseEventLogRecord.getMessage());

            commonBaseEventLogRecord.setMillis(START_TIME);

            assertEquals(commonBaseEventLogRecord.getMillis(), START_TIME);

            commonBaseEventLogRecord.setMillis(0);

            assertEquals(commonBaseEventLogRecord.getMillis(), 0);

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            //Different objects and different state:
            String[] parameters = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            //Different objects and same state:
            parameters = new String[] { "Parameter", "Parameter", "Parameter"};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            //Same objects and same state:
            String parameter = "Parameter";
            parameters = new String[] { parameter, parameter, parameter};

            commonBaseEventLogRecord.setParameters(parameters);

            assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

            commonBaseEventLogRecord.setParameters(null);

            assertNull(commonBaseEventLogRecord.getParameters());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

            assertEquals(commonBaseEventLogRecord.getResourceBundle(), RESOURCE_BUNDLE);

            commonBaseEventLogRecord.setResourceBundle(null);

            assertNull(commonBaseEventLogRecord.getResourceBundle());

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            assertEquals(commonBaseEventLogRecord.getResourceBundleName(), RESOURCE_BUNDLE_NAME);

            commonBaseEventLogRecord.setResourceBundleName(null);

            assertNull(commonBaseEventLogRecord.getResourceBundleName());

            commonBaseEventLogRecord.setSequenceNumber(Long.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), Long.MAX_VALUE);

            commonBaseEventLogRecord.setSequenceNumber(0);

            assertEquals(commonBaseEventLogRecord.getSequenceNumber(), 0);

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final String SOURCE_CLASS_NAME = "Source Class Name";

            commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

            assertEquals(commonBaseEventLogRecord.getSourceClassName(), SOURCE_CLASS_NAME);

            commonBaseEventLogRecord.setSourceClassName(null);

            assertNull(commonBaseEventLogRecord.getSourceClassName());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final String SOURCE_METHOD_NAME = "Source Method Name";

            commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            assertEquals(commonBaseEventLogRecord.getSourceMethodName(), SOURCE_METHOD_NAME);

            commonBaseEventLogRecord.setSourceMethodName(null);

            assertNull(commonBaseEventLogRecord.getSourceMethodName());

            commonBaseEventLogRecord.setThreadID(Integer.MAX_VALUE);

            assertEquals(commonBaseEventLogRecord.getThreadID(), Integer.MAX_VALUE);

            commonBaseEventLogRecord.setThreadID(0);

            assertEquals(commonBaseEventLogRecord.getThreadID(), 0);

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            commonBaseEventLogRecord.setThrown(throwable);

            assertEquals(commonBaseEventLogRecord.getThrown(), throwable);

            commonBaseEventLogRecord.setThrown(null);

            assertNull(commonBaseEventLogRecord.getThrown());

            assertNull(commonBaseEventLogRecord.getCommonBaseEvent());
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Test all the APIs in the super class (e.g. LogRecord) if a Common Base
     * Event is not used at construction (no Common Base Event capabilities)
     * with the double argument constructor.
     */
    public void testLogRecordAPIsDoubleArgumentConstructor() {

        try {

            LogRecord logRecord = new LogRecord(Level.OFF, null);

            long sequenceNumber = logRecord.getSequenceNumber();

            final int THREAD_ID = logRecord.getThreadID();

            CommonBaseEvent commonBaseEvent = null;
            CommonBaseEventLogRecord commonBaseEventLogRecord = null;
            Level[] levels = new Level[] { Level.ALL, Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.INFO, Level.OFF, Level.SEVERE, Level.WARNING, new CommonBaseEventLevel("TestLevel", 123456789)};

            for (int counter = 0; counter < levels.length; counter++) {

                sequenceNumber++;

                long startTime = System.currentTimeMillis();

                commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent, levels[counter]);

                long endTime = System.currentTimeMillis();

                assertNull(commonBaseEventLogRecord.getMessage());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                assertEquals(commonBaseEventLogRecord.getLevel().getName(), levels[counter].getName());

                assertEquals(commonBaseEventLogRecord.getLevel().intValue(), levels[counter].intValue());

                assertTrue(commonBaseEventLogRecord.getMillis() >= startTime);

                assertTrue(commonBaseEventLogRecord.getMillis() <= endTime);

                assertTrue((endTime - startTime) <= 1000);

                assertEquals(commonBaseEventLogRecord.getSequenceNumber(), sequenceNumber);

                assertEquals(commonBaseEventLogRecord.getThreadID(), THREAD_ID);

                assertNull(commonBaseEventLogRecord.getLoggerName());

                assertNull(commonBaseEventLogRecord.getParameters());

                assertNull(commonBaseEventLogRecord.getResourceBundle());

                assertNull(commonBaseEventLogRecord.getResourceBundleName());

                assertNull(commonBaseEventLogRecord.getSourceClassName());

                assertNull(commonBaseEventLogRecord.getSourceMethodName());

                assertNull(commonBaseEventLogRecord.getThrown());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                for (int index = 0; index < levels.length; index++) {

                    commonBaseEventLogRecord.setLevel(levels[index]);

                    assertEquals(commonBaseEventLogRecord.getLevel().getName(), levels[index].getName());

                    assertEquals(commonBaseEventLogRecord.getLevel().intValue(), levels[index].intValue());

                    assertNull(commonBaseEventLogRecord.getCommonBaseEvent());
                }

                try {

                    commonBaseEventLogRecord.setLevel(null);

                    fail();

                } catch (NullPointerException n) {
                    //Ignore since null levels are not permitted.
                }

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final String LOGGER_NAME = "Logger Name";

                commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

                assertEquals(commonBaseEventLogRecord.getLoggerName(), LOGGER_NAME);

                commonBaseEventLogRecord.setLoggerName(null);

                assertNull(commonBaseEventLogRecord.getLoggerName());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final String MESSAGE = "Message";

                commonBaseEventLogRecord.setMessage(MESSAGE);

                assertEquals(commonBaseEventLogRecord.getMessage(), MESSAGE);

                commonBaseEventLogRecord.setMessage(null);

                assertNull(commonBaseEventLogRecord.getMessage());

                commonBaseEventLogRecord.setMillis(startTime);

                assertEquals(commonBaseEventLogRecord.getMillis(), startTime);

                commonBaseEventLogRecord.setMillis(0);

                assertEquals(commonBaseEventLogRecord.getMillis(), 0);

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                //Different objects and different state:
                String[] parameters = new String[] { "Paramater1", "Paramater2", "Paramater3"};

                commonBaseEventLogRecord.setParameters(parameters);

                assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

                commonBaseEventLogRecord.setParameters(null);

                assertNull(commonBaseEventLogRecord.getParameters());

                //Different objects and same state:
                parameters = new String[] { "Parameter", "Parameter", "Parameter"};

                commonBaseEventLogRecord.setParameters(parameters);

                assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

                commonBaseEventLogRecord.setParameters(null);

                assertNull(commonBaseEventLogRecord.getParameters());

                //Same objects and same state:
                String parameter = "Parameter";
                parameters = new String[] { parameter, parameter, parameter};

                commonBaseEventLogRecord.setParameters(parameters);

                assertEquals(commonBaseEventLogRecord.getParameters(), parameters);

                commonBaseEventLogRecord.setParameters(null);

                assertNull(commonBaseEventLogRecord.getParameters());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                    public Enumeration getKeys() {
                        return null;
                    }

                    public Locale getLocale() {
                        return null;
                    }

                    protected Object handleGetObject(String key) {
                        return null;
                    }

                    protected void setParent(ResourceBundle parent) {
                    }
                };

                commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

                assertEquals(commonBaseEventLogRecord.getResourceBundle(), RESOURCE_BUNDLE);

                commonBaseEventLogRecord.setResourceBundle(null);

                assertNull(commonBaseEventLogRecord.getResourceBundle());

                final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

                commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

                assertEquals(commonBaseEventLogRecord.getResourceBundleName(), RESOURCE_BUNDLE_NAME);

                commonBaseEventLogRecord.setResourceBundleName(null);

                assertNull(commonBaseEventLogRecord.getResourceBundleName());

                commonBaseEventLogRecord.setSequenceNumber(Long.MAX_VALUE);

                assertEquals(commonBaseEventLogRecord.getSequenceNumber(), Long.MAX_VALUE);

                commonBaseEventLogRecord.setSequenceNumber(0);

                assertEquals(commonBaseEventLogRecord.getSequenceNumber(), 0);

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final String SOURCE_CLASS_NAME = "Source Class Name";

                commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

                assertEquals(commonBaseEventLogRecord.getSourceClassName(), SOURCE_CLASS_NAME);

                commonBaseEventLogRecord.setSourceClassName(null);

                assertNull(commonBaseEventLogRecord.getSourceClassName());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final String SOURCE_METHOD_NAME = "Source Method Name";

                commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

                assertEquals(commonBaseEventLogRecord.getSourceMethodName(), SOURCE_METHOD_NAME);

                commonBaseEventLogRecord.setSourceMethodName(null);

                assertNull(commonBaseEventLogRecord.getSourceMethodName());

                commonBaseEventLogRecord.setThreadID(Integer.MAX_VALUE);

                assertEquals(commonBaseEventLogRecord.getThreadID(), Integer.MAX_VALUE);

                commonBaseEventLogRecord.setThreadID(0);

                assertEquals(commonBaseEventLogRecord.getThreadID(), 0);

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());

                final int THROWABLES_LEVELS = 100;

                Throwable throwable = null;
                Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

                for (int index = (THROWABLES_LEVELS - 1); index > 0; index--) {

                    throwable = new Throwable("ChildThrowable" + index, child);
                    child = throwable;
                }

                throwable = new Throwable("Throwable", child);

                commonBaseEventLogRecord.setThrown(throwable);

                assertEquals(commonBaseEventLogRecord.getThrown(), throwable);

                commonBaseEventLogRecord.setThrown(null);

                assertNull(commonBaseEventLogRecord.getThrown());

                assertNull(commonBaseEventLogRecord.getCommonBaseEvent());
            }
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    /**
     * Tests pre-populated fields of the parameter Common Base Event.
     */
    public void testCommonBaseEventPopulatedFields() {

        try {
            LogRecord logRecord = new LogRecord(Level.OFF, null);

            final long SEQUENCE_NUMBER = (logRecord.getSequenceNumber() + 1);

            final int THREAD_ID = logRecord.getThreadID();

            final long CURRENT_TIME = System.currentTimeMillis();

            final String LOGGER_NAME = "Logger Name";

            final String MESSAGE = "Message";

            final String[] PARAMETERS = new String[] { "Paramater1", "Paramater2", "Paramater3"};

            final ResourceBundle RESOURCE_BUNDLE = new ResourceBundle() {

                public Enumeration getKeys() {
                    return null;
                }

                public Locale getLocale() {
                    return null;
                }

                protected Object handleGetObject(String key) {
                    return null;
                }

                protected void setParent(ResourceBundle parent) {
                }
            };

            final String RESOURCE_BUNDLE_NAME = "ResourceBundleName";

            final String SOURCE_CLASS_NAME = "Source Class Name";

            final String SOURCE_METHOD_NAME = "Source Method Name";

            final int THROWABLES_LEVELS = 100;

            Throwable throwable = null;
            Throwable child = new Throwable("ChildThrowable" + THROWABLES_LEVELS);

            for (int counter = (THROWABLES_LEVELS - 1); counter > 0; counter--) {

                throwable = new Throwable("ChildThrowable" + counter, child);
                child = throwable;
            }

            throwable = new Throwable("Throwable", child);

            CommonBaseEvent commonBaseEvent = eventFactory.createCommonBaseEvent();

            CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(commonBaseEvent);

            commonBaseEventLogRecord.setMillis(CURRENT_TIME);

            commonBaseEventLogRecord.setLevel(Level.ALL);

            commonBaseEventLogRecord.setLoggerName(LOGGER_NAME);

            commonBaseEventLogRecord.setMessage(MESSAGE);

            commonBaseEventLogRecord.setParameters(PARAMETERS);

            commonBaseEventLogRecord.setResourceBundle(RESOURCE_BUNDLE);

            commonBaseEventLogRecord.setResourceBundleName(RESOURCE_BUNDLE_NAME);

            commonBaseEventLogRecord.setSourceClassName(SOURCE_CLASS_NAME);

            commonBaseEventLogRecord.setSourceMethodName(SOURCE_METHOD_NAME);

            commonBaseEventLogRecord.setThrown(throwable);

            String commonBaseEventXML = EventFormatter.toCanonicalXMLString(commonBaseEventLogRecord.getCommonBaseEvent(), true);

            CommonBaseEvent serialziedCommonBaseEvent = EventFormatter.eventFromCanonicalXML(commonBaseEventXML);

            //Equal based on state only:
            assertEquals(commonBaseEvent, serialziedCommonBaseEvent);

            assertNotSame(commonBaseEvent, serialziedCommonBaseEvent);

            CommonBaseEventLogRecord serialziedCommonBaseEventLogRecord = new CommonBaseEventLogRecord(serialziedCommonBaseEvent);

            assertNotSame(commonBaseEventLogRecord, serialziedCommonBaseEventLogRecord);

            assertEquals(serialziedCommonBaseEventLogRecord.getThreadID(), THREAD_ID);

            assertEquals(serialziedCommonBaseEventLogRecord.getSequenceNumber(), SEQUENCE_NUMBER);

            assertEquals(serialziedCommonBaseEventLogRecord.getMillis(), CURRENT_TIME);

            assertEquals(commonBaseEventLogRecord.getLevel().getName(), Level.ALL.getName());

            assertEquals(commonBaseEventLogRecord.getLevel().intValue(), Level.ALL.intValue());

            assertEquals(serialziedCommonBaseEventLogRecord.getLoggerName(), LOGGER_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getMessage(), MESSAGE);

            assertEquals(Arrays.asList(serialziedCommonBaseEventLogRecord.getParameters()), Arrays.asList(PARAMETERS));

            assertNull(serialziedCommonBaseEventLogRecord.getResourceBundle());

            assertEquals(serialziedCommonBaseEventLogRecord.getResourceBundleName(), RESOURCE_BUNDLE_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getSourceClassName(), SOURCE_CLASS_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getSourceMethodName(), SOURCE_METHOD_NAME);

            assertNull(serialziedCommonBaseEventLogRecord.getThrown());

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL).size(), 1);

            assertTrue(containsLevelExtendedDataElement(serialziedCommonBaseEventLogRecord.getCommonBaseEvent(), Level.ALL));

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).size(), 1);

            assertEquals(((ExtendedDataElement) (serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LOGGER_NAME).get(0))).getValuesAsString(), LOGGER_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getMsgDataElement().getMsgCatalogId(), MESSAGE);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getCreationTimeAsLong(), CURRENT_TIME);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getMsgDataElement().getMsgCatalogTokens().size(), PARAMETERS.length);

            assertEquals(Arrays.asList(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getMsgDataElement().getMsgCatalogTokensAsStrings()), Arrays.asList(PARAMETERS));

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getMsgDataElement().getMsgCatalog(), RESOURCE_BUNDLE_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getMsgDataElement().getMsgCatalogType(), "Java");

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).size(), 1);

            assertEquals(((ExtendedDataElement) (serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SEQUENCE_NUMBER).get(0))).getValuesAsLong(), SEQUENCE_NUMBER);

            assertEquals(((ExtendedDataElement) (serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).get(0))).getValuesAsString(), SOURCE_CLASS_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_CLASS_NAME).size(), 1);

            assertEquals(((ExtendedDataElement) (serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).get(0))).getValuesAsString(), SOURCE_METHOD_NAME);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_SOURCE_METHOD_NAME).size(), 1);

            assertEquals(serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).size(), 1);

            assertEquals(((ExtendedDataElement) (serialziedCommonBaseEventLogRecord.getCommonBaseEvent().getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_THREAD_ID).get(0))).getValuesAsInt(), THREAD_ID);
        } catch (AssertionFailedError a) {
            throw (a);
        } catch (Throwable t) {
            assertTrue(t.toString(), false);
        }
    }

    private boolean containsLevelExtendedDataElement(CommonBaseEvent commonBaseEvent, Level level) {

        if ((commonBaseEvent != null) && (level != null)) {

            try {

                List levelExtendedDataElements = commonBaseEvent.getExtendedDataElements(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL);

                if ((levelExtendedDataElements.size() == 1) && (((ExtendedDataElement) (levelExtendedDataElements.get(0))).getTypeAsInt() == ExtendedDataElement.TYPE_NO_VALUE_VALUE)) {

                    List levelNameExtendedDataElements = ((ExtendedDataElement) (levelExtendedDataElements.get(0))).getChildren(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_NAME);
                    List levelValueExtendedDataElements = ((ExtendedDataElement) (levelExtendedDataElements.get(0))).getChildren(CommonBaseEventLogRecord.EXTENDED_DATA_ELEMENT_NAME_LEVEL_VALUE);

                    if ((levelNameExtendedDataElements.size() == 1) && (levelValueExtendedDataElements.size() == 1)) {

                        String levelName = ((ExtendedDataElement) (levelNameExtendedDataElements.get(0))).getValuesAsString();

                        int levelValue = ((ExtendedDataElement) (levelValueExtendedDataElements.get(0))).getValuesAsInt();

                        return ((level.getName().equals(levelName)) && (level.intValue() == levelValue));
                    }
                }
            } catch (Throwable t) {
                //Ignore since API returns false.
            }
        }

        return false;
    }

    private String concatenateListElements(List list) {

        StringBuffer buffer = new StringBuffer();
        ListIterator listIterator = list.listIterator();

        while (listIterator.hasNext()) {
            buffer.append(String.valueOf(listIterator.next()));
        }

        if (buffer.toString().equals("null")) { return null; }

        return (buffer.toString());
    }

    private boolean areListElementsValidSize(List list, int maxSize) {

        ListIterator listIterator = list.listIterator();

        while (listIterator.hasNext()) {

            if (String.valueOf(listIterator.next()).length() > maxSize) { return false; }
        }

        return true;
    }
}