/**********************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.models.cbe.util;


import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.cbe.CBEPackage;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LeftOperand;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByElement;
import org.eclipse.hyades.models.hierarchy.extensions.OrderByOperators;
import org.eclipse.hyades.models.hierarchy.extensions.Query;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleBinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.extensions.TimeBasedCorrelationQuery;
import org.eclipse.hyades.models.hierarchy.util.IExtendedQueryService;
import org.eclipse.hyades.models.hierarchy.util.IFilterElement;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.hierarchy.util.ILogFilterCriteria;
import org.eclipse.hyades.models.hierarchy.util.ISortElement;

/**
 * @author apnan
 *
 */
public class LogQueryBuilder {

	public static Query createQuery(List inputList, ILogFilterCriteria element, EAttribute attribute){
		SimpleSearchQuery query =  addAgents(inputList, element);
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setType(HierarchyPackage.eINSTANCE.getTRCAgent());
		query.getOutputElements().add(leftOperand);		
		leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setFeature(attribute); 
		query.getOutputElements().add(leftOperand);

		return query;
		
	}
	
	/**
	 * Creates a model query based on a list of input EObjects and a filter criteria.
	 * Once a query is created it can be executed using {@link #executeQuery(Query, ResourceSet)}
	 * @param inputList list of EObjects
	 * @param element filter criteria
	 * @return model query
	 */
	public static Query createQuery(List inputList, ILogFilterCriteria element){
		
		Query query = null;
		EObject inputObject = null;
		int s = inputList.size();
		for(int i=0;i<s;i++){
			inputObject = (EObject)inputList.get(i);
			if(inputObject instanceof TRCAgentProxy)
			{
				if(query==null){
					query = createQueryFromAgentProxy((TRCAgentProxy)inputObject, element);
				}
				else{ 
					if(i==0){				
						addAgentToQuery((SimpleSearchQuery)query, (TRCAgentProxy)inputObject, element);
					}else{
						addAgentToQuery((SimpleSearchQuery)query, (TRCAgentProxy)inputObject, null);
					}
				}
				
			}else if(inputObject instanceof CorrelationContainerProxy){
				
				query = createQueryFromAgents(((CorrelationContainerProxy)inputObject).getCorrelatedAgents(), element);
				
			}
		}
		if(query!=null)	
			query.setDistinct(true);
		return query;
	}
	
	/**
	 * Executes a model query. Delegates the execution call to the first HyadesResourceExtension found.
	 * @see IExtendedQueryService#executeQuery 
	 * 
	 * The returned QueryResult object will then be used to read the actual list of common base events.
	 * For every output element a resultEntry will be returned. For now only Common Base Events are used as output elements.  	
	 * 
	 * QueryResult queryResult = LogQueryBuilder.executeQuery(query, agentProxy.eResource().getResourceSet());
	 * ResultEntry resultEntry =(ResultEntry)queryResult.getResultEntries().get(0); 
	 * List myCBEs = resultEntry.getObjects();
	 *
	 * @param query
	 * @param resourceSet
	 * @return QueryResult
	 */
	
	public static QueryResult executeQuery(Query query, ResourceSet resourceSet){
		
		if(query==null || query.getSources().size()==0)
			return null;
		
		QueryResult queryResult = null;
		String url = (String)query.getSources().get(0);
		IHyadesResourceExtension hyadesResourceExtension = (IHyadesResourceExtension)HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(url));
		if (hyadesResourceExtension != null)
		{
			List notLoadedTypes=new ArrayList();
	    	notLoadedTypes.add(HierarchyPackage.eINSTANCE.getAbstractDefaultEvent());
	    	notLoadedTypes.add(HierarchyPackage.eINSTANCE.getTRCAgentProxy());

			queryResult = hyadesResourceExtension.executeQuery(query, resourceSet,notLoadedTypes); 
			return queryResult;
		}
		return queryResult;

	}

	/**
	 * This method should be used when creating a query for searching records in the log views based on a search criteria.
	 * The method will append to the existing log filter criteria the search criteria, returning a new query.	 
	 * @param inputList list of EObjects
	 * @param element filter criteria
	 * @return search query
	 */

	public static Query createSearchQuery(List inputList, ILogFilterCriteria filterElement, ILogFilterCriteria searchElement){
		
		Query query = createQuery(inputList, filterElement);
		if(query==null){
			query = createQuery(inputList, searchElement);
		}else{
			query = getMergedQuery(query, searchElement);
		}
		return query;
	}
	
	/**
	 * This method should be used when a filter needs to be appended to an existing query. 
	 * An example would be when creating a query for searching records in the log view based on a search criteria.
	 * The method will append the search criteria to the existing log query.	 
	 * 
	 * @param query existing filter query
	 * @return merged query
	 */

	public static Query getMergedQuery(Query query, ILogFilterCriteria mergeElement){
		
		SimpleSearchQuery searchQuery = (SimpleSearchQuery)EcoreUtil.copy(query);
		List whereCriteria = searchQuery.getWhereExpressions();
		IFilterElement[] searchFilters = mergeElement.getFilters();
		List attributes = CBEPackage.eINSTANCE.getCBECommonBaseEvent().getEAllAttributes();
		EAttribute attribute = null;
		for(int i=0;i<searchFilters.length;i++){
			attribute = findAttributeInList(searchFilters[i].getAttribute(), attributes);
			if(attribute!=null && !isFilterInWhereClause(searchFilters[i], whereCriteria)){
				whereCriteria.add(createBinaryExpression(searchFilters[i], attribute));
			}
		}

		return searchQuery;
	}
	
	public static Query createCorrelation(List agentProxies){
		TimeBasedCorrelationQuery query = ExtensionsFactory.eINSTANCE.createTimeBasedCorrelationQuery();
		int s = agentProxies.size();
		EObject inputObject = null;
		double delta = 0;
		for(int i=0;i<s;i++){
			inputObject = (EObject)agentProxies.get(i);
			if(inputObject instanceof TRCAgentProxy)
			{
				addAgentToQuery((SimpleSearchQuery)query, (TRCAgentProxy)inputObject, null);
				
				if(((TRCAgentProxy)inputObject).getProcessProxy()!=null && ((TRCAgentProxy)inputObject).getProcessProxy().getNode()!=null)
					delta = ((TRCAgentProxy)inputObject).getProcessProxy().getNode().getDeltaTime();
				else
					delta = 0;
				query.getDeltaTime().add(new Double(delta));
			}
		}
	
		return query;
		
	}
	
	public static Query createCorrelatedEventsQuery(List agentProxies){
		
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery)createCorrelation(agentProxies);
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		query.getOutputElements().add(leftOperand);
		return query;
		
	}
	
	public static Query createOutboundCorrelationQuery(List agentProxies, ILogFilterCriteria filter){
		
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery)createCorrelation(agentProxies);
		
		LeftOperand leftOperand1 = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand1.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Key());
		query.getOutputElements().add(leftOperand1);
		
		LeftOperand leftOperand2 = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand2.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Value());
		query.getOutputElements().add(leftOperand2);		
		return getMergedQuery(query, filter);

	}
	
	public static Query createInboundCorrelationQuery(List agentProxies, ILogFilterCriteria filter){
		
		TimeBasedCorrelationQuery query = (TimeBasedCorrelationQuery)createCorrelation(agentProxies);
		
		LeftOperand leftOperand1 = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand1.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Value());
		query.getOutputElements().add(leftOperand1);
		
		LeftOperand leftOperand2 = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand2.setFeature(HierarchyPackage.eINSTANCE.getCorrelationEntry_Key());
		query.getOutputElements().add(leftOperand2);		
		return getMergedQuery(query, filter);
		
	}
		
	private static void addAgentToQuery(SimpleSearchQuery query, TRCAgentProxy inputObject, ILogFilterCriteria element){
		
		TRCAgent agent = (TRCAgent)inputObject.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
		if(agent!=null){
			query.getSources().add(EcoreUtil.getURI(agent).toString());		
			if(element!=null){
				query.getWhereExpressions().addAll(createWhereClause(CBEPackage.eINSTANCE.getCBECommonBaseEvent(), element.getFilters()));
				addSeverityFilters(query, element);
				query.getOrderByExpresions().addAll(createOrderByClause(CBEPackage.eINSTANCE.getCBECommonBaseEvent(),element.getSortColumns()));
			}
		}
		
	}

	private static void addSeverityFilters(SimpleSearchQuery query, ILogFilterCriteria element) {
		
		Map options = element.getFilterOptions();
		if(options!=null){
			if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") && options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")){
				return;
			}else{
				if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && 
				   options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("0") &&
				   options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")){					
					query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(49)));
					query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(71)));					
				}else
					
					if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("1") && 
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") &&
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")){					
						query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
						query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(71)));					
				}else
				if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") &&
					options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") &&
					options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")){
					query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
					query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));					
				}else
					if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") &&
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") &&
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("0")){
							query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.GT), new Integer(29)));
							query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));
				}else 
					if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") &&
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("1") &&
					   options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")){
							query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(50)));
				}					
                else					
				if(options.get(ILogFilterCriteria.OPTION_FILTER_SEV1).equals("0") &&
				    options.get(ILogFilterCriteria.OPTION_FILTER_SEV2).equals("0") &&	
					options.get(ILogFilterCriteria.OPTION_FILTER_SEV3).equals("1")){
					query.getWhereExpressions().add(createBinaryExpression(CBEPackage.eINSTANCE.getCBECommonBaseEvent_Severity(), RelationalOperators.get(RelationalOperators.LT), new Integer(30)));					
				}
				
			}
		}
	}

	private static Query createQueryFromAgents(List inputList, ILogFilterCriteria element){
		SimpleSearchQuery query =  addAgents(inputList, element);
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent()); 
		query.getOutputElements().add(leftOperand);		

		return query;
	}
	
	private static SimpleSearchQuery addAgents(List inputList, ILogFilterCriteria element) {
		int s = inputList.size();
		TRCAgentProxy agentProxy = null;
		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		for(int i=0;i<s;i++){
			agentProxy = (TRCAgentProxy)inputList.get(i);
			addAgentToQuery(query, agentProxy, element);
		}
		return query;
	}

	private static Query createQueryFromAgentProxy(TRCAgentProxy inputObject, ILogFilterCriteria element){

		SimpleSearchQuery query = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setType(CBEPackage.eINSTANCE.getCBECommonBaseEvent());
		query.getOutputElements().add(leftOperand);
		addAgentToQuery(query, inputObject, element);
		return query;
	}
	
		
	private static List createWhereClause(EClass eClass, IFilterElement[] elements){
		List whereCriteria = new ArrayList();
		List attributes = eClass.getEAllAttributes();
		EAttribute attribute = null;
		for(int i=0;i<elements.length;i++){
			attribute = findAttributeInList(elements[i].getAttribute(), attributes);
			if(attribute!=null){
				whereCriteria.add(createBinaryExpression(elements[i], attribute));
			}
		}
		return whereCriteria;
	}
	
	private static SimpleBinaryExpression createBinaryExpression(IFilterElement element, EAttribute attribute){
		
		SimpleBinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createSimpleBinaryExpression();
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setFeature(attribute);
		binaryExpression.setOperand(leftOperand);
		String value = element.getValue();
		if(element.getOperator().equals("=")) {
			if(value.indexOf('*')>-1){
				value = value.replace('*','%');
				binaryExpression.setOperator(RelationalOperators.LIKE_LITERAL);
			}else{
				binaryExpression.setOperator(RelationalOperators.EQ_LITERAL);
			}
		}else if(element.getOperator().equals("<")) {
			binaryExpression.setOperator(RelationalOperators.LT_LITERAL);
		}else if(element.getOperator().equals(">")) {
			binaryExpression.setOperator(RelationalOperators.GT_LITERAL);
		}else if(element.getOperator().equals("LIKE")){
			binaryExpression.setOperator(RelationalOperators.LIKE_LITERAL);
		}
		binaryExpression.setRightOperand(convertToType(attribute.getEType(),value));
		return binaryExpression;
		
	}
	
	/**
	 * @param attribute
	 * @param value
	 * @return
	 */
	private static Object convertToType(EClassifier attribute, String value) {
		if(attribute instanceof EDataType)
		{			
			return EcoreFactory.eINSTANCE.createFromString((EDataType)attribute, value);
//			if(object instanceof String){
//				formatFilterValue((String)object);
//			}
//			return object;
		}
		return value;
	}

	/**
	 * Formats strings in order to be used in SQL statements.
	 * @param string
	 */
	private static void formatFilterValue(String string) {
		StringBuffer sb = new StringBuffer();
		int l = string.length();
		for(int i=0;i<l;i++){
			char ch = string.charAt(i);
			if(ch=='%'){
				sb.append("\\%");				
			}else if(ch=='_'){
				sb.append("\\_");				
			}else{
				sb.append(ch);
			}
		}
		string = sb.toString();
	}

	private static SimpleBinaryExpression createBinaryExpression(EAttribute attribute, RelationalOperators operator, Object value){
		
		SimpleBinaryExpression binaryExpression = ExtensionsFactory.eINSTANCE.createSimpleBinaryExpression();
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setFeature(attribute);
		binaryExpression.setOperand(leftOperand);
		binaryExpression.setOperator(operator);
		binaryExpression.setRightOperand(value);		
        return binaryExpression;
        
	}
	
	private static List createOrderByClause(EClass eClass, ISortElement[] elements){

		if(elements==null)
			return Collections.EMPTY_LIST;
		
		List orderByCriteria = new ArrayList();
		List attributes = eClass.getEAllAttributes();
		EAttribute attribute = null;
		for(int i=0;i<elements.length;i++){
			attribute = findAttributeInList(elements[i].getSortColumn(), attributes);
			if(attribute!=null){
				orderByCriteria.add(createOrderByElement(elements[i], attribute));
			}
		}
		return orderByCriteria;
		
	}
	
	private static OrderByElement createOrderByElement(ISortElement element, EAttribute attribute){
		OrderByElement orderByElement = ExtensionsFactory.eINSTANCE.createOrderByElement();
		LeftOperand leftOperand = ExtensionsFactory.eINSTANCE.createLeftOperand();
		leftOperand.setFeature(attribute);		
		orderByElement.setOperand(leftOperand);
		if(element.isAscending())
			orderByElement.setOperator(OrderByOperators.ASC_LITERAL);
		else
			orderByElement.setOperator(OrderByOperators.DESC_LITERAL);
		return orderByElement;
	}
	
	private static EAttribute findAttributeInList(String name, List attributeList){	
		int s = attributeList.size();
		for(int i=0;i<s;i++){			
			EAttribute attr = (EAttribute) attributeList.get(i);
			if(attr.getName().equalsIgnoreCase((name)))
				return attr;
		}
		return null;
	}
	
	private static boolean isFilterInWhereClause(IFilterElement element, List whereClause){	
		
		int s = whereClause.size();
		SimpleBinaryExpression where = null;
		for(int i=0;i<s;i++){			
			where = (SimpleBinaryExpression) whereClause.get(i);
			if(where!=null && where.getOperand().getFeature().getName().equalsIgnoreCase(element.getAttribute()) &&
			   where.getOperator()==RelationalOperators.get(element.getOperator()) && element.getValue().equals(where.getRightOperand()))
				return true;
		}
		return false;
	}
	
	public static boolean compare(Query query1, Query query2){
		
		List outputElements1 = query1.getOutputElements();
		List outputElements2 = query2.getOutputElements();
		if(outputElements1.size()!= outputElements2.size())
			return false;
		int s = outputElements1.size();
		for(int i=0;i<s;i++){
			LeftOperand left1 = (LeftOperand)outputElements1.get(i);
			LeftOperand left2 = (LeftOperand)outputElements2.get(i);
			if((left1!=null && left2==null)||(left1==null&&left2!=null))
				return false;
			if(left1!=null && left2!=null && left1.getType()!=left2.getType()){
				return false;
			}else{
				if(left1!=null && left2!=null && left1.getFeature()!=left2.getFeature())
					return false;
			}
		}
		List sourceElements1 = query1.getSources();
		List sourceElements2 = query2.getSources();
		if(sourceElements1.size()!= sourceElements2.size())
			return false;
		s = sourceElements1.size();
		for(int i=0;i<s;i++){
			if(sourceElements1.get(i)!=null && !sourceElements1.get(i).equals(sourceElements2.get(i))){
				return false;
			}
		}

		if(query1 instanceof SimpleSearchQuery && query2 instanceof SimpleSearchQuery){
			List whereElements1 = ((SimpleSearchQuery)query1).getWhereExpressions();
			List whereElements2 = ((SimpleSearchQuery)query2).getWhereExpressions();
			if(whereElements1.size()!= whereElements2.size())
				return false;
			s = whereElements1.size();
			for(int i=0;i<s;i++){
				SimpleBinaryExpression expression1 = (SimpleBinaryExpression)whereElements1.get(i);
				SimpleBinaryExpression expression2 = (SimpleBinaryExpression)whereElements2.get(i);
				if((expression1!=null && expression2==null)||(expression1==null&&expression2!=null))
					return false;
				if(expression1.getOperand().getFeature()!=expression2.getOperand().getFeature())
					return false;				
				if(expression1.getOperator().getValue() !=expression2.getOperator().getValue())
					return false;
				if(!expression1.getRightOperand().equals(expression2.getRightOperand()))
					return false;				
			}
			
		}
		return true;
	}
}
