/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.internal.reports;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.test.http.TestHttpPlugin;
import org.eclipse.hyades.ui.sample.svg.generator.GraphicDocumentProperties;
import org.eclipse.hyades.ui.sample.svg.generator.SVGBarChart;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author amathur
 *
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public class ReportGraph 
{
	IFile outputFile = null;
	ArrayList list = null;
	ReportGraphConfiguration config = null;
	
	public ReportGraph(ArrayList list, ReportGraphConfiguration configuration)
	{
		this.list = list;
		this.config=configuration;		
	}
	
	
	public InputStream generate(IPath svgPath)
		throws Exception {
			
	
		// Defines a new color palette and initializes it with white in all the positions
		ArrayList colourPalette = new ArrayList(); 

		// creates a Document by collecting information from the Component Test objects
		Document newDoc = generateDataDocument(list, colourPalette);
		if(newDoc == null)
			return null;

		// turn the DOM into an SVG chart
		GraphicDocumentProperties properties = new GraphicDocumentProperties();
		properties.setDataDocument(newDoc);
		
		//TODO: make width dynamic
		properties.setGraphicWidth(new Integer(config.getMinWidth()).toString());
		
		properties.setGraphicHeight(new Integer(config.getHeight()).toString());
		//properties.setTimestampSuppressed("true");
		properties.setLegendSuppressed("true"); //$NON-NLS-1$
		
		//Turn off preferences
		properties.setUserPreferencesSuppressed("true"); //$NON-NLS-1$

		//CR - Add custom colour palettes 
		String[] newPalette = {"red", "green", "blue", "yellow", "black", "cyan", "#996699", "#FF00CC", "#000099"}; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$ //$NON-NLS-8$ //$NON-NLS-9$
		properties.addGraphicColorPalette(TestHttpPlugin.getString("ReportGraph.0"), newPalette); //$NON-NLS-1$

		//		create the bar chart SVG
		SVGBarChart svgBarChart = new SVGBarChart();
		Document generatedDocument = svgBarChart.generateGraphicDocument(properties);
		
		// write the SVG to a file
		IPath svgPathExt = svgPath.removeFileExtension().addFileExtension("svg"); //$NON-NLS-1$
		String svgFileName = createFile(svgPathExt,svgBarChart.serializeGeneratedDocumentToString(generatedDocument)).getName();
		String svgURL = TestHttpPlugin.getString("SVG_URL"); //$NON-NLS-1$
		
		// create the HTML to return to caller
		StringBuffer sb = new StringBuffer();
		sb.append("<html>\n").append("<head>\n").append("<title>").append(getUserText("chart.Title")).append("</title>").append("</head>\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$
		sb.append("<body>\n").append("<embed type=\"image/svg+xml\" src=\""); //$NON-NLS-1$ //$NON-NLS-2$
		sb.append(svgFileName).append("\" ").append("width=\"").append(config.getMinWidth()).append("\" "); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		sb.append("height=\"").append(config.getHeight()).append("\" "); //$NON-NLS-1$ //$NON-NLS-2$
		sb.append("name=\"mySvgDocument\""); //$NON-NLS-1$
		sb.append("/>\n"); //$NON-NLS-1$
		sb.append("<p><font face=\"arial\" font size = \"1\">").append(svgURL).append("</font><p>"); //$NON-NLS-1$ //$NON-NLS-2$
		sb.append("</body>").append("</html>"); //$NON-NLS-1$ //$NON-NLS-2$

		IPath htmlPath = svgPath.removeFileExtension().addFileExtension("html"); //$NON-NLS-1$
		outputFile = createFile(htmlPath, sb.toString());
		
		return new ByteArrayInputStream(sb.toString().getBytes());
	}

	protected Document generateDataDocument(ArrayList list, ArrayList colourPalette)
	throws Exception
	{
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document document = builder.newDocument();

		// create the root element, i.e. configuration
		Element root = document.createElement("configuration"); //$NON-NLS-1$
		document.appendChild(root);
		
		root.setAttribute("title", config.getTitle()); //$NON-NLS-1$
		root.setAttribute("legendtitle", config.getLegendTitle()); //$NON-NLS-1$
		root.setAttribute("timestampPrefix", config.getTimeStampPrefix()); //$NON-NLS-1$
		

		// Create the element dataUpdate
		Element dataUpdateElement = document.createElement("dataUpdate"); //$NON-NLS-1$
		root.appendChild(dataUpdateElement);
		Date dateNow = new Date();
		SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");	 //$NON-NLS-1$
		dataUpdateElement.setAttribute("timestamp", dateFormatter.format(dateNow)); //$NON-NLS-1$

		/* Create a list of dataset elements (each dataset correponds to a line in the graph) inside the 
		 * dataUpdate element.  If the line is added then the colourPalette will be changed so it can be
		 * drawn in a specific color,
		 */
		int position = 0;			
		//CR Temp for TEST	
		if(addBarData(document, dataUpdateElement, position, getUserText("chart.expectedAttempt"), list)) //$NON-NLS-1$
		{
			colourPalette.add("#D3D3D3"); //grey //$NON-NLS-1$
			position++;
		}

		//CR Temp for TEST			
		createXData(document, dataUpdateElement, list);
		
		// Create the categorization element which contains the information
		// related Y axis
		//CR Temp for TEST		
		createYData(document, dataUpdateElement, list);

		return document;
	}
	
	/**
	 * Adds a line data to the document.  The line will represent one of the values
	 * of the execution statistics.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param int the position of the line
	 * @param String the name of the line
	 * @param int the type of statistic value will be represented on the line
	 * @param DayStatistic[] the sorted daily statistics
	 * @return boolean if the line data was added
	 */
	protected boolean addBarData(Document document, Element dataUpdateElement, int position, String name, ArrayList list)
	{
			
		Element dataset = document.createElement("dataSet"); //$NON-NLS-1$
		dataUpdateElement.appendChild(dataset);
		dataset.setAttribute("label",name); //$NON-NLS-1$
		dataset.setAttribute("position", new Integer(position).toString()); //$NON-NLS-1$
		dataset.setAttribute("dataRange1Ref", "day"); //$NON-NLS-1$ //$NON-NLS-2$
		dataset.setAttribute("dataRange2Ref", "count"); //$NON-NLS-1$ //$NON-NLS-2$
		
		Element data = document.createElement("dataPoint"); //$NON-NLS-1$
		dataset.appendChild(data);	
		data.setAttribute("value1", "0"); //$NON-NLS-1$ //$NON-NLS-2$
		data.setAttribute("value2", "0"); //$NON-NLS-1$ //$NON-NLS-2$

		Iterator iter = list.iterator();
		int i = 1;
		
		//
		while (iter.hasNext()) {
			
			IReportGraphDataProvider provider = (IReportGraphDataProvider) iter.next();
			data = document.createElement("dataPoint"); //$NON-NLS-1$
			dataset.appendChild(data);
			
			data.setAttribute("value1", Integer.toString(i++)); //$NON-NLS-1$
			data.setAttribute("value2", provider.getValue());			 //$NON-NLS-1$
		}
		return true;
	}
	
	/**
	 * Creates the data that is presented at the X axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 * @param DayStatistic[] the sorted daily statistics
	 */
	protected void createXData(Document document, Element dataUpdateElement, ArrayList list)
	{
		Element dataRangeElement = document.createElement("dataRange"); //$NON-NLS-1$
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Title information
		StringBuffer xTitle = new StringBuffer();
		///xTitle.append(xTitle);
		dataRangeElement.setAttribute("id", "day"); //$NON-NLS-1$ //$NON-NLS-2$
		dataRangeElement.setAttribute("location", "S"); //$NON-NLS-1$ //$NON-NLS-2$
		//CR- Fix categorization attribute
		dataRangeElement.setAttribute("type", "categorization"); //$NON-NLS-1$ //$NON-NLS-2$
		dataRangeElement.setAttribute("label", xTitle.toString()); //$NON-NLS-1$
		
		// Adds a "0"  day
		Element segmentMarkerElement = document.createElement("segmentMarker"); //$NON-NLS-1$

		// Add the X axis points
		Iterator iter = list.iterator();
		int i = 1;

		while (iter.hasNext()) {
			
			//get each url and store it in a string
			IReportGraphDataProvider provider = (IReportGraphDataProvider) iter.next();
			segmentMarkerElement = document.createElement("segmentMarker"); //$NON-NLS-1$
			dataRangeElement.appendChild(segmentMarkerElement);
			String name = provider.getId();
			
			StringBuffer buf = new StringBuffer();	
			
			//Search for all slashes and add a white space to cause name to wrap
			if (name.indexOf('/') != -1) {
				StringTokenizer st = new StringTokenizer(name, "/"); //$NON-NLS-1$
				
				while (st.hasMoreTokens()) {
					buf.append(st.nextToken() + "/ "); //$NON-NLS-1$
					}
					String str = buf.toString();
					
					//Remove last instance of slash
					if (str.endsWith("/ ") == true) { //$NON-NLS-1$
						int b = str.lastIndexOf("/ "); //$NON-NLS-1$
					 	str = str.substring(0, b);
					}
					//Search for queries and delete them to help limit length of URLs
					int index = 0;
					index = str.indexOf('?', index);
						if (index != -1) {
							String substring = str.substring(0, index);
							System.out.println(substring);
							segmentMarkerElement.setAttribute("label", substring); //$NON-NLS-1$
						}
						else {
							segmentMarkerElement.setAttribute("label", str); //$NON-NLS-1$
						}
			}
			else {
				segmentMarkerElement.setAttribute("label", name); //$NON-NLS-1$
			}
			segmentMarkerElement.setAttribute("value", Integer.toString(i++)); //$NON-NLS-1$
			
			if (i > 6) {
				int m = config.getMinWidth() + 80;
				config.setMinWidth(m);
			}
		}
		
	}

	/**
	 * Creates the data that is presented at the Y axis.
	 * 
	 * @param Document the Document
	 * @param Element the data element
	 */
	protected void createYData(Document document, Element dataUpdateElement, ArrayList list)
	{
		Element dataRangeElement = document.createElement("dataRange"); //$NON-NLS-1$
		dataUpdateElement.appendChild(dataRangeElement);
		
		// Define the Y axis range
		dataRangeElement.setAttribute("id", "count"); //$NON-NLS-1$ //$NON-NLS-2$
		dataRangeElement.setAttribute("location", "W"); //$NON-NLS-1$ //$NON-NLS-2$
		dataRangeElement.setAttribute("label", config.getYTitle()); //$NON-NLS-1$

		// First figure out the min & max
		// Assume 10 steps
		int number = list.size();
		Iterator iter = list.iterator();
		// Figure out the min & max
		double min = 1000000;
		double max = 0;
		while (iter.hasNext()) {
			IReportGraphDataProvider dataProvider = (IReportGraphDataProvider) iter.next();
			double val=-1;
			String dataString =dataProvider.getValue(); 
			String dataName = dataProvider.getId();
			try {
				val = Double.parseDouble(dataString);
			}
			catch (NumberFormatException e)
			{
				val = 0;
			}
 
			if (val < min) {
				min = val;
			}
			if (val > max) {
				max = val;
			}
		}
		
		// Figure out the multiplication factor
		
		double factor = 1;
		
		if (max < .1) {
			factor = .01;		
		}
		else if (max < .5) {
			factor = .05;
		}
		else if (max < 1) {
			factor = .1;
		}
		else if (max < 5) {
			factor = .5;
		}

		else if (max < 10) {
			factor = 1;
		}
		else if (max < 20) {
			factor = 2;
		}
		else if (max > 20) {
			factor = (double)(int)(max / 9); //change to integer
		
		}

		for ( int m = 0; m < 11; m++) {
			Element segmentMarkerElement = document.createElement("segmentMarker"); //$NON-NLS-1$
			dataRangeElement.appendChild(segmentMarkerElement);
			
			String val = Double.toString(m*factor);
			 
			//due to floating point issues, we need to verify that this has only a set # of decimal place
			int nDecimals = 1;
			if (factor < .1)
			 	nDecimals = 2;
			
			if (val.indexOf('.')>=1)
			{
				val += "00"; //ensure string is long enough //$NON-NLS-1$
				val = val.substring(0, val.indexOf('.')+1+nDecimals);
			}
			
			//fix this
			segmentMarkerElement.setAttribute("value", val); //$NON-NLS-1$
			
			//segmentMarkerElement.setAttribute("label", Integer.toString((int)metric.getAvgResponse()));
			segmentMarkerElement.setAttribute("label", val); //$NON-NLS-1$
			
		}
	}


	/**
	 * Returns the values that will be presented for the user.
	 * 
	 * @param String key The key for the value
	 * @return String the value
	 */
	protected String getUserText(String key)
	{
		if("chart.Title".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.1"); //$NON-NLS-1$
		
		if("chart.Legend".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.2"); //$NON-NLS-1$
			
		if("chart.timestampPrefix".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.3"); //$NON-NLS-1$
		
		if("chart.Xtitle".equals(key))	 //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.4"); //$NON-NLS-1$
			
		if("chart.Ytitle".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.5"); //$NON-NLS-1$


		if("chart.attempt".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.6"); //$NON-NLS-1$
			
		if("chart.error".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.7"); //$NON-NLS-1$
		
		if("chart.pass".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.8"); //$NON-NLS-1$
		
		if("chart.fail".equals(key))	 //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.9"); //$NON-NLS-1$
			
		if("chart.inconclusive".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.10"); //$NON-NLS-1$


		if("chart.expectedAttempt".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.11"); //$NON-NLS-1$
			
		if("chart.expectedError".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.12"); //$NON-NLS-1$
		
		if("chart.expectedPass".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.13"); //$NON-NLS-1$
		
		if("chart.expectedFail".equals(key))	 //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.14"); //$NON-NLS-1$
			
		if("chart.expectedInconclusive".equals(key)) //$NON-NLS-1$
			return TestHttpPlugin.getString("ReportGraph.15"); //$NON-NLS-1$
		
		return key;
	}

	/**
	 * Creates a file in the same location and name of the <i>originalFile</i> but with a different
	 * extension.
	 * 
	 * @param String IPath the IPath for the generated svg file
	 * @param String extension The extension of the new file
	 * @param String content The content of the new file.
	 * @return IFile The created file;
	 */
	protected IFile createFile(IPath svgPath, String content) throws CoreException
	{
				
		IFile file = null;
		
		try 
		{
			file =	ResourcesPlugin.getWorkspace().getRoot().getFile(svgPath);
			ByteArrayInputStream inputStream = new ByteArrayInputStream(content.getBytes());
			if (file.exists()) 
			{
				try 
				{
					file.setContents(
						inputStream,
						true,
						false,
						new NullProgressMonitor());
				}
				catch (CoreException e1) {
					e1.printStackTrace();
				}
			}
			else
			{
				file.create(inputStream,true,new NullProgressMonitor());
			}
					
		}
		catch (RuntimeException e) {
			e.printStackTrace();
		}

		return file;
	}
	


}
