/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.trace.loaders;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCTraceObject;
import org.eclipse.hyades.uml2sd.trace.TraceSDPlugin;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.FoundInPage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCallStack;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsableSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsableSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedLifelines;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceCollapsedSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUtils;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifeline;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceLifelineDraft;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceMessageAndCallAssociation;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TracePage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceProcess;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceSyncMessage;
import org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceSyncMessageReturn;
import org.eclipse.hyades.uml2sd.trace.preferences.ITraceInteractionPreferenceListener;
import org.eclipse.hyades.uml2sd.trace.selection.IDateSelection;
import org.eclipse.hyades.uml2sd.trace.selection.IEObjectSelection;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDAdvancedPagingProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDCollapseProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.provider.ISDInternalMesFilterProvider;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.FilterCriteria;
import org.eclipse.hyades.uml2sd.ui.actions.widgets.FilterListDialog;
import org.eclipse.hyades.uml2sd.ui.core.BaseMessage;
import org.eclipse.hyades.uml2sd.ui.core.Frame;
import org.eclipse.hyades.uml2sd.ui.core.GraphNode;
import org.eclipse.hyades.uml2sd.ui.core.ITimeRange;
import org.eclipse.hyades.uml2sd.ui.core.Lifeline;
import org.eclipse.hyades.uml2sd.ui.core.LifelineCategories;
import org.eclipse.hyades.uml2sd.ui.drawings.impl.ImageImpl;
import org.eclipse.jface.dialogs.DialogSettings;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPart;

/**
 * The abstract trace loader.
 * It is based on the parsing of MethodInvocations from the model.
 * Lifelines attribution is made by the concrete classes.
 */
public class TraceInteractions extends BaseTraceInteractions
	implements ISDFilterProvider, ISDAdvancedPagingProvider,
			   ISDInternalMesFilterProvider, ISDCollapseProvider, ITraceInteractionPreferenceListener {

	/**
	 * Categories:
	 * - first one for classes
	 * - second one for instances of classes
	 * - third one for collapsed classes and/or objects
	 */
	public static final int CATEGORY_CLASSES = 0;
	public static final int CATEGORY_OBJECTS = 1;
	public static final int CATEGORY_NODE = 2;
	public static final int CATEGORY_AGENT = 3;
	public static final int CATEGORY_PROCESS = 4;
	public static final int CATEGORY_THREAD = 5;
	public static final int CATEGORY_COLLAPSED = 6;
	public static final int CATEGORY_CARD = 7;

	protected static LifelineCategories[] traceCategories;
	static {
		traceCategories = new LifelineCategories[CATEGORY_CARD];
		traceCategories[CATEGORY_CLASSES] = new LifelineCategories();
		traceCategories[CATEGORY_CLASSES].setName(TraceSDPlugin.getResourceString("STR_CLASS")); //$NON-NLS-1$
		traceCategories[CATEGORY_CLASSES].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("class.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_OBJECTS] = new LifelineCategories();
		traceCategories[CATEGORY_OBJECTS].setName(TraceSDPlugin.getResourceString("STR_OBJECT")); //$NON-NLS-1$
		traceCategories[CATEGORY_OBJECTS].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("classobject_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_NODE] = new LifelineCategories();
		traceCategories[CATEGORY_NODE].setName(TraceSDPlugin.getResourceString("STR_NODE")); //$NON-NLS-1$
		traceCategories[CATEGORY_NODE].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("node_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_AGENT] = new LifelineCategories();
		traceCategories[CATEGORY_AGENT].setName(TraceSDPlugin.getResourceString("STR_AGENT")); //$NON-NLS-1$
		traceCategories[CATEGORY_AGENT].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("agent_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_PROCESS] = new LifelineCategories();
		traceCategories[CATEGORY_PROCESS].setName(TraceSDPlugin.getResourceString("STR_PROCESS")); //$NON-NLS-1$
		traceCategories[CATEGORY_PROCESS].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("process_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_THREAD] = new LifelineCategories();
		traceCategories[CATEGORY_THREAD].setName(TraceSDPlugin.getResourceString("STR_THREAD")); //$NON-NLS-1$
		traceCategories[CATEGORY_THREAD].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("thread_obj.gif"))); //$NON-NLS-1$
		traceCategories[CATEGORY_COLLAPSED] = new LifelineCategories();
		traceCategories[CATEGORY_COLLAPSED].setName(TraceSDPlugin.getResourceString("STR_COLLAPSED_LIFELINE")); //$NON-NLS-1$
		traceCategories[CATEGORY_COLLAPSED].setImage(new ImageImpl(TraceSDPlugin.getResourceImage("collapsed_obj.gif"))); //$NON-NLS-1$
	}
	
	/**
	 * Internal message filtering:
	 * internal means that the message destination is the same as the origin (same lifeline)
	 * see also ISDInternalMesFilterProvider
	 */
	private boolean internalMessagesFiltered = false;
	
	/**
	 * Paging related attributes
	 */
	private int maximumMessagesByPage = (int)TraceSDPlugin.getDefault().getPreferenceStore().
		getLong(ITraceInteractionPreferenceListener.PAGE_SIZE);
	private int nbMessages, nbMessagesInThisPage, pageNumber = 0;
	private boolean hasNextPage = false;

	/**
	 * Implementation of IUml2SDLoader
	 */
	public void onSetViewer() {
		view.setSDFilterProvider(this);
		view.setSDPagingProvider(this);
		view.setInternalMessageFilterProvider(this);
		view.setCollapsingProvider(this);
		view.getSDWidget().setReorderMode(true);
		currentFilters = FilterListDialog.getGlobalFilters();
		collapsedLifelines = loadCollapsedLifelines();
		collapsedMessages = loadCollapsedMessages();
	}
	
	/**
	 * 
	 */
	public void clearInternals() {
		super.clearInternals();
		pageNumber = 0;
		tracePages = null;
		selectedThreadList = null;
	}
	
	/**
	 * @return
	 */
	protected TraceInteractionUpdate createTraceInteractionUpdate() {
		TraceInteractionUpdate ret = new TraceInteractionUpdate();
		if (selectedThreadList != null) {
			ret.setSelectedThreadList(selectedThreadList);
		}
		return ret;
	}

	/**
	 * Parse the model again for a new translation on the same eObject (page changes, ...)
	 * @param update
	 */
	protected synchronized void updateSD(TraceInteractionUpdate update) {
		// Prepare for showing same date after update
		if (currentProcesses==null)
			return;
		Frame oldFrame = frame;
		if (frame != null &&
			!update.isDateSelectionChanged() &&
			!update.isPageChanged() &&
			!update.isFindRequired()) {
			double visibleDate = 0;
			currentProcesses.init();
			TraceSyncMessage firstSM = (TraceSyncMessage)(frame.getSyncMessage(frame.getFirstVisibleSyncMessage()));
			if (firstSM != null) {
				visibleDate = firstSM.getStartDate() + currentProcesses.getStartTime();
			} else {
				TraceSyncMessageReturn firstSMR = (TraceSyncMessageReturn)(frame.getSyncMessageReturn(frame.getFirstVisibleSyncMessageReturn()));
				if (firstSMR != null) {
					visibleDate = firstSMR.getStartDate() + currentProcesses.getStartTime();
				}
			}
			if (visibleDate != 0) {
				update.setDateSelectionChanged(true);
				update.setDateToShow(visibleDate);
			}
			if (TraceSDPlugin.debugLoader) {
				TraceSDPlugin.debugTraceLoader("visibleDate="+visibleDate+" "+update.isPageChanged()+" "+update.isDateSelectionChanged()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
		}
		createFrame();
		if (currentProcesses == null) {
			return;
		}
		if ((!update.isPageChanged()) &&
			(update.isInternalMessageFilteringChanged() ||
			 update.isMaxMessagesByPageChanged() ||
			 update.isFilteringChanged() ||
			 update.isThreadSelectionChanged() ||
			 update.isHorizontalCollapsingChanged() ||
			 update.isVerticalCollapsingChanged())) {
			computeModel();
		}
		try {
			fillPage(update);
		} catch (FoundInPage foundInPage) {
			if (TraceSDPlugin.debugLoader) {
				foundInPage.printStackTrace(System.out);
			}
			if (update.isDateSelectionChanged() ||
				update.isFindRequired()) {
				update.setPageChanged(true);
				pageNumber = foundInPage.getPage();
				updateSD(update);
				return;
			}
		} catch (Throwable e) {
			e.printStackTrace();
		}
		if (!update.isPageChanged()) {
			if (update.isDateSelectionChanged() &&
				frame.lifeLinesCount() > 0 &&
				update.getVisibleGraphNode() == null) {
				// Not found because after the last event: show the last message of the last page
				update.setPageChanged(true);
				pageNumber = pages-1;
				createFrame();
				try {
					fillPage(update);
				} catch (FoundInPage foundInPage) {
					// Nothing to do
				}
				update.setPreviousVisibleGraphNode(null);
				update.setVisibleGraphNode(frame.getSyncMessageReturn(frame.syncMessageReturnCount()-1));
			}
			else if (update.isFindRequired() &&
				(update.getFindResults() == null || update.getFindResults().size() == 0)) {
				// Not found!
				frame = oldFrame;
				return;
			}
		}
		if (update.getVisibleGraphNode() != null) {
			if (update.getPreviousVisibleGraphNode() != null &&
			    update.getPreviousVisibleGraphNode() instanceof BaseMessage &&
				update.getVisibleGraphNode() instanceof BaseMessage) {
				if (view != null && !view.getSDWidget().isDisposed() && !isCurrentLoadCanceled()) {
					setFrameName();
					view.setFrameAndEnsureVisibleSync(frame, update.getPreviousVisibleGraphNode());
					if (update.isExternalDateSelectionChanged()) {
						view.getTimeCompressionBar().highlightRegionSync(
								(BaseMessage)update.getPreviousVisibleGraphNode(),
							    (BaseMessage)update.getVisibleGraphNode());
					}
					return;
				}
			} else {
				if (view != null && !view.getSDWidget().isDisposed() && !isCurrentLoadCanceled()) {
					setFrameName();
					view.setFrameAndEnsureVisibleSync(frame, update.getVisibleGraphNode());
					return;
				}
			}
		}
		if (view != null && !view.getSDWidget().isDisposed() && !isCurrentLoadCanceled()) {
			setFrameName();
			view.setFrameSync(frame);
			return;
		}
	}
	
	/**
	 * @return
	 */
	protected void createFrame() {
		super.createFrame();
		frame.setLifelineCategories(traceCategories);
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("TraceInteractions.createFrame()"); //$NON-NLS-1$
		}
	}

	/**
	 * @param process
	 */
	public void setFrameName() {
		if (frame == null) {
			return;
		}
		if (currentProcesses != null && currentProcesses.size() >= 1) {
			if (currentProcesses.size() == 1) {
				TRCProcess process = ((TraceProcess)currentProcesses.get(0)).getProcess();
				frame.setName(process.getAgent().getAgentProxy().getProcessProxy().getName()+
						  " [Pid "+process.getPid()+"]"+pageMessage()); //$NON-NLS-1$ //$NON-NLS-2$
			} else {
				frame.setName(TraceSDPlugin.getResourceString("STR_SEVERAL_PROCESSES")+pageMessage()); //$NON-NLS-1$
			}
		} else {
			frame.setName(TraceSDPlugin.getResourceString("STR_NO_DATA_AVAILABLE")); //$NON-NLS-1$
		}
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("setFrameName() "+frame.getName()); //$NON-NLS-1$
		}
	}

	/**
	 * This is the title
	 */
	public String getTitleString() {
		String ret = getFinalTitleString();
		if (pages <= 1) {
			return ret;
		}
		return ret + pageMessage();
	}
	
	/**
	 * @return
	 */
	protected String pageMessage() {
		String m = ""; //$NON-NLS-1$
		if (pages <= 1) {
			return m;
		}
		m = " [" + (1+pageNumber) + "/" + pages + " " + //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			TraceSDPlugin.getResourceString("STR_PAGES") + //$NON-NLS-1$
			"]"; //$NON-NLS-1$
		return m;
	}
	
	private int totalNbMessages;
	private int pages;
	private ArrayList tracePages;
	
	/**
	 * Compute the number of pages with and also without internal messages
	 * @param invocations
	 */
	protected void computeModel() {
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("computeModel() "+maximumMessagesByPage); //$NON-NLS-1$
		}
		totalNbMessages = 0;
		instancesMap = new HashMap();
		currentProcesses.init();
		for (TRCFullMethodInvocation methodInvocation = currentProcesses.consumeMethodInvocation();
		 	 methodInvocation != null &&
			 !isCurrentLoadCanceled();
		 	 methodInvocation = currentProcesses.consumeMethodInvocation()) {
			TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
			if (isMessageCollapsed(invokedBy)) {
				continue;
			}
			EObject instanceFrom = null;
			EObject instanceTo = null;
			if (invokedBy != null) {
				instanceFrom = getLifelineEObjectFromMethodInvocation(invokedBy);
				instanceTo = getLifelineEObjectFromMethodInvocation(methodInvocation);
				if (internalMessagesFiltered &&	instanceFrom == instanceTo) {
					continue;
				}
			}
			TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(null, methodInvocation);
			TraceLifelineDraft traceLifelineFrom = null;
			if (invokedBy != null) {
				traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(null, invokedBy);
			}
			if (traceLifelineTo.isFiltered() ||
				(traceLifelineFrom != null &&
				 traceLifelineFrom.isFiltered())) {
				continue;
			}
			String name = methodInvocation.getMethod().getName();
			if (isSyncMessageFiltered(name)) {
				continue;
			}
			// Count one way and return if set
			totalNbMessages++;
			if (methodInvocation.getExitTime() > 0.0) {
				totalNbMessages++;
			}
		}
		pages = totalNbMessages / maximumMessagesByPage;
		if (totalNbMessages % maximumMessagesByPage > 0) {
			pages++;
		}
		instancesMap = null;
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("computeModel() interrupted("+isCurrentLoadCanceled()+") has counted "+pages+" pages"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
	}
	
	private HashMap instancesMap;
	private int currentMethodInvocationIndex;
	
	/**
	 * @param update
	 */
	public void fillPage(TraceInteractionUpdate update) {
		if (!update.isThreadSelectionChanged()) {
			update.setSelectedThreadList(selectedThreadList); // let's keep previous one
		}
		tracePages = new ArrayList();
		instancesMap = new HashMap();
		traceThreadMap = new HashMap();
		traceThreadList = new ArrayList();
		nbMessages = 0;
		nbMessagesInThisPage = 0;
		hasNextPage = false;
		currentMethodInvocationIndex = 0;
		currentProcesses.init(); // init again
		for (TRCFullMethodInvocation methodInvocation = currentProcesses.consumeMethodInvocation();
	 	 	 methodInvocation != null &&
			 !isCurrentLoadCanceled() &&
			 nbMessagesInThisPage < maximumMessagesByPage;
	 	 	 methodInvocation = currentProcesses.consumeMethodInvocation()) {
			if (TraceSDPlugin.debugLoader) {
				TraceSDPlugin.debugTraceLoader(nbMessages+":"+nbMessages/maximumMessagesByPage+" "+methodInvocation);			 //$NON-NLS-1$ //$NON-NLS-2$
			}
			
			if (isThreadFiltered(update, methodInvocation.getThread())) {
				continue;
			}

			// Let's continue with pending message returns
			finishPendingReturns(update,
								 TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation),
								 methodInvocation.getTicket());
			
			TraceSyncMessage syncMessage = createSyncMessage(methodInvocation, update);
			
			TraceCallStack callStack = getTraceCallStack(methodInvocation.getThread());
			callStack.pushCall(syncMessage, methodInvocation);
		}
		// Let's continue with pending message returns
		finishPendingReturns(update, Double.MAX_VALUE, Long.MAX_VALUE);

		// Let's do last execution occurrences
		Collection c = instancesMap.entrySet();
		for (Iterator i = c.iterator(); i.hasNext(); ) {
			Map.Entry e = (Map.Entry)i.next();
			TraceLifelineDraft draft = (TraceLifelineDraft)e.getValue();
			if (TraceSDPlugin.debugLoader) {
				TraceSDPlugin.debugTraceLoader(draft.getLifeline()+":"+draft.getNbUser()); //$NON-NLS-1$
			}
			if (draft.getNbUser() > 0) {
				draft.setEnd(nbMessagesInThisPage+1);
				if (draft.getStart() == Integer.MAX_VALUE) {
					draft.setStart(0);
				}
				setExecutionOccurence(draft);
			}
		}
		instancesMap = null;
		traceThreadMap = null;
		traceThreadList = null;
	}
	
	/**
	 * @param update
	 * @param uptodate
	 */
	protected void finishPendingReturns(TraceInteractionUpdate update,
									    double uptodate,
										long ticket) {
		while (getTraceThreadListFirstTime() < uptodate ||
			   (getTraceThreadListFirstTime() == uptodate &&
			   	getTraceThreadListFirstTicket() < ticket)) {
			TraceCallStack callStack = (TraceCallStack)traceThreadList.get(0);
			TraceMessageAndCallAssociation tmaca = callStack.popCall();
			TRCFullMethodInvocation methodInvocation = tmaca.getMethodInvocation();
			if (methodInvocation.getExitTime() > 0.0) {
				TraceSyncMessageReturn syncMessageReturn = createSyncMessageReturn(methodInvocation, update);
				if (tmaca.getMessage() != null && syncMessageReturn != null) {
					syncMessageReturn.setMessage(tmaca.getMessage());
				}
			}
		}
	}

	/**
	 * @param methodInvocation
	 * @param update
	 */
	private TraceSyncMessage
			     createSyncMessage(TRCFullMethodInvocation methodInvocation,
								   TraceInteractionUpdate update) {
		TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
		if (isMessageCollapsed(invokedBy)) {
			return null;
		}
		TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(frame, methodInvocation);
		TraceLifelineDraft traceLifelineFrom = null;
		if (invokedBy != null) {
			traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(frame, invokedBy);
		}
		if (traceLifelineTo.isFiltered() ||
			(traceLifelineFrom != null &&
			 traceLifelineFrom.isFiltered())) {
			return null;
		}
		if (internalMessagesFiltered && traceLifelineFrom == traceLifelineTo) {
			return null;
		}
		if (isSyncMessageFiltered(methodInvocation.getMethod().getName())) {
			return null;
		}
		nbMessages++;
		String name = getMessageName(methodInvocation, traceLifelineFrom == null);
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("in:"+nbMessages+":"+nbMessages/maximumMessagesByPage+" "+name+" "+TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (nbMessagesInThisPage == maximumMessagesByPage) {
			hasNextPage = true;
			return null;
		}
		if (traceLifelineFrom != null && traceLifelineFrom != traceLifelineTo) {
			traceLifelineFrom.addAUser();
		}
		traceLifelineTo.addAUser();
		if ((nbMessages-1) % maximumMessagesByPage == 0) {
			tracePages.add(new TracePage(TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation)));
		} else if ((nbMessages-1) % maximumMessagesByPage == maximumMessagesByPage-1) {
			((TracePage)tracePages.get(tracePages.size()-1)).setLastTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		}
		if (update.isPageChanged() &&
			(nbMessages-1) < pageNumber*maximumMessagesByPage) {
			return null;
		}
		if (!update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation) < update.getDateToShow()) {
					return null;
				} else {
					throw new FoundInPage(tracePages.size()-1);
				}
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (update.getFindStartingPage() <= tracePages.size()-1 &&
					TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					throw new FoundInPage(tracePages.size()-1);
				} else {
					return null;
				}
			}
		}
		if (traceLifelineFrom != null && traceLifelineFrom != traceLifelineTo && traceLifelineFrom.getNbUser() == 1) {
			traceLifelineFrom.setStart(nbMessagesInThisPage+1);
		}
		if (traceLifelineTo.getNbUser() == 1) {
			traceLifelineTo.setStart(nbMessagesInThisPage+1);
		}
		TraceSyncMessage syncMessage;
		if (isMessageCollapsed(methodInvocation)) {
			syncMessage = new TraceCollapsedSyncMessage();
		} else if (methodInvocation.getInvokes() != null &&
				   methodInvocation.getInvokes().size() > 0) {
			syncMessage = new TraceCollapsableSyncMessage();
		} else {
			syncMessage = new TraceSyncMessage();
		}
		if (update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation) < update.getDateToShow()) {
					update.setPreviousVisibleGraphNode(syncMessage);
				} else if (update.getVisibleGraphNode() == null) {
					update.setVisibleGraphNode(syncMessage);
				}
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					update.addFindResult(syncMessage);
				}
			}
		}
		syncMessage.model = methodInvocation;
		if (traceLifelineFrom != null) {
			traceLifelineFrom.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessage.setStartLifeline(traceLifelineFrom.getLifeline());
		} else { 
			traceLifelineTo.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
		}
		syncMessage.setName(name);
		syncMessage.setEndLifeline(traceLifelineTo.getLifeline());
		syncMessage.setTime(TraceInteractionUtils.getAbsoluteEntryTime(methodInvocation));
		if (isCurrentLoadCanceled()) {
			return null;
		}
		frame.addMessage(syncMessage);
		nbMessagesInThisPage++;
		return syncMessage;
	}
	
	/**
	 * @param methodInvocation
	 * @param update
	 */
	private TraceSyncMessageReturn
				 createSyncMessageReturn(TRCFullMethodInvocation methodInvocation,
										 TraceInteractionUpdate update) {
		TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
		if (isMessageCollapsed(invokedBy)) {
			return null;
		}
		TraceLifelineDraft traceLifelineTo = getTraceLifelineDraftForMethodInvocation(frame, methodInvocation);
		TraceLifelineDraft traceLifelineFrom = null;
		if (invokedBy != null) {
			traceLifelineFrom = getTraceLifelineDraftForMethodInvocation(frame, invokedBy);
		}
		if (traceLifelineTo.isFiltered() ||
			(traceLifelineFrom != null &&
			 traceLifelineFrom.isFiltered())) {
			return null;
		}
		if (internalMessagesFiltered && traceLifelineFrom == traceLifelineTo) {
			return null;
		}
		if (isSyncMessageFiltered(methodInvocation.getMethod().getName())) {
			return null;
		}
		nbMessages++;
		String name = getMessageName(methodInvocation, traceLifelineFrom == null);
		if (TraceSDPlugin.debugLoader) {
			TraceSDPlugin.debugTraceLoader("out:"+nbMessages+":"+nbMessages/maximumMessagesByPage+" "+name+" "+TraceInteractionUtils.getAbsoluteExitTime(methodInvocation)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (nbMessagesInThisPage == maximumMessagesByPage) {
			hasNextPage = true;
			return null;
		}
		if (traceLifelineFrom != null && traceLifelineFrom != traceLifelineTo) {
			traceLifelineFrom.removeAUser();
		}
		traceLifelineTo.removeAUser();
		if ((nbMessages-1) % maximumMessagesByPage == 0) {
			tracePages.add(new TracePage(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation)));
		} else if ((nbMessages-1) % maximumMessagesByPage == maximumMessagesByPage-1) {
			((TracePage)tracePages.get(tracePages.size()-1)).setLastTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		}
		if (update.isPageChanged() &&
			(nbMessages-1) < pageNumber*maximumMessagesByPage) {
			return null;
		}
		if (!update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteExitTime(methodInvocation) < update.getDateToShow()) {
					return null;
				} else {
					throw new FoundInPage(tracePages.size()-1);
				}
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (update.getFindStartingPage() <= tracePages.size()-1 &&
					TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					throw new FoundInPage(tracePages.size()-1);
				} else {
					return null;
				}
			}
		}
		if (traceLifelineFrom != null && traceLifelineFrom != traceLifelineTo && traceLifelineFrom.getNbUser() == 0) {
			traceLifelineFrom.setEnd(nbMessagesInThisPage+1);
			setExecutionOccurence(traceLifelineFrom);
		}
		if (traceLifelineTo.getNbUser() == 0) {
			traceLifelineTo.setEnd(nbMessagesInThisPage+1);
			setExecutionOccurence(traceLifelineTo);
		}
		TraceSyncMessageReturn syncMessageReturn;
		if (isMessageCollapsed(methodInvocation)) {
			syncMessageReturn = new TraceCollapsedSyncMessageReturn();
		} else if (methodInvocation.getInvokes() != null &&
				   methodInvocation.getInvokes().size() > 0) {
			syncMessageReturn = new TraceCollapsableSyncMessageReturn();
		} else {
			syncMessageReturn = new TraceSyncMessageReturn();
		}
		if (update.isPageChanged()) {
			if (update.isDateSelectionChanged()) {
				if (TraceInteractionUtils.getAbsoluteExitTime(methodInvocation) < update.getDateToShow()) {
					update.setPreviousVisibleGraphNode(syncMessageReturn);
				} else if (update.getVisibleGraphNode() == null) {
					update.setVisibleGraphNode(syncMessageReturn);
				}
			}
			if (update.isDateSelectionChanged() &&
				TraceInteractionUtils.getAbsoluteExitTime(methodInvocation) >= update.getDateToShow() &&
				update.getVisibleGraphNode() == null) {
				update.setVisibleGraphNode(syncMessageReturn);
			}
			if (update.isFindRequired() &&
				update.getFindCriteria().isSyncMessageSelected()) {
				if (TraceInteractionUtils.matchCriteria(name, update.getFindCriteria())) {
					update.addFindResult(syncMessageReturn);
				}
			}
		}
		syncMessageReturn.model = methodInvocation;
		if (traceLifelineFrom != null) {
			traceLifelineFrom.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
			syncMessageReturn.setEndLifeline(traceLifelineFrom.getLifeline());
		} else { 
			traceLifelineTo.getLifeline().setCurrentEventOccurrence(nbMessagesInThisPage+1);
		}
		syncMessageReturn.setName(name);
		syncMessageReturn.setStartLifeline(traceLifelineTo.getLifeline());
		syncMessageReturn.setTime(TraceInteractionUtils.getAbsoluteExitTime(methodInvocation));
		if (isCurrentLoadCanceled()) {
			return null;
		}
		frame.addMessage(syncMessageReturn);
		nbMessagesInThisPage++;
		return syncMessageReturn;
	}
	
	/**
	 * @param methodInvocation
	 * @param withThread
	 * @return
	 */
	private String getMessageName(TRCFullMethodInvocation methodInvocation, boolean withThread) {
		if (withThread) {
			return methodInvocation.getMethod().getName() + " ("+methodInvocation.getThread().getName()+")"; //$NON-NLS-1$ //$NON-NLS-2$
		} else {
			return methodInvocation.getMethod().getName();
		}
	}
	
	/**
	 * Get the instance corresponding to this method invocation or create it
	 * @param f
	 * @param mi
	 * @return
	 */
	protected TraceLifelineDraft 
				getTraceLifelineDraftForMethodInvocation
					(Frame f, TRCFullMethodInvocation mi) {
		EObject obj = getLifelineEObjectFromMethodInvocation(mi);
		TRCTraceObject o = mi.getOwningObject();
		String name = getLifeLineTitle(obj, true);
		String id = getLifeLineId(obj);
		TraceLifelineDraft draft = (TraceLifelineDraft)instancesMap.get(id);
		if (draft == null) {
			draft = new TraceLifelineDraft();
			if (isLifelineFiltered(name)) {
				draft.setFiltered(true);
				return draft;
			}
			ArrayList list = collapsedLifelines != null ?
							 (ArrayList)collapsedLifelines.get(id) : 
							 null;
			if (list != null) {
				for (ArrayList l = (ArrayList)collapsedLifelines.get(list);
					 l != null;
					 list = l, l = (ArrayList)collapsedLifelines.get(l));
				name = getCollapsedLifelineName(list);
				if (name.length() > 300) {
					name = name.substring(0, 297)+"..."; //$NON-NLS-1$
				}
				if (isLifelineFiltered(name)) {
					draft.setFiltered(true);
					return draft;
				}
				if (f == null) {
					return draft;
				}
				TraceCollapsedLifelines traceCollapsedLifelines = new TraceCollapsedLifelines();
				draft.setLifeline(traceCollapsedLifelines);
				traceCollapsedLifelines.setCollapsedLifelines(list);
				traceCollapsedLifelines.setName(name);
				putInstancesMap(list, draft);
				traceCollapsedLifelines.setCategory(CATEGORY_COLLAPSED);
			} else {
				if (f == null) {
					return draft;
				}
				TraceLifeline traceLifeline = new TraceLifeline();
				draft.setLifeline(traceLifeline);
				traceLifeline.setEObject(obj);
				instancesMap.put(id, draft);
				if (o.getIsA() != null) {
					traceLifeline.setName(name);
					traceLifeline.setCategory(getLifeLineCategory(obj));
				}
			}
			f.addLifeLine(draft.getLifeline());
			draft.setStart(Integer.MAX_VALUE);
			draft.setEnd(0);
		}
		return draft;
	}
	
	/**
	 * @param draft
	 * @param list
	 */
	private void putInstancesMap(ArrayList list, TraceLifelineDraft draft) {
		for (Iterator i = list.iterator(); i.hasNext(); ) {
			Object oo = (Object)i.next();
			if (oo instanceof String) {
				instancesMap.put(oo, draft);
			} else if (oo instanceof ArrayList) {
				putInstancesMap((ArrayList)oo, draft);
			} // else ignored
		}
	}

	/**
	 * @param draft
	 * @param list
	 */
	private String getCollapsedLifelineName(List list) {
		String ret = "["; //$NON-NLS-1$
		int j = 0;
		for (Iterator i = list.iterator(); i.hasNext(); j++) {
			Object oo = (Object)i.next();
			if (oo instanceof String) {
				if (j > 0) {
					ret += ","; //$NON-NLS-1$
				}
				String[] split = ((String)oo).split("#"); //$NON-NLS-1$
				ret += split.length == 2 ? split[1] : (String)oo;
			} else if (oo instanceof List) {
				if (j > 0) {
					ret += ","; //$NON-NLS-1$
				}
				ret += getCollapsedLifelineName((List)oo);
			} // else ignored
		}
		return ret + "]"; //$NON-NLS-1$
	}

	
	/**
	 * @param part
	 * @param selection
	 */
	private void externalDateSelectionChanged(IDateSelection selection,
											  TraceInteractionUpdate update) {
		if (frame == null) {
			return;
		}
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("External Date selection is "+selection); //$NON-NLS-1$
		}
		IDateSelection ids = (IDateSelection)selection;
		EObject eo = selection.getEObject();
		if (!(eo instanceof TRCProcess) || currentProcesses == null || !currentProcesses.contains((TRCProcess)eo)) {
			return;
		}
		TRCProcess process = (TRCProcess)eo;
		double dateSelection = ids.getStartDate(), absoluteDateSelection = dateSelection;
		double absoluteEntryTime = TraceInteractionUtils.getAbsoluteEntryTime(process.getAgent());
		if (absoluteDateSelection < absoluteEntryTime) {
			// determine if we received an absolute or a process relative date
			absoluteDateSelection += absoluteEntryTime;
		} else {
			dateSelection -= absoluteEntryTime;
		}
		
		// Is it in the current page? (this question is answered with date relative to the process)
		if (ids.getMeaning() == IDateSelection.NOW) {
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("dateSelection="+dateSelection+"; absoluteDateSelection="+absoluteDateSelection+"; getAbsoluteEntryTime(process)="+TraceInteractionUtils.getAbsoluteEntryTime(process)); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		    }
			ITimeRange bounds[] = new ITimeRange[2];
			if (frame.findDateBounds(absoluteDateSelection, bounds)) {
				if (bounds[0] instanceof GraphNode) {
					view.getSDWidget().ensureVisible((GraphNode)bounds[0]);
					view.getTimeCompressionBar().highlightRegion((BaseMessage)bounds[0], (BaseMessage)bounds[1]);
					if (TraceSDPlugin.debugEvents) {
						TraceSDPlugin.debugTraceEvents("Found a graph node"); //$NON-NLS-1$
					}
					return;
				}
				if (TraceSDPlugin.debugEvents) {
					TraceSDPlugin.debugTraceEvents("Found something else than a graph node: ignored"); //$NON-NLS-1$
				}
				frame.resetTimeCompression();
				return;
			}
			// How many pages in there?
			frame.resetTimeCompression();
			if (pages == 1) {
				ITimeRange sm = bounds[0] != null ? bounds[0] : bounds[1];
				if (sm != null && sm instanceof GraphNode) {
					view.getSDWidget().ensureVisible((GraphNode)sm);
					if (TraceSDPlugin.debugEvents) {
						TraceSDPlugin.debugTraceEvents(bounds[0] == null?"Before any graph node":"After any graph node"); //$NON-NLS-1$ //$NON-NLS-2$
					}
					return;
				}
				if (TraceSDPlugin.debugEvents) {
					TraceSDPlugin.debugTraceEvents("Found something else than a graph node (#2): ignored"); //$NON-NLS-1$
				}
				return;
			}
			// Look in previous pages
			if (tracePages != null &&
				tracePages.size() > 0 &&
				dateSelection <= ((TracePage)tracePages.get(tracePages.size()-1)).getLastTime()) {
				for (int i = 0; i < tracePages.size(); i++) {
					TracePage tp = (TracePage)tracePages.get(i);
					if (dateSelection >= tp.getFirstTime() &&
						dateSelection <= tp.getLastTime()) {
						update.setDateSelectionChanged(true);
						update.setExternalDateSelectionChanged(true);
						update.setDateToShow(absoluteDateSelection);
						update.setPageChanged(true);
						pageNumber = i;
						if (TraceSDPlugin.debugEvents) {
							TraceSDPlugin.debugTraceEvents("Found in a previous page"); //$NON-NLS-1$
						}
						return;
					}
				}
			}
			// Worst case: must look in all pages for an absolute date (no longer relative)
			update.setDateSelectionChanged(true);
			update.setExternalDateSelectionChanged(true);
			update.setDateToShow(absoluteDateSelection);
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("Looking in all pages"); //$NON-NLS-1$
			}
		}
	}

	protected Timer subscribeSelectionTimer;

	/**
	 * @param selection
	 */
	private void subscribeForDateSelectionChanged(final IDateSelection selection) {
		if (subscribeSelectionTimer != null) {
			subscribeSelectionTimer.cancel();
		}
		subscribeSelectionTimer = new Timer();
		subscribeSelectionTimer.schedule(new TimerTask() {
			public void run() {
				Display.getDefault().syncExec(new Runnable() {
					public void run() {
						TraceInteractionUpdate update = new TraceInteractionUpdate();
						externalDateSelectionChanged(selection, update);
			    		if (update.needsUpdate()) {
			        		updateSDBackground(update);
			    		}
					}
				});
				subscribeSelectionTimer = null;
			}
		}, 50);
	}
		
	/**
	 * @param update
	 * @param thread
	 * @return
	 */
	private boolean isThreadFiltered(TraceInteractionUpdate update, TRCThread thread) {
		if (update.getSelectedThreadList() != null) {
			return !update.getSelectedThreadList().contains(thread); // selected means it is not filtered
		} else {
			return false; // nothing is selected so nothing is filtered!
		}
	}
	
	private ArrayList selectedThreadList;

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#finishExternalExtendedSelection(org.eclipse.hyades.uml2sd.trace.loaders.internal.TraceInteractionUpdate)
	 */
	protected void finishExternalExtendedSelection(TraceInteractionUpdate update) {
		if (update.isThreadSelectionChanged()) {
			selectedThreadList = update.getSelectedThreadList(); // new threads selection received
		}
	}

	/**
	 * @param part
	 * @param selection
	 * @param update
	 */
	protected boolean externalExtendedSelectionChanged(IWorkbenchPart part,
										    		   Object selection,
													   TraceInteractionUpdate update) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("TraceInteractions: External extended selection from "+part+" is "+selection); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (selection instanceof IDateSelection) {
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("TraceInteractions: IDateSelection: "+((IDateSelection)selection).getStartDate()); //$NON-NLS-1$
			}
    		subscribeForDateSelectionChanged((IDateSelection)selection);
    		return true;
    	} else if (selection instanceof IEObjectSelection) {
			EObject eObject = ((IEObjectSelection)selection).getEObject();
			if (TraceSDPlugin.debugEvents) {
				TraceSDPlugin.debugTraceEvents("TraceInteractions: IEObjectSelection: "+eObject); //$NON-NLS-1$
			}
			if (eObject instanceof TRCThread) {
				update.setThreadSelectionChanged(true);
				update.addSelectedThread((TRCThread)eObject);
	    		return true;
			}
    		return false;
    	}
		return false;
	}


	/**
	 * Implementation of ISDGraphNodeSupporter
	 */
	public boolean isSyncMessageSupported() {
		return true;
	}
	
	/**
	 * implementation of ISDPagingProvider
	 */
	public boolean hasNextPage() {
		return hasNextPage;
	}
	public boolean hasPrevPage() {
		return pageNumber > 0;
	}
	public void nextPage() {
		pageNumber++;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(true);
		updateSDBackground(update);
	}
	public void prevPage() {
		pageNumber--;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * Implementation of ISDAdvancedPagingProvider
	 */
	public int maxItemsByPageCount() {
		return maximumMessagesByPage;
	}
	public int itemsTotalCount() {
		return totalNbMessages;
	}
	public String noItemsText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_NO_ITEMS"); //$NON-NLS-1$
	}
	public String oneItemText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_ONE_ITEM"); //$NON-NLS-1$
	}
	public String itemsText() {
		return TraceSDPlugin.getResourceString("PAGES_COUNT_SEVERAL_ITEMS"); //$NON-NLS-1$
	}
	public int currentPage() {
		return pageNumber;
	}
	public int pagesCount() {
		return pages;
	}
	public void pageSettingsChanged(int maxItemsByPage_, int pageNumber_) {
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setMaxMessagesByPageChanged(maximumMessagesByPage != maxItemsByPage_);
		maximumMessagesByPage = maxItemsByPage_;
		update.setPageChanged(pageNumber != pageNumber_);
		pageNumber = pageNumber_;
		if (update.isMaxMessagesByPageChanged() ||
			update.isPageChanged()) {
			updateSDBackground(update);
		}
	}
	public void pageNumberChanged(int pageNumber_) {
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setPageChanged(pageNumber != pageNumber_);
		pageNumber = pageNumber_;
		if (update.isPageChanged()) {
			updateSDBackground(update);
		}
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#notFoundYet(org.eclipse.hyades.uml2sd.ui.actions.widgets.Criteria)
	 */
	public boolean notFoundYet(Criteria toSearch) {
		// looking in other pages if any
		if (pages > 1 && pageNumber+1 < pages) {
			TraceInteractionUpdate update = new TraceInteractionUpdate();
			update.setFindRequired(true);
			update.setFindCriteria(toSearch);
			update.setFindStartingPage(pageNumber+1);
			updateSD(update);
			ArrayList findResults_ = update.getFindResults();
			if (findResults_ != null && !findResults_.isEmpty()) {
				findResults = findResults_;
				currentFindIndex = 0;
				GraphNode current = (GraphNode)findResults.get(currentFindIndex);
				view.getSDWidget().moveTo(current);
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Implementation of ISDInternalMesFilterProvider
	 */
	private static final String INTERNAL_MESSAGES = "internalMessages"; //$NON-NLS-1$
	
	public void setInternalMessageFiltered(boolean value) {
		if (internalMessagesFiltered != value) {
			internalMessagesFiltered = value;
			DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
			settings.put(INTERNAL_MESSAGES, internalMessagesFiltered);
			TraceInteractionUpdate update = new TraceInteractionUpdate();
			update.setInternalMessageFilteringChanged(true);
			updateSDBackground(update);
		}
	}
	
	public boolean getInternalMessageFiltered() {
		DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
		internalMessagesFiltered = settings.getBoolean(INTERNAL_MESSAGES); // defaults to false
		return internalMessagesFiltered;
	}

	private ArrayList currentFilters;

	public ArrayList getCurrentFilters() {
		return currentFilters;
	}
	
	private boolean isLifelineFiltered(String name) {
		if (currentFilters != null) {
			for (Iterator i = currentFilters.iterator(); i.hasNext(); ) {
				FilterCriteria tf = (FilterCriteria)i.next();
				if (tf.isActive() && tf.getCriteria().isLifeLineSelected()) {
					if (TraceInteractionUtils.matchCriteria(name, tf.getCriteria())) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	private boolean isSyncMessageFiltered(String name) {
		if (currentFilters != null) {
			for (Iterator i = currentFilters.iterator(); i.hasNext(); ) {
				FilterCriteria tf = (FilterCriteria)i.next();
				if (tf.isActive() && tf.getCriteria().isSyncMessageSelected()) {
					if (TraceInteractionUtils.matchCriteria(name, tf.getCriteria())) {
						return true;
					}
				}
			}
		}
		return false;
	}
	
	/**
	 * 
	 */
	public void filterSelectedGraphNodes(ArrayList graphNodes) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("filterGraphNodes("+(graphNodes!=null ? graphNodes.size() : 0)+" selected graph nodes)"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (currentFilters == null) {
			currentFilters = new ArrayList();
		}
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			Criteria c = new Criteria();
			if (gn instanceof IEObjectSelection) {
				EObject eo = (EObject)((IEObjectSelection)gn).getEObject();
				if (eo instanceof TRCFullMethodInvocation) {
					c.setExpression(((TRCFullMethodInvocation)eo).getMethod().getName());
				}
			}
			c.setCaseSenstiveSelected(true);
			if (gn instanceof TraceLifeline) {
				c.setExpression(gn.getName());
				c.setLifeLineSelected(true);
			} else if (gn instanceof TraceSyncMessage) {
				c.setExpression(((TRCFullMethodInvocation)((TraceSyncMessage)gn).getEObject()).getMethod().getName());
				c.setSyncMessageSelected(true);
			} else if (gn instanceof TraceSyncMessageReturn) {
				c.setExpression(((TRCFullMethodInvocation)((TraceSyncMessageReturn)gn).getEObject()).getMethod().getName());
				c.setSyncMessageSelected(true);
			}
			FilterListDialog.addGlobalFilter(new FilterCriteria(c, true));
			currentFilters = FilterListDialog.getGlobalFilters();
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setFilteringChanged(true);
		updateSDBackground(update);
	}

	/**
	 * Come back from the filters dialog
	 */
	public boolean filter(ArrayList filters) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("filter("+(filters!=null ? filters.size() : 0)+" filters)"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		currentFilters = filters;
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setFilteringChanged(true);
		updateSDBackground(update);
		return true;
	}

	/*
	 * Tree of collapsed lifelines: Horizontal collapsing
	 */
	private static final String HORIZONTAL_COLLAPSING = "horizontalCollapsing"; //$NON-NLS-1$
	private HashMap collapsedLifelines;

	/**
	 * 
	 */
	public void collapseLifelines(List graphNodes) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("collapseLifelines("+(graphNodes!=null ? graphNodes.size() : 0)+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		if (collapsedLifelines == null) {
			collapsedLifelines = new HashMap();
		}
		List list = new ArrayList();
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			Object key = null;
			if (gn instanceof TraceCollapsedLifelines) {
				key = ((TraceCollapsedLifelines)gn).getCollapsedLifelines();
			} else if (gn instanceof TraceLifeline) {
				key = getLifeLineId(((TraceLifeline)gn).getEObject());
			} else {
				continue; // Error ignored
			}
			list.add(key);
			collapsedLifelines.put(key, list);
		}
		saveCollapsedLifelines();
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setHorizontalCollapsingChanged(true);
		updateSDBackground(update);
	}

	/**
	 * 
	 */
	public void expandLifelines(List graphNodes) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("expandLifelines("+(graphNodes!=null ? graphNodes.size() : 0)+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			GraphNode gn = (GraphNode)i.next();
			if (gn instanceof TraceCollapsedLifelines) {
				ArrayList sublist = ((TraceCollapsedLifelines)gn).getCollapsedLifelines();
				for (Iterator j = sublist.iterator(); j.hasNext(); ) {
					Object key = j.next();
					collapsedLifelines.put(key, null);
				}
			} else  {
				continue; // Error ignored
			}
		}
		saveCollapsedLifelines();
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setHorizontalCollapsingChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * Implementation of ISDCollapseProvider
	 */
	public void collapseTwoLifelines(Lifeline lifeline1, Lifeline lifeline2) {
		List list = new ArrayList();
		// Reverse order for some usability reason
		list.add(lifeline2);
		list.add(lifeline1);
		collapseLifelines(list);
	}
	
	private String parseHorizontalCollapsingTree(Object o) {
		String result = null;
		if (o instanceof List) {
			List list = (List)o;
			result = "{"; //$NON-NLS-1$
			String prefix = ""; //$NON-NLS-1$
			for (Iterator i = list.iterator(); i.hasNext(); ) {
				Object u = i.next();
				String parsingResult = parseHorizontalCollapsingTree(u);
				if (parsingResult != null) {
					result += prefix+parsingResult;
					prefix = ","; //$NON-NLS-1$
				}
			}
			result += "}"; //$NON-NLS-1$
		} else if (o instanceof String) {
			result = (String)o;
		}
		return result;
	}

	private void saveCollapsedLifelines() {
		String result = ""; //$NON-NLS-1$
		if (collapsedLifelines != null) {
			Set entries = collapsedLifelines.entrySet();
			List uniqueTopLists = new ArrayList();
			for (Iterator i = entries.iterator(); i.hasNext(); ) {
				Map.Entry e = (Map.Entry)i.next();
				Object value = e.getValue();
				if (collapsedLifelines.get(value) == null &&
					!uniqueTopLists.contains(value)) {
					uniqueTopLists.add(value);
				}
			}
			String prefix = "{"; //$NON-NLS-1$
			for (Iterator i = uniqueTopLists.iterator(); i.hasNext(); ) {
				String parsingResult = parseHorizontalCollapsingTree(i.next());
				if (parsingResult != null) {
					result += prefix+parsingResult;
					prefix = ","; //$NON-NLS-1$
				}
			}
		}
		if (result.length() > 0) {
			result += "}"; //$NON-NLS-1$
		}
		DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
		settings.put(HORIZONTAL_COLLAPSING, result);
		TraceSDPlugin.debugTraceEvents("Saving "+HORIZONTAL_COLLAPSING+"="+result); //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	private List splitOneLevel(String input) {
		List list = new ArrayList();
		int level = 0, start = 1;
		int end = input.length()-1;
		for (int i = 1; i < end; i++) {
			char c = input.charAt(i);
			if (c == '{') {
				level++;
			} else if (c == '}') {
				level--;
			} else if ((c == ',' || i == end-1) && level == 0) {
				String member = input.substring(start, c==','?i:i+1);
				list.add(member);
				start = i+1;
			}
		}
		return list;
	}
	
	private Object parseHorizontalCollapsingString(String input, boolean deep) {
		if (input != null &&
			input.length() > 2 &&
			input.charAt(0) == '{' &&
			input.charAt(input.length()-1) == '}') {
			List members = splitOneLevel(input);
			if (members != null && members.size() > 0) {
				List list = new ArrayList();
				for (Iterator i = members.iterator(); i.hasNext(); ) {
					Object o = parseHorizontalCollapsingString((String)i.next(), true);
					if (o != null && deep) {
						list.add(o);
						collapsedLifelines.put(o, list);
					}
				}
				if (list.size() > 0){
					return list;
				}
			}
		} else if (input != null &&
				   input.length() > 0) {
			String parts[] = input.split("#"); //$NON-NLS-1$
			if (parts.length != 2 || TraceInteractionUtils.doesAgentProxyStillExists(parts[0])) {
				return input; // Already a string atom
			} else {
				return null;
			}
		}
		return null;
	}
	
	private HashMap loadCollapsedLifelines() {
		DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
		String result = settings.get(HORIZONTAL_COLLAPSING);
		collapsedLifelines = new HashMap();
		parseHorizontalCollapsingString(result, false/*first level is not a lifeline, just a list*/);
		return collapsedLifelines;
	}
	
	/**
	 * Set of collapsed messages: vertical collapsing
	 */
	private HashMap collapsedMessages;
	
	private static final String VERTICAL_COLLAPSING = "verticalCollapsing"; //$NON-NLS-1$
	
	private Object getMethodInvocationContainerKey(TRCFullMethodInvocation methodInvocation) {
		return methodInvocation.getProcess().getAgent().getAgentProxy().getRuntimeId();
	}
	
	private Object getMethodInvocationCollapsingId(TRCFullMethodInvocation methodInvocation) {
		return methodInvocation.getThread().getId()+":"+methodInvocation.getTicket(); //$NON-NLS-1$
	}
	
	private HashSet getContainerKeyHashSet(TRCFullMethodInvocation methodInvocation) {
		if (collapsedMessages == null) {
			return null;
		}
		Object object = collapsedMessages.get(getMethodInvocationContainerKey(methodInvocation));
		return object != null ? (HashSet)object : null;
	}

	private void saveCollapsedMessages() {
		String result = ""; //$NON-NLS-1$
		if (collapsedMessages != null) {
			Set keys = collapsedMessages.keySet();
			for (Iterator i = keys.iterator(); i.hasNext(); ) {
				String key = (String)i.next();
				HashSet containerSet = (HashSet)collapsedMessages.get(key);
				if (containerSet.size() > 0) {
					result += key+"="; //$NON-NLS-1$
					String value;
					int n = 0;
					for (Iterator j = containerSet.iterator(); j.hasNext(); ) {
						value = (String)j.next();
						TraceSDPlugin.debugTraceEvents("Save: key="+key+" ticket="+value); //$NON-NLS-1$ //$NON-NLS-2$
						result += value+(j.hasNext()?";":""); //$NON-NLS-1$ //$NON-NLS-2$
					}
					result += "]"; //$NON-NLS-1$
				}
			}
		}
		DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
		settings.put(VERTICAL_COLLAPSING, result);
	}

	private HashMap loadCollapsedMessages() {
		DialogSettings settings = (DialogSettings)TraceSDPlugin.getDefault().getDialogSettings();
		HashMap ret = null;
		String result = settings.get(VERTICAL_COLLAPSING);
		if (result != null && result.length() > 1) {
			String agents[] = result.split("]"); //$NON-NLS-1$
			for (int i = 0; i < agents.length; i++) {
				String parts[] = agents[i].split("="); //$NON-NLS-1$
				if (TraceInteractionUtils.doesAgentProxyStillExists(parts[0])) {
					HashSet containerSet;
					String values[] = parts[1].split(";"); //$NON-NLS-1$
					if (values.length > 0) {
						if (ret == null) {
							ret = new HashMap();
						}
						ret.put(parts[0], containerSet = new HashSet());
						for (int j = 0; j < values.length; j++) {
							containerSet.add(values[j]);
							TraceSDPlugin.debugTraceEvents("Load: key="+parts[0]+" ticket="+values[j]); //$NON-NLS-1$ //$NON-NLS-2$
						}
					}
				} else {
					TraceSDPlugin.debugTraceEvents("Agent proxy "+parts[0]+" does no longer exist"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		return ret;
	}

	/**
	 * @param methodInvocation
	 * @return
	 */
	private boolean isMessageCollapsed(TRCFullMethodInvocation methodInvocation) {
		if (methodInvocation == null || collapsedMessages == null) {
			return false;
		}
		HashSet containerSet = getContainerKeyHashSet(methodInvocation);
		if (containerSet != null &&
			containerSet.contains(getMethodInvocationCollapsingId(methodInvocation))) {
			return true;
		}
		for (TRCFullMethodInvocation invokedBy = (TRCFullMethodInvocation)methodInvocation.getInvokedBy();
			 invokedBy != null;
			 invokedBy = (TRCFullMethodInvocation)invokedBy.getInvokedBy()) {
			containerSet = getContainerKeyHashSet(invokedBy);
			if (containerSet != null &&
				containerSet.contains(getMethodInvocationCollapsingId(invokedBy))) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 
	 */
	public void collapseCalledMessages(ArrayList graphNodes) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("collapseCalledMessages("+(graphNodes!=null ? graphNodes.size() : 0)+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			Object gn = i.next();
			TRCFullMethodInvocation methodInvocation = null;
			if (gn instanceof TraceSyncMessage) {
				TraceSyncMessage sm = (TraceSyncMessage)gn;
				methodInvocation = (TRCFullMethodInvocation)sm.getEObject();
			} else if (gn instanceof TraceSyncMessageReturn) {
				TraceSyncMessageReturn smr = (TraceSyncMessageReturn)gn;
				methodInvocation = (TRCFullMethodInvocation)smr.getEObject();
			} // else ignored
			collapseCalledMessage(methodInvocation);
		}
		saveCollapsedMessages();
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setVerticalCollapsingChanged(true);
		updateSDBackground(update);
	}
	
	/**
	 * @param methodInvocation
	 */
	private void collapseCalledMessage(TRCFullMethodInvocation methodInvocation) {
		if (methodInvocation != null) {
			if (collapsedMessages == null) {
				collapsedMessages = new HashMap();
			}
			HashSet containerSet = getContainerKeyHashSet(methodInvocation);
			if (containerSet == null) {
				collapsedMessages.put(getMethodInvocationContainerKey(methodInvocation), containerSet = new HashSet());
			}
			containerSet.add(getMethodInvocationCollapsingId(methodInvocation));
		}
	}

	/**
	 * 
	 */
	public void expandCalledMessages(ArrayList graphNodes) {
		if (TraceSDPlugin.debugEvents) {
			TraceSDPlugin.debugTraceEvents("expandCalledMessages("+(graphNodes!=null ? graphNodes.size() : 0)+")"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		for (Iterator i = graphNodes.iterator(); i.hasNext(); ) {
			Object gn = i.next();
			TRCFullMethodInvocation methodInvocation = null;
			if (gn instanceof TraceSyncMessage) {
				TraceSyncMessage sm = (TraceSyncMessage)gn;
				methodInvocation = (TRCFullMethodInvocation)sm.getEObject();
			} else if (gn instanceof TraceSyncMessageReturn) {
				TraceSyncMessageReturn smr = (TraceSyncMessageReturn)gn;
				methodInvocation = (TRCFullMethodInvocation)smr.getEObject();
			} // else ignored
			if (methodInvocation != null) {
				HashSet containerSet = getContainerKeyHashSet(methodInvocation);
				if (containerSet == null) {
					continue; // strange, but...
				}
				containerSet.remove(getMethodInvocationCollapsingId(methodInvocation));
			}
		}
		saveCollapsedMessages();
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setVerticalCollapsingChanged(true);
		updateSDBackground(update);
	}

	/**
	 * Implementation of ITraceInteractionPreferenceListener
	 * @param event
	 */
	public void applyPreferences() {
		int max = (int)TraceSDPlugin.getDefault().getPreferenceStore().
							getLong(ITraceInteractionPreferenceListener.PAGE_SIZE);
		if (maximumMessagesByPage == max) {
			return;
		}
		TraceInteractionUpdate update = new TraceInteractionUpdate();
		update.setMaxMessagesByPageChanged(true);
		maximumMessagesByPage = max;
		updateSDBackground(update);
	}

	
	private static TraceInteractions instance;

	/**
	 * returns the firstly created instance of TraceInteractions
	 */
	public static TraceInteractions getInstance() {
		if (instance == null) {
			instance = new TraceInteractions();
		}
		return instance;
	}
	
	private ILifelineLoader lifelineLoader;

	/**
	 * @param lifelineLoader The lifelineLoader to set.
	 */
	public void setLifelineLoader(ILifelineLoader lifelineLoader_) {
		lifelineLoader = lifelineLoader_;
	}

	/**
	 * @return Returns the lifelineLoader.
	 */
	public ILifelineLoader getLifelineLoader() {
		return lifelineLoader;
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#getFinalTitleString()
	 */
	public String getFinalTitleString() {
		if (lifelineLoader != null) {
			return lifelineLoader.getFinalTitleString();
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#getLifelineEObjectFromMethodInvocation(org.eclipse.hyades.models.trace.TRCFullMethodInvocation)
	 */
	protected EObject getLifelineEObjectFromMethodInvocation(TRCFullMethodInvocation mi) {
		if (lifelineLoader != null) {
			return lifelineLoader.getLifelineEObjectFromMethodInvocation(mi);
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#getLifeLineTitle(org.eclipse.emf.ecore.EObject, boolean)
	 */
	protected String getLifeLineTitle(EObject to, boolean long_) {
		if (lifelineLoader != null) {
			return lifelineLoader.getLifeLineTitle(to, long_);
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#getLifeLineTitle(org.eclipse.emf.ecore.EObject, boolean)
	 */
	protected String getLifeLineId(EObject to) {
		if (lifelineLoader != null) {
			return lifelineLoader.getLifeLineId(to);
		}
		return null;
	}

	/**
	 * @see org.eclipse.hyades.uml2sd.trace.loaders.BaseTraceInteractions#getLifeLineCategory(org.eclipse.emf.ecore.EObject)
	 */
	protected int getLifeLineCategory(EObject to) {
		if (lifelineLoader != null) {
			return lifelineLoader.getLifeLineCategory(to);
		}
		return 0;
	}

}
