/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.CoverageStatisticPage;
import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticViewer;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.ColumnExtensionValue;
import org.eclipse.hyades.trace.views.util.internal.Coverage;
import org.eclipse.hyades.trace.views.util.internal.CoverageAnalysis;
import org.eclipse.hyades.trace.views.util.internal.CoverageFolder;
import org.eclipse.hyades.trace.views.util.internal.IColumnExtension;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

/*
 * This view contains the coverage statistics data.  This data concerns itself
 * with which methods were called or not.  It has the package, class, and
 * method levels.
 * 
 * It was modified from earlier to extend a MultiLevelStatisticMethodView base
 * class that other views with similar properties extend.
 * The functionality is more or less the same as it was before.  One change is that
 * each level is now stored in a preference string.  So reloading levels preserves
 * the previous settings.
 * 
 * Concerning the Calls functionality.  In checking to see if the calls column
 * is meaningfull.  The current solution is that we check to see if calls column
 * is meaningfull, and if it isn't we fill the Calls cells blank.  So they don't
 * contain non-meaningfull data.
 */
public class CoverageStatisticView extends MultiLevelStatisticMethodView{
	 
	/**
	 * Use this TRCOption key with value "true" to say that calls column is useless.
	 * It won't appear if this flag is found in at least one of the TRCConfiguration of the runs belonging
	 * to the selected node in the profiling monitor. 
	 */
	public static final String CALLS_NOT_COMPUTED = "COVERAGE_STATISTICS.CALLS_NOT_COMPUTED";
	
	private int _numberOfColumns = 7;
	
	private static int _level = DEFAULT_LEVEL;		
	
	private boolean _isCallsMeaningfull;
	
	public CoverageStatisticView(Composite parent, TraceViewerPage page){
		super(parent, page);
		_isCallsMeaningfull = true;
	}
	
	public class CoverageStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		public Object getParent(Object element) {
			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
			_isCallsMeaningfull = isCallsMeaningful();
		}

		/**
		 * return the children of the coverage element
		 */
		public Object[] getChildren(Object element) {
			ArrayList childs = ((Coverage)element).getChilds();
			if (childs == null) {
				return null;
			}
			return childs.toArray();
		}

		/**
		 * test if the coverage element has children
		 */
		public boolean hasChildren(Object element) {
			return (getChildren(element) != null);
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {

			ArrayList tmpList = new ArrayList();
			tmpList.clear();

			CoverageAnalysis covAnalysis =
				((CoverageStatisticPage) _page).getCovAnalysis();

			if (covAnalysis.getRoot() != null) {
				switch (getLevel()) {
					case 1 : // Package level
						tmpList.add(covAnalysis.getRoot());
						return tmpList.toArray();

					case 2 : // Class level
						tmpList.add(covAnalysis.getClassLevel());
						return tmpList.toArray();

					case 3 : // Method level
						tmpList.add(covAnalysis.getMethodLevel());
						return tmpList.toArray();

					default :
						return tmpList.toArray();
				}

			}
			/* When we "relaunch" a process the active page have no content until we make
				a "refreh views" */
			else { //No elements to display
				return tmpList.toArray();
			}
		}
	}
	
	public class CoverageStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public CoverageStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		/**
		 *  return for the coverage element to display the associated image 
		 */
		public Image getColumnImage(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTable().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {

				case 1 : //Class names column
					if (cov.getTraceObj() instanceof TRCMethod)
						return TracePluginImages.getImage(
							TracePluginImages.IMG_CLASS_COL);
					else
						return null;

				case 2 : // Package column

					if (cov.getTraceObj() instanceof TRCMethod) {
						if (getLevel() == PACKAGE_LEVEL || getLevel() == CLASS_LEVEL)
							return null;
						else
							return TracePluginImages.getImage(
								TracePluginImages.IMG_PACKAGE_COL);
					} else if (cov.getTraceObj() instanceof TRCClass)
						return TracePluginImages.getImage(
							TracePluginImages.IMG_PACKAGE_COL);
					else
						return null;
			}
			return null;
		}
		
		
		public String getColumnText(Object obj, int col) {

			StatisticTableColumnInfo info =
				StatisticTableColumnInfo.getStatisticTableColumnInfo(
					_viewer.getTable().getColumn(col));
			int pos = info.getColumnData().getInitalPos();
			Coverage cov = (Coverage) obj;

			switch (pos) {
				case 0 : // packages / classes and methods names
					if (cov.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov.getTraceObj())
							.getName()
							.equals(""))
							return TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							return ((TRCPackage) cov.getTraceObj()).getName();
					} else if (cov.getTraceObj() instanceof TRCClass) {
						return ((TRCClass) cov.getTraceObj()).getName();
					}
					if (cov.getTraceObj() instanceof TRCMethod) {
						TRCMethod m = (TRCMethod) cov.getTraceObj();
						return m.getName() + m.getSignature();
					} else {
						return TraceUIPlugin.getString("ST_SUMMARY");
					}

				case 1 : // class names
					if (cov.getTraceObj() instanceof TRCMethod)
						return ((TRCMethod) cov.getTraceObj())
							.getDefiningClass()
							.getName();
					else
						return "";

				case 2 : // package
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (getLevel() == 1 || getLevel() == 2)
							return "";
						else {
							if ((((TRCMethod) cov.getTraceObj())
								.getDefiningClass()
								.getPackage()
								.getName())
								.equals(""))
								return TraceUIPlugin.getString(
									"DEFAULT_PACKAGE");
							else
								return ((TRCMethod) cov.getTraceObj())
									.getDefiningClass()
									.getPackage()
									.getName();
						}
					} else if (cov.getTraceObj() instanceof TRCClass) {
						if ((((TRCClass) cov.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							return TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							return ((TRCClass) cov.getTraceObj())
								.getPackage()
								.getName();
					} else
						return "";

				case 3 : //Calls
					if (_isCallsMeaningfull)
						return String.valueOf(cov.getCalls());
					else
						return "";

				case 4 : //Method missed
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (cov.getCalls() == 0)
							return TraceUIPlugin.getString("ST_MISSED");

						else
							return "";
					} else {
						return String.valueOf(cov.getMethodsMissed());
					}

				case 5 : //method hits
					if (cov.getTraceObj() instanceof TRCMethod) {
						if (cov.getCalls() != 0)
							return TraceUIPlugin.getString("ST_HIT");
						else
							return "";
					} else {
						return String.valueOf(cov.getMethodsHit());
					}

				case 6 : //%Method hit
					if (cov.getTraceObj() instanceof TRCMethod)
						return "";
					else
						return TString.formatAsPercentage(
								cov.getPercentage());

			}
			//if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(
						pos - _numberOfColumns);
				return (
					(IColumnExtension)
						(elt.getClassOfColumnExtension())).getColumnText(
					obj,
					elt.getInitialColumnPos());
			}

			return "";
		}
	}

	public class CoverageStatisticSorter extends MultiLevelStatisticSorter{
		public int compare(Viewer viewer, Object e1, Object e2) {

			Coverage cov1 = (Coverage) e1;
			Coverage cov2 = (Coverage) e2;

			String name1 = null;
			String name2 = null;

			switch (_pos) {
				case 0 : //item names
					if (cov1.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov1.getTraceObj())
							.getName()
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 = ((TRCPackage) cov1.getTraceObj()).getName();
					} else if (cov1.getTraceObj() instanceof TRCClass)
						name1 = ((TRCClass) cov1.getTraceObj()).getName();
					else if (cov1.getTraceObj() instanceof TRCMethod)
						name1 = ((TRCMethod) cov1.getTraceObj()).getName();

					if (cov2.getTraceObj() instanceof TRCPackage) {
						if (((TRCPackage) cov2.getTraceObj())
							.getName()
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 = ((TRCPackage) cov2.getTraceObj()).getName();
					} else if (cov2.getTraceObj() instanceof TRCClass)
						name2 = ((TRCClass) cov2.getTraceObj()).getName();
					else if (cov2.getTraceObj() instanceof TRCMethod)
						name2 = ((TRCMethod) cov2.getTraceObj()).getName();

					return (int) _sortSequence
						* (int) name1.compareToIgnoreCase(name2);

				case 1 : //Class names

					if ((cov1.getTraceObj() instanceof TRCMethod)
						&& (cov2.getTraceObj() instanceof TRCMethod)) {
						TRCMethod m1 = (TRCMethod) cov1.getTraceObj();
						TRCMethod m2 = (TRCMethod) cov2.getTraceObj();

						return (int) _sortSequence
							* (int) (m1
								.getDefiningClass()
								.getName()
								.compareToIgnoreCase(
									m2.getDefiningClass().getName()));
					} else
						return 0;

				case 2 : //Package
					if ((cov1.getTraceObj() instanceof TRCMethod)
						&& (cov2.getTraceObj() instanceof TRCMethod)
						&& (getLevel() == 3)) {
						TRCMethod m1 = (TRCMethod) cov1.getTraceObj();
						TRCMethod m2 = (TRCMethod) cov2.getTraceObj();

						if ((m1.getDefiningClass().getPackage().getName())
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 =
								m1.getDefiningClass().getPackage().getName();

						if ((m2.getDefiningClass().getPackage().getName())
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 =
								m2.getDefiningClass().getPackage().getName();

						return (int) _sortSequence
							* (int) (name1.compareToIgnoreCase(name2));

					} else if (
						(cov1.getTraceObj() instanceof TRCClass)
							&& (cov2.getTraceObj() instanceof TRCClass)) {
						if ((((TRCClass) cov1.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							name1 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name1 =
								((TRCClass) cov1.getTraceObj())
									.getPackage()
									.getName();

						if ((((TRCClass) cov2.getTraceObj())
							.getPackage()
							.getName())
							.equals(""))
							name2 = TraceUIPlugin.getString("DEFAULT_PACKAGE");
						else
							name2 =
								((TRCClass) cov2.getTraceObj())
									.getPackage()
									.getName();

						return (int) _sortSequence
							* (int) (name1.compareToIgnoreCase(name2));
					} else
						return 0;

				case 3 : //Calls
					if (_isCallsMeaningfull)
						return (int) _sortSequence
							* (cov1.getCalls() - cov2.getCalls());
					else
						return 0;

				case 4 : //Methods missed
					return (int) _sortSequence
						* (cov1.getMethodsMissed() - cov2.getMethodsMissed());

				case 5 : //Methods hit
					return (int) _sortSequence
						* (cov1.getMethodsHit() - cov2.getMethodsHit());

				case 6 : //% methods hit
					return (int) _sortSequence
						* (int) ((cov1.getPercentage() - cov2.getPercentage()) * 10000);

			}
			//	 if column extended
			if (_listOfColumExtension.size() != 0) {
				ColumnExtensionValue elt =
					(ColumnExtensionValue) _listOfColumExtension.get(_pos-_numberOfColumns);
				return (int) _sortSequence * ((IColumnExtension)
							(elt.getClassOfColumnExtension())).compare(
								elt.getInitialColumnPos(), e1, e2);
			}

			return 0;
		}
	}

	public class CoverageStatisticFilter extends StatisticFilter {

		public CoverageStatisticFilter() {
			super();
		}

		/**
		 * filter the elements according to the pattern
		 */
		public boolean select(Viewer viewer, Object parent, Object element) {

			Coverage cov = (Coverage) element;
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(cov.getTraceObj() instanceof TRCPackage))
						return true;
					if (((TRCPackage) cov.getTraceObj()).getName().equals(""))
						compareText =
							TraceUIPlugin.getString("DEFAULT_PACKAGE");
					else
						compareText =
							((TRCPackage) cov.getTraceObj()).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(cov.getTraceObj() instanceof TRCClass))
						return true;
					compareText = ((TRCClass) cov.getTraceObj()).getName();
					break;

				case METHOD_LEVEL : //method level
					if (!(cov.getTraceObj() instanceof TRCMethod))
						return true;
					compareText = ((TRCMethod) cov.getTraceObj()).getName();
					break;
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
	}
	
	public boolean isEmpty()
	{
		CoverageAnalysis covAnalysis =
			((CoverageStatisticPage) _page).getCovAnalysis();

		return (covAnalysis.getRoot() == null ||
				covAnalysis.getRoot().getMethods() == 0);		
	}
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":221,"
		+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
		+ ":1:0"
		+ ":100,"
		+ TraceUIPlugin.getString("STR_COL_PACKAGE")
		+ ":2:0"
		+ ":100,"
		+ TraceUIPlugin.getString("STR_COL_CALLS")
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120,"
		+ TraceUIPlugin.getString("STR_COL_RATE")
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:120";
	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
		+ ":1:0"
		+ ":100,"
		+ TraceUIPlugin.getString("STR_COL_PACKAGE")
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":150,"
		+ TraceUIPlugin.getString("STR_COL_CALLS")
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:60,"		
		+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100,"
		+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100,"
		+ TraceUIPlugin.getString("STR_COL_RATE")
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:100";		
	}
	
	protected String getDefaultColumnsTemplateMethodLevel()
	{
		return TraceUIPlugin.getString("STR_COL_ITEM_NAMES")
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ TraceUIPlugin.getString("STR_COL_CLASS_NAME")
		+ ":1:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":100,"
		+ TraceUIPlugin.getString("STR_COL_PACKAGE")
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":100,"
		+ TraceUIPlugin.getString("STR_COL_CALLS")
		+ ":3:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:45,"		
		+ TraceUIPlugin.getString("STR_COL_METHOD_MISSED")
		+ ":4:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90,"
		+ TraceUIPlugin.getString("STR_COL_METHOD_HIT")
		+ ":5:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90,"
		+ TraceUIPlugin.getString("STR_COL_RATE")
		+ ":6:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":right:90";
	}
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
			columns = "";
		
		return columns + getColumnsTemplateExtensions();
	}	

	public String getColumnsTemplateExtensions()
	{
		String columnName = "";
		String pos = "";
		String style = "";
		String width = "";
		String align = "";
		int colNumber = _numberOfColumns;
		
		_templateOfColumExtension = "";
		_listOfColumExtension.clear();

		IExtensionPoint extPoint =
		Platform.getPluginRegistry().getExtensionPoint(
		"org.eclipse.hyades.trace.views.columnExtension");

		if (extPoint != null) {
			IConfigurationElement[] members =
			extPoint.getConfigurationElements();

			if (members.length != 0) {
				//For each column
				for (int i = 0; i < members.length; i++) {
					columnName = members[i].getAttribute("name");
					pos = members[i].getAttribute("pos");
					style =	members[i].getAttribute("style");
					width = members[i].getAttribute("width");
					align = members[i].getAttribute("align");
					_templateOfColumExtension +=
					","
					+ columnName
					+ ":"
					+ colNumber
					+ ":"
					+ style
					+ ":"
					+ align
					+ ":"
					+ width;
					colNumber++;

					try {
						ColumnExtensionValue columnVal = new ColumnExtensionValue();
						columnVal.setClassOfColumnExtension(
							(IColumnExtension) members[i].createExecutableExtension("class"));
						columnVal.setInitialColumnPos((new Integer(pos)).intValue());
						_listOfColumExtension.add(columnVal);
					} catch (CoreException e) {
						System.err.println(e);
					}
				}
			}
		}
		return _templateOfColumExtension;
	}
	
	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());
		
		if (getLevel() == CoverageStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsPack60";
		else if (getLevel() == CoverageStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsClass60";
		else if (getLevel() == CoverageStatisticView.METHOD_LEVEL)
			return PREFERENCE_KEY_PREFIX + "CovStatsMethod60";
		
		return "";
	}
	
	private boolean isCallsMeaningful() {
		CoverageAnalysis covAnalysis = ((CoverageStatisticPage) _page).getCovAnalysis();
		CoverageFolder root = covAnalysis.getRoot();
		if (root == null) {
			return true;
		}
		ArrayList traceObjects = root.getTraceObjects();
		for (Iterator i = traceObjects.iterator(); i.hasNext(); ) {
			Object o = i.next();
			if (o instanceof TRCAgentProxy) {
				TRCAgentProxy agentProxy = (TRCAgentProxy)o;
				if (!isCallsMeaningful(agentProxy)) {
					return false;
				}
			} else if (o instanceof TRCNode) {
				TRCNode node = (TRCNode)o;
				if (!isCallsMeaningful(node)) {
					return false;
				}
			} else if (o instanceof TRCMonitor) {
				TRCMonitor monitor = (TRCMonitor)o;
				if (!isCallsMeaningful(monitor)) {
					return false;
				}
			}
		}
		return true;
	}
	
	private boolean isCallsMeaningful(TRCMonitor monitor) {
		EList nodes = monitor.getNodes();
		for (Iterator j = nodes.iterator(); j.hasNext(); ) {
			TRCNode node = (TRCNode)j.next();
			if (!isCallsMeaningful(node)) {
				return false;
			}
		}
		return true;
	}

	private boolean isCallsMeaningful(TRCNode node) {
		EList processProxies = node.getProcessProxies();
		for (Iterator j = processProxies.iterator(); j.hasNext(); ) {
			TRCProcessProxy processProxy = (TRCProcessProxy)j.next();
			EList agentProxies = processProxy.getAgentProxies();
			for (Iterator k = agentProxies.iterator(); k.hasNext(); ) {
				TRCAgentProxy agentProxy = (TRCAgentProxy)k.next();
				if (!isCallsMeaningful(agentProxy)) {
					return false;
				}
			}
		}
		return true;
	}

	private boolean isCallsMeaningful(TRCAgentProxy agentProxy) {
		EList configs = agentProxy.getConfigurations();
		for (Iterator j = configs.iterator(); j.hasNext(); ) {
			TRCConfiguration config = (TRCConfiguration)j.next();
			EList options = config.getOptions();
			for (Iterator k = options.iterator(); k.hasNext(); ) {
				TRCOption option = (TRCOption)k.next();
				if (CALLS_NOT_COMPUTED.equals(option.getKey()) &&
					option.getValue().equalsIgnoreCase("true")) {
					return false;
				}
			}
		}
		return true;
	}

	public IContentProvider getContentProvider()
	{
		return new CoverageStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new CoverageStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new CoverageStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new CoverageStatisticFilter();
	}	
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.COVERAGE_STATS_VIEW;
	}	
	
	/**
	 * Update the "model selection" when an element is selected in the table
	 */
	public void updateModelSelection() {
	
		ISelection selection = getTableViewer().getSelection();
		if (selection != null && !selection.isEmpty()) {
			Object sel = ((IStructuredSelection) selection).getFirstElement();
			EObject obj = ((Coverage)sel).getTraceObj();
			
			notifyViewSelectionChanged(this,obj);
		}
	}
	
	/**
	 * expand the fisrt element of the table which is "Summary"
	 */
	protected void expandFirstElement() {
		TableTree table = ((TableTreeViewer) getTableViewer()).getTableTree();
		TableTreeItem[] items = table.getItems();

		if (items.length != 0)
			 ((MultiLevelStatisticTreeViewer) getTableViewer()).expandItem(items[0]);
	}	
	
	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {
		super.menuAboutToShow(menu);
		menu.add(fSeparator);		
		menu.add(((MultiLevelStatisticViewer)getTraceViewerPage().getTraceViewer()).openSource());
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}	
	
	protected Object getItemModelData(TableTreeItem item)
	{
		if (item.getData()!= null)
			return ((Coverage)item.getData()).getTraceObj();
		else
			return null;
	}	

	protected TableTreeItem getItemsRoot(TableTree table)
	{
		if (table.getItemCount() > 0)
			return table.getItems()[0];
		else
			return null;
	}
}
