/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCHeapObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.models.trace.impl.TRCClassImpl;
import org.eclipse.hyades.models.trace.impl.TRCMethodImpl;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;
import org.eclipse.hyades.trace.ui.TraceViewerPage;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.views.adapter.internal.ExecutionStatisticPage2;
import org.eclipse.hyades.trace.views.util.internal.ColumnData;
import org.eclipse.hyades.trace.views.util.internal.PerftraceUtil;
import org.eclipse.hyades.trace.views.util.internal.StatisticTableColumnInfo;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;

/*
 * This view contains execution statistics data.  That is time related data for
 * the package level, class level, method level, and instance level.
 */
public class ExecutionStatisticView extends MultiLevelStatisticMethodView{
	
	private static int _level = DEFAULT_LEVEL;
	
	public static final int INSTANCE_LEVEL = 4;	
	
	private final TRCPackageImpl.TRCPackageSnapshot pdelta1 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCPackageImpl.TRCPackageSnapshot pdelta2 = new TRCPackageImpl.TRCPackageSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta1 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCClassImpl.TRCClassSnapshot cdelta2 = new TRCClassImpl.TRCClassSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot mdelta1 = new TRCMethodImpl.TRCMethodSnapshot();
	private final TRCMethodImpl.TRCMethodSnapshot mdelta2 = new TRCMethodImpl.TRCMethodSnapshot();	
	
	public ExecutionStatisticView(Composite parent, TraceViewerPage page){
		super(parent, page);
	}
	
	protected int getClassPosForLevel(int pos)
	{
        if (getLevel()== ExecutionStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;
           		case 4:
           			return 5;
           		case 5:
           			return 6;
           		case 6:
           			return 7;
           		case 7:
           			return 8;
           		case 8:
           			return 9;           			
			}
        }
        else if (getLevel()== ExecutionStatisticView.INSTANCE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 2;
           		case 3:
           			return 3;
           		case 4:
           			return 6;
           		case 5:
           			return 8;
			}
        }
        return pos;
	}
	
	protected int getMethodPosForLevel(int pos)
	{
        if (getLevel()== ExecutionStatisticView.PACKAGE_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 4;
           		case 3:
           			return 5;
           		case 4:
           			return -1;
           		case 5:
           			return 6;
           		case 6:
           			return -1;
           		case 7:
           			return 7;
           		case 8:
           			return -1;           			
			}
        }
        else if (getLevel()== ExecutionStatisticView.CLASS_LEVEL)
        {
           	switch (pos)
			{
           		case 2:
           			return 3;
           		case 3:
           			return 4;           			
           		case 4:
           			return 5;
           		case 5:
           			return -1;
           		case 6:
           			return 6;
           		case 7:
           			return -1;
           		case 8:
           			return 7;
           		case 9:
           			return -1;           			
			}
        }       
        return pos;
	}		
	
	public class ExecutionStatisticContentProvider extends MultiLevelStatisticContentProvider
	{
		public Object[] getChildren(Object element)
		{
			tmpList.clear();
			
			if(element instanceof TRCPackage)
				return ((TRCPackage)element).getClasses().toArray();
			else if (element instanceof TRCClass)
			{
				if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				{
					TRCClass cls = (TRCClass)element;
					List list = cls.getObjects();

					/*
					 * #48680: Use only TRCFullTraceObjects if they
					 * are present. 
					 */
					Iterator iter = cls.getObjects().iterator();
					while (iter.hasNext()) {
						Object obj = iter.next();
						if (obj instanceof TRCFullTraceObject) {
							tmpList.add(obj);
						}
					}
					
					/*
					 * #48680: If there are no TRCFullTraceObjects,
					 * then include the TRCHeapObjects.
					 */
					if (tmpList.isEmpty()) {
						tmpList.addAll(list);
					}
					
					return tmpList.toArray();
				}
				else
				{
					return ((TRCClass) element).getMethods().toArray();
				}
			}
			else
				return tmpList.toArray(); 	
		}
		
		public Object getParent(Object element) {
			
		/*	if(element instanceof TRCClass && getLevel() != ExecutionStatisticView.CLASS_LEVEL)
				return ((TRCClass)element).getPackage();
			else if(element instanceof TRCMethod && getLevel() != ExecutionStatisticView.METHOD_LEVEL)
				return ((TRCMethod)element).getDefiningClass();
			else
				return null;*/
			return null;
		}		

		public boolean hasChildren(Object element) {
			if (element instanceof TRCPackage)
				return true;
			else if (element instanceof TRCClass)
			{
				if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				{
					TRCClass cls = (TRCClass)element;
				    return (cls.getObjects().size() > 0);
				}
				else
				{
					return ((TRCClass) element).getMethods().size() > 0;
				}
			}

			return false;
		}
		
		/**
		 * return the list of elements to display in the table tree
		 */
		public Object[] getElements(Object inputElement) {

			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
			{
				Object[] packages = PerftraceUtil.getAllPackages(_page.getMOFObject());
				ArrayList list = new ArrayList();
				for(int idx=0; idx<packages.length; idx++)
				{
					TRCPackage pack = (TRCPackage)packages[idx];
					if(pack.getClasses().size() > 0)
					   list.add(pack);
				}
			
				return list.toArray();
			}
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
			{
				return PerftraceUtil.getAllClasses(_page.getMOFObject());
			}
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
			{
				Object[] classes =  PerftraceUtil.getAllClasses(_page.getMOFObject());
				ArrayList methods = new ArrayList();
				
				for(int idx=0; idx<classes.length; idx++)
				{
					methods.addAll(((TRCClass)classes[idx]).getMethods());
				}
				
				return methods.toArray();
			}
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			{
				return PerftraceUtil.getAllClasses(_page.getMOFObject());
			}
			else
			{
				return new ArrayList().toArray();
			}
		}
	}
	
	public class ExecutionStatisticLabelProvider extends MultiLevelStatisticLabelProvider
	{
		public ExecutionStatisticLabelProvider(StatisticView viewer) {
			super(viewer);
		}
		
		protected String getColumnTextPackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage) obj;
				TRCPackageImpl.TRCPackageSnapshot ps = (TRCPackageImpl.TRCPackageSnapshot)p.retrieveSnapshot();
				
				switch(pos)
				{
					case 0://package name
						return PerftraceUtil.getPackageName(p, _page.getMOFObject()).toString();
					case 1:
						return "";
					case 2: //base time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getBaseTime()/_maxTime);
						  }
						  else
						  {				
							  return TString.formatTimeValue(ps.getBaseTime());						  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);
						 return TString.formatTimeValue(pdelta1.getBaseTime());						  					  	
					  }
					  
					case 3: //average base time
					{
					   int calls = ps.getCalls();
					   if(calls == 0)
					      calls = 1;
					      
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage(ps.getBaseTime() / (calls*_maxTime));
					  }
					  else
					  {													  					
						return TString.formatTimeValue(ps.getBaseTime() / calls);
					  }
					}					  
					case 4: //inherited base time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {				
							  return TString.formatTimeValue(ps.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						 return TString.formatTimeValue(pdelta1.getInheritedBaseTime());						  					  	
					  }
					  
					  					  
					case 5: //cumulative time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getCumulativeTime()/_maxTime);
						  }
						  else
						  {									
							  return TString.formatTimeValue(ps.getCumulativeTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						 return TString.formatTimeValue(pdelta1.getCumulativeTime());						  					  	
					  }
					  
					  
					case 6: //inherited cumulative time
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(ps.getInheritedCumulativeTime()/_maxTime);
						  }
						  else
						  {									
							  return TString.formatTimeValue(ps.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						 return TString.formatTimeValue(pdelta1.getInheritedCumulativeTime());						  					  	
					  }
					  
					  
					case 7: //calls
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getCalls()/_totalCalls);
						  }
						  else
						  {					
							  return String.valueOf(ps.getCalls());					  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
						return String.valueOf(pdelta1.getCalls());					  	
					  }
					  
					  
					case 8: //inherited calls
					  if(!info.isDeltaColumn())
					  {																				
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage((double)ps.getInheritedCalls()/_totalCalls);
						  }
						  else
						  {					
							  return String.valueOf(ps.getInheritedCalls());					  
						  }
					  }
					  else
					  {
						p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						return String.valueOf(pdelta1.getInheritedCalls());					  	
					  }					  
				}				  				  				 				   				  
			}
			else
			{
				return getColumnTextClassLevel(obj, col);
			}
            							
			return "";
		}
		
		protected String getColumnTextClassLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
            
			if (obj instanceof TRCClass) {
				
				TRCClass clas = (TRCClass) obj;
				TRCClassImpl.TRCClassSnapshot cs = (TRCClassImpl.TRCClassSnapshot)clas.retrieveSnapshot();
				
				switch (pos) {
					case 0 : //class name
						return ((TRCClass) obj).getName();
					case 1:
						return "";
					case 2 : //package name
   						return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();
					case 3 : //base time
					  if(!info.isDeltaColumn())
					  {																														
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getBaseTime()/_maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getBaseTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						return TString.formatTimeValue(cdelta1.getBaseTime());					  	
					  }
					  
					case 4: //average base time
					{
					   int calls = cs.getCalls();
					   if(calls == 0)
					      calls = 1;
					      
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage(cs.getBaseTime() / (calls*_maxTime));
					  }
					  else
					  {													  					
						return TString.formatTimeValue(cs.getBaseTime() / calls);
					  }
					}					  
					case 5 : //inherited base time
					  if(!info.isDeltaColumn())
					  {																																			
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedBaseTime()/_maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getInheritedBaseTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedBaseTime());					  	
					  }

					case 6 : //cumulative time
					  if(!info.isDeltaColumn())
					  {																																								
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getCumulativeTime() / _maxTime);
						  }
						  else
						  {													  					
					
							return TString.formatTimeValue(cs.getCumulativeTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getCumulativeTime());					  	
					  }
						  
					case 7 : //inherited cumulative time
					  if(!info.isDeltaColumn())
					  {																																													
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(cs.getInheritedCumulativeTime() / _maxTime);
						  }
						  else
						  {													  										
							return TString.formatTimeValue(cs.getInheritedCumulativeTime());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						return TString.formatTimeValue(cdelta1.getInheritedCumulativeTime());					  	
					  }
						  
					case 8 : //calls
					  if(!info.isDeltaColumn())
					  {																																																		
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getCalls()) / _totalCalls);
						  }
						  else
						  {													  										
							  return String.valueOf(cs.getCalls());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						return String.valueOf(cdelta1.getCalls());					  	
					  }
						  
					case 9 : //inherited calls
					  if(!info.isDeltaColumn())
					  {																																																							
						  if(isShowPercent())
						  {
						  	  return TString.formatAsPercentage(((double)cs.getInheritedCalls()) / _totalCalls);
						  }
						  else
						  {													  										
							  return String.valueOf(cs.getInheritedCalls());
						  }
					  }
					  else
					  {
						clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return String.valueOf(cdelta1.getInheritedCalls());					  	
					  }	
				}
			}
			else
			{
				return getColumnTextMethodLevel(obj, col);
			}

			return "";
		}

		protected String getColumnTextMethodLevel(Object obj, int col){
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
			TRCMethod m = (TRCMethod)obj;
			TRCMethodImpl.TRCMethodSnapshot ms = (TRCMethodImpl.TRCMethodSnapshot)m.retrieveSnapshot();
            
            //TRCMethod
			switch(pos)
			{				
				case 0://method name
					StringBuffer buf = new StringBuffer();					
					return buf.append(m.getName()).append(m.getSignature()).toString();
				case 1:
					return "";					
				case 2: //class name
				  return m.getDefiningClass().getName();

				case 3: //package name
					return PerftraceUtil.getPackageName(m.getDefiningClass().getPackage(), _page.getMOFObject()).toString();
				 				  
				case 4: //base time
				  if(!info.isDeltaColumn())
				  {				
					  if(isShowPercent())
					  {
					  	  return PerftraceUtil.formatAsPercentage(ms.getBaseTime() / _maxTime);
					  }
					  else
					  {													  													
					  	return TString.formatTimeValue(ms.getBaseTime());
					  }
				  }
				  else
				  {
					m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
					return TString.formatTimeValue(mdelta1.getBaseTime());					  	
				  }
				  
				case 5: //average base time
				{
				   int calls = ms.getCalls();
				   if(calls == 0)
				      calls = 1;
				      
				  if(isShowPercent())
				  {
				  	  return TString.formatAsPercentage(ms.getBaseTime() / (calls*_maxTime));
				  }
				  else
				  {													  					
					return TString.formatTimeValue(ms.getBaseTime() / calls);
				  }
				}
				  
				case 6: //cumulative time
				  if(!info.isDeltaColumn())
				  {								
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage(ms.getCumulativeTime() / _maxTime);
					  }
					  else
					  {													  									
						return TString.formatTimeValue(ms.getCumulativeTime());
					  }
				  }
				  else
				  {
					m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);					  	
					return TString.formatTimeValue(mdelta1.getCumulativeTime());					  	
				  }
				  
				case 7: //calls
				  if(!info.isDeltaColumn())
				  {								
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)ms.getCalls() / _totalCalls);
					  }
					  else
					  {													  									
					  	  return String.valueOf(ms.getCalls());
					  }
				  }
				  else
				  {
					m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);
					return String.valueOf(mdelta1.getCalls());					  	
				  }
			}				  				  				 				   				  
            							
			return "";
		}
		
		protected String getColumnTextInstanceLevel(Object obj, int col){
			
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(obj instanceof TRCClass)	
			{
				return getColumnTextClassLevel(obj, col);
			}
			else
			{
				if(info.isDeltaColumn())
				   return "";

				TRCObject object = (TRCObject)obj;				
				switch(pos)
				{
					case 0://class name
					{
						TRCClass clas = PerftraceUtil.getClass(object);
						if(clas ==object.getProcess().getClassClass())
						    return clas.getName();					
										
						return clas.getName() + "." + object.getId();
					}
					  
					case 1: //new item
					{
					    return "";
					}
					
					case 2: //package name
					{
						TRCClass clas = PerftraceUtil.getClass(object);						
					    return PerftraceUtil.getPackageName(clas.getPackage(), _page.getMOFObject()).toString();
					}					
					  					  
					case 3: //base time
					
					if(object instanceof TRCFullTraceObject)
					{
					  
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getBaseTime()/_maxTime);
					  }
					  else
					  {													
						  return TString.formatTimeValue(((TRCFullTraceObject)object).getBaseTime());
					  }
					}
					return "";
					  					  
					case 4: //cumulative time
					
					if(object instanceof TRCFullTraceObject)
					{
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getCumulativeTime()/_maxTime);
					  }
					  else
					  {																	
						  return TString.formatTimeValue(((TRCFullTraceObject)object).getCumulativeTime());
					  }
					}
					return "";
					  
					case 5: //calls
					
					if(object instanceof TRCFullTraceObject)
					{
					  if(isShowPercent())
					  {
					  	  return TString.formatAsPercentage((double)((TRCFullTraceObject)object).getCalls()/_totalCalls);
					  }
					  else
					  {																	
						  return String.valueOf(((TRCFullTraceObject)object).getCalls());
					  }
					}
					return "";
				}
			}
			return "";
		}
		
		public String getColumnText(Object obj, int col) {
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return getColumnTextPackageLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return getColumnTextClassLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return getColumnTextMethodLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return getColumnTextInstanceLevel(obj, col);
			else
				return "";
		}
		
		protected Image getColumnImagePackageLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCPackage)	
			{
				TRCPackage p = (TRCPackage)obj;

				switch(pos)
				{
					case 1: //new item column
					  if(p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES)<0)
					  	  return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					  else
					  	  return null;
				
					case 2: //base time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);					
					  d = pdelta1.getBaseTime();
					  
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;

					case 3: //inherited base time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
					  d = pdelta1.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  break;
					  					  
					case 4: //cumulative time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
					  d = pdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 5: //inherited cumulative time
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
					  d = pdelta1.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 6: //calls
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
					  state = pdelta1.getCalls();
					  break;
					  
					case 7: //inherited calls
					  p.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
					  state = pdelta1.getInheritedCalls();
					  break;
					
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageClassLevel(obj, col);
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
		}
		
		protected Image getColumnImageClassLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getClassPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			if(obj instanceof TRCClass)	
			{
				TRCClass clas = (TRCClass)obj;

				switch(pos)
				{
					case 1: //new item column
					  if(clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES)<0)
					  	  return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
					  else
					  	  return null;					
				
					case 2: //package
						return TracePluginImages.getImage(TracePluginImages.IMG_PACKAGE_COL);

					case 3: //base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);					
					  d = cdelta1.getBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;

					case 4: //inherited base time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
					  d = cdelta1.getInheritedBaseTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  break;
					  					  
					case 5: //cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
					  d = cdelta1.getCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 6: //inherited cumulative time
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
					  d = cdelta1.getInheritedCumulativeTime();
					  if(d<0) state = -1;
					  else if(d>0) state = 1;
					  
					  break;
					  
					case 7: //calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
					  state = cdelta1.getCalls();
					  break;
					  
					case 8: //inherited calls
					  clas.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
					  state = cdelta1.getInheritedCalls();
					  break;
					
				}				  				  				 				   				  
			}
			else
			{
				return getColumnImageMethodLevel(obj, col);
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;		
		}
		
		protected Image getColumnImageMethodLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = getMethodPosForLevel(info.getColumnData().getInitalPos());			
			
			if(info.isDeltaColumn())
			   return null;
			
			int state = 0;
			double d = 0;
			
			TRCMethod m = (TRCMethod)obj;

			switch(pos)
			{
				case 1: //new item column
				  if(m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME)<0)
				  	  return TracePluginImages.getImage(TracePluginImages.IMG_DELTANEW);
				  else
				  	  return null;			
			
				case 2: //class
					return TracePluginImages.getImage(TracePluginImages.IMG_CLASS_COL);

				case 3: //package
					return TracePluginImages.getImage(TracePluginImages.IMG_PACKAGE_COL);
					
				case 4: //base time
				  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
				  d = mdelta1.getBaseTime();
				  if(d<0) state = -1;
				  else if(d>0) state = 1;
				  
				  break;
				  
				case 5: //average base time
				  state = 0;				  
				  break;
				  					  					  
				case 6: //cumulative time
				  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
				  d = mdelta1.getCumulativeTime();
				  if(d<0) state = -1;
				  else if(d>0) state = 1;
				  
				  break;
					  
				case 7: //calls
				  m.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);				
				  state = mdelta1.getCalls();
				  break;
				  
			}
                  
			if (state < 0) 
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTADOWN);
			if(state > 0)
				return TracePluginImages.getImage(TracePluginImages.IMG_DELTAUP);
      							
			return null;
		
		}
		
		protected Image getColumnImageInstanceLevel(Object obj, int col)
		{
			StatisticTableColumnInfo info = StatisticTableColumnInfo.getStatisticTableColumnInfo(_viewer.getTable().getColumn(col));
            int pos = info.getColumnData().getInitalPos();			
			
			if(info.isDeltaColumn())
			   return null;
			   
			if(obj instanceof TRCClass)	
			{
				return getColumnImageClassLevel(obj, col);
			}
			else
			{
				switch(pos)
				{
					case 1: //new item column
					  	  return null;		
					case 2: //package
						return TracePluginImages.getImage(TracePluginImages.IMG_PACKAGE_COL);
				}	
			}

			return null;
			
		}
		
		public Image getColumnImage(Object obj, int col) {
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return getColumnImagePackageLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return getColumnImageClassLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return getColumnImageMethodLevel(obj, col);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return getColumnImageInstanceLevel(obj, col);
			return null;
		}
	}

	public class ExecutionStatisticSorter extends MultiLevelStatisticSorter{
		
		protected int comparePackageLevel(Viewer viewer, Object e1, Object e2)
		{
			double d = 0;
 			if(e1 instanceof TRCPackage && e2 instanceof TRCPackage)	
			{
				TRCPackage p1 = (TRCPackage)e1;
				TRCPackage p2 = (TRCPackage)e2;
				TRCPackageImpl.TRCPackageSnapshot ps1 = (TRCPackageImpl.TRCPackageSnapshot)p1.retrieveSnapshot();
				TRCPackageImpl.TRCPackageSnapshot ps2 = (TRCPackageImpl.TRCPackageSnapshot)p2.retrieveSnapshot();

				switch(_pos)
				{
					case 0://package name
					  return (int) _sortSequence * (int)p1.getName().compareToIgnoreCase(p2.getName());
					  
					case 1 : // new items
						  if(p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0
						  		&& !(p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0))
						  	  return _sortSequence;
						  else if(!(p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0)
						  		&& p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__TOTAL_INSTANCES) < 0)
						  	 return _sortSequence * (-1);
						  else
						  	  return 0;				  
					  					  
					case 2: //base time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getBaseTime() - ps2.getBaseTime();						  
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__BASE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__BASE_TIME);
						d = pdelta1.getBaseTime() - pdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					  
					case 3: //average base time
					{
					   int calls1 = ps1.getCalls();
					   if(calls1 == 0)
					      calls1 = 1;
					      
					   int calls2 = ps2.getCalls();
					   if(calls2 == 0)
					      calls2 = 1;
					      
					      
					  d = ps1.getBaseTime()/calls1 - ps2.getBaseTime()/calls2;
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					}					  
					    
			
					case 4: //inherited base time
						
					  if(!_info.isDeltaColumn())
					  {											
						  d = ps1.getInheritedBaseTime() - ps2.getInheritedBaseTime();
					  } 
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_BASE_TIME);
						d = pdelta1.getInheritedBaseTime() - pdelta2.getInheritedBaseTime();					  	
					  }
					  	
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					  					  
					case 5: //cumulative time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getCumulativeTime() - ps2.getCumulativeTime();
					  } 
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__CUMULATIVE_TIME);
						d = pdelta1.getCumulativeTime() - pdelta2.getCumulativeTime();					  	
					  } 
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					  
					case 6: //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {										
						  d = ps1.getInheritedCumulativeTime() - ps2.getInheritedCumulativeTime();
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_CUMULATIVE_TIME);
						d = pdelta1.getInheritedCumulativeTime() - pdelta2.getInheritedCumulativeTime();					  	
					  }  
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;  
					
					case 7: //calls
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getCalls() - ps2.getCalls());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__CALLS);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__CALLS);
						return (int) _sortSequence * (pdelta1.getCalls() - pdelta2.getCalls());					  	
					  }
				
					case 8: //inherited calls
					  if(!_info.isDeltaColumn())
					  {										
						  return (int) _sortSequence * (ps1.getInheritedCalls() - ps2.getInheritedCalls());
					  }
					  else
					  {
						p1.computeDelta(pdelta1,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						p2.computeDelta(pdelta2,TracePackage.TRC_PACKAGE__INHERITED_CALLS);
						return (int) _sortSequence * (pdelta1.getInheritedCalls() - pdelta2.getInheritedCalls());					  	
					  }					  
				}				  				  				 				   				  
			}
 			else
 			{
 				return compareClassLevel(viewer, e1, e2);
 			}

			return 0;
		}
		
		protected int compareClassLevel(Viewer viewer, Object e1, Object e2)
		{
			int pos = getClassPosForLevel(_pos);
			
            double d =0;
            
			if (e1 instanceof TRCClass && e2 instanceof TRCClass) {
					
				TRCClass c1 = (TRCClass) e1;
				TRCClass c2 = (TRCClass)e2;				
				TRCClassImpl.TRCClassSnapshot cs1 = (TRCClassImpl.TRCClassSnapshot)c1.retrieveSnapshot();
				TRCClassImpl.TRCClassSnapshot cs2 = (TRCClassImpl.TRCClassSnapshot)c2.retrieveSnapshot();
				
				switch (pos) {
					case 0 : //class name
						return (int) _sortSequence
							* (int) c1.getName().compareToIgnoreCase(c2.getName());
						
					case 1 : // new items
						  if(c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0
						  		&& !(c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0))
						  	  return _sortSequence;
						  else if(!(c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0)
						  		&& c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__TOTAL_INSTANCES) < 0)
						  	 return _sortSequence * (-1);
						  else
						  	  return 0;		

					case 2 :	//package name
						return (int) _sortSequence
							* (int) c1.getPackage().getName()
								.compareToIgnoreCase(c2.getPackage().getName());

					case 3 : //base time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getBaseTime() - cs2.getBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__BASE_TIME);
						d = cdelta1.getBaseTime() - cdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 4: //average base time
					{
					   int calls1 = cs1.getCalls();
					   if(calls1 == 0)
					      calls1 = 1;
					      
					   int calls2 = cs2.getCalls();
					   if(calls2 == 0)
					      calls2 = 1;
					      
					      
					  d = cs1.getBaseTime()/calls1 - cs2.getBaseTime()/calls2;
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					}						  
					case 5: //inherited base time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getInheritedBaseTime() - cs2.getInheritedBaseTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_BASE_TIME);
						d = cdelta1.getInheritedBaseTime() - cdelta2.getInheritedBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 6 : //cumulative time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getCumulativeTime() - cs2.getCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CUMULATIVE_TIME);
						d = cdelta1.getCumulativeTime() - cdelta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 7 : //inherited cumulative time
					  if(!_info.isDeltaColumn())
					  {																									
						  d = cs1.getInheritedCumulativeTime() - cs2.getInheritedCumulativeTime();
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CUMULATIVE_TIME);
						d = cdelta1.getInheritedCumulativeTime() - cdelta2.getInheritedCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 8 : //calls
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence * (cs1.getCalls() - cs2.getCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__CALLS);
						return (int) _sortSequence * (cdelta1.getCalls() - cdelta2.getCalls());					  	
					  }

					case 9 : //inherited calls
					  if(!_info.isDeltaColumn())
					  {																									
						return (int) _sortSequence 	* (cs1.getInheritedCalls() - cs2.getInheritedCalls());
					  }
					  else
					  {
						c1.computeDelta(cdelta1,TracePackage.TRC_CLASS__INHERITED_CALLS);
						c2.computeDelta(cdelta2,TracePackage.TRC_CLASS__INHERITED_CALLS);
						return (int) _sortSequence * (cdelta1.getInheritedCalls() - cdelta2.getInheritedCalls());					  	
					  }

				}
			} 
			else
			{
				return compareMethodLevel(viewer, e1, e2);
			}
			
			return 0;
		}

		protected int compareMethodLevel(Viewer viewer, Object e1, Object e2)
		{
			int pos = getMethodPosForLevel(_pos);
			
			double d = 0;
 			if(e1 instanceof TRCMethod && e2 instanceof TRCMethod)	
			{
				TRCMethod m1 = (TRCMethod) e1;
				TRCMethod m2 = (TRCMethod) e2;				 
				TRCMethodImpl.TRCMethodSnapshot ms1 = (TRCMethodImpl.TRCMethodSnapshot)m1.retrieveSnapshot();
				TRCMethodImpl.TRCMethodSnapshot ms2 = (TRCMethodImpl.TRCMethodSnapshot)m2.retrieveSnapshot();
				
				switch(pos)
				{
					case 0://method name
					  StringBuffer buf1 = new StringBuffer();
					  StringBuffer buf2 = new StringBuffer();					  
					  return (int) _sortSequence * (int)buf1.append(m1.getName()).append(m1.getSignature()).toString()
					     .compareToIgnoreCase(buf2.append(m2.getName()).append(m2.getSignature()).toString());
					  
					case 1 : // new items
						  if((m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME) < 0)
						  		&& !(m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__BASE_TIME) < 0))
						  	  return _sortSequence;
						  else if(!(m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME) < 0)
						  		&& (m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__BASE_TIME) < 0))
						  		return _sortSequence * (-1);
						  else
						  	  return 0;					  

					case 2://class name
					  return (int) _sortSequence * (int)m1.getDefiningClass().getName().compareToIgnoreCase(m2.getDefiningClass().getName());

					case 3://package name
					  return (int) _sortSequence * (int)m1.getDefiningClass().getPackage().getName().compareToIgnoreCase(m2.getDefiningClass().getPackage().getName());
					  					  					  
					case 4: //base time
					  if(!_info.isDeltaColumn())
					  {																														
						  d = ms1.getBaseTime() - ms2.getBaseTime();
					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__BASE_TIME);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__BASE_TIME);
						d = mdelta1.getBaseTime() - mdelta2.getBaseTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					
					case 5: //average base time
					{
					   int calls1 = ms1.getCalls();
					   if(calls1 == 0)
					      calls1 = 1;
					      
					   int calls2 = ms2.getCalls();
					   if(calls2 == 0)
					      calls2 = 1;
					      
					      
					  d = ms1.getBaseTime()/calls1 - ms2.getBaseTime()/calls2;
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					}
					  
					case 6: //cumulative time
					  if(!_info.isDeltaColumn())
					  {																														
						  d = ms1.getCumulativeTime() - ms2.getCumulativeTime();
					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__CUMULATIVE_TIME);
						d = mdelta1.getCumulativeTime() - mdelta2.getCumulativeTime();					  	
					  }
					  
					  if(d < 0)
					     return  -1 * _sortSequence;
					  if(d > 0) 
					    return _sortSequence;
					    
					  return 0;    
					  
					case 7: //calls
					  if(!_info.isDeltaColumn())
					  {																														
						  return (int) _sortSequence * (ms1.getCalls() - ms2.getCalls());
					  }
					  else
					  {
						m1.computeDelta(mdelta1,TracePackage.TRC_METHOD__CALLS);
						m2.computeDelta(mdelta2,TracePackage.TRC_METHOD__CALLS);
						return (int) _sortSequence * (mdelta1.getCalls() - mdelta2.getCalls());					  	
					  }
				}				  				  				 				   				  
			}
			return 0;
		}
		
		protected int compareInstanceLevel(Viewer viewer, Object e1, Object e2)
		{
			double d = 0;
			
 			if(e1 instanceof TRCClass)	
			{
 				return compareClassLevel(viewer, e1, e2);
			}
			else
			{				 
				TRCObject obj1 = (TRCObject) e1;
				TRCObject obj2 = (TRCObject) e2;  
				
				switch(_pos)
				{
					case 0://instance name
					  return (int) _sortSequence * (int)(obj1.getId() - obj2.getId());
					  
					case 1: //new item
						return 0;
					  
					case 2: //package
						return 0; 
	
					case 3: //base time
					
					if (true)
					{
						d = 0;
	
						if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
						{

							TRCHeapObject object2 = (TRCHeapObject) e2;				
							TRCClass class2 = PerftraceUtil.getClass(object2);
							double baseTime1 = ((TRCFullTraceObject)obj1).getBaseTime();
							double baseTime2 = class2.getBaseTime();
							d = baseTime1 - baseTime2;
						}
						else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
						{

							TRCHeapObject object1 = (TRCHeapObject) e1;				
							TRCClass class1 = PerftraceUtil.getClass(object1);
							double baseTime2 = ((TRCFullTraceObject)obj2).getBaseTime();
							double baseTime1 = class1.getBaseTime();
							d = baseTime1 - baseTime2;
						}
						else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
						{
							d = ((TRCFullTraceObject)obj1).getBaseTime() -((TRCFullTraceObject)obj2).getBaseTime();	
						}
	
						if(d < 0)
							return  -1 * _sortSequence;
						if(d > 0) 
							return _sortSequence;
						return 0;	
					} 
					
					case 4: //cumulative time

					if (true)
					{
						d = 0;
						
						if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
						{
		
								TRCHeapObject object2 = (TRCHeapObject) e2;				
								TRCClass class2 = PerftraceUtil.getClass(object2);
								double time1 = ((TRCFullTraceObject)obj1).getCumulativeTime();
								double time2 = class2.getCumulativeTime();
								d = time1 - time2;
						}
						else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
						{
		
							TRCHeapObject object1 = (TRCHeapObject) e1;				
							TRCClass class1 = PerftraceUtil.getClass(object1);
							double time2 = ((TRCFullTraceObject)obj2).getCumulativeTime();
							double time1 = class1.getCumulativeTime();
							d = time1 - time2;
						}
						else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
						{
							d = ((TRCFullTraceObject)obj1).getCumulativeTime()-((TRCFullTraceObject)obj2).getCumulativeTime();	
						}
						
						if(d < 0)
							return  -1 * _sortSequence;
						if(d > 0) 
							return _sortSequence;
						return 0;	
						
					} 
					  
					case 5: //calls
					
					if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCHeapObject)
					{
						TRCHeapObject object2 = (TRCHeapObject) e2;				
						TRCClass class2 = PerftraceUtil.getClass(object2);
						return (int) _sortSequence * (((TRCFullTraceObject)obj1).getCalls()-class2.getCalls());
							
					}
					else if(obj1 instanceof TRCHeapObject && obj2 instanceof TRCFullTraceObject)
					{
						TRCHeapObject object1 = (TRCHeapObject) e1;				
						TRCClass class1 = PerftraceUtil.getClass(object1);
						return (int) _sortSequence * (class1.getCalls()-((TRCFullTraceObject)obj2).getCalls());
					}
					else if(obj1 instanceof TRCFullTraceObject && obj2 instanceof TRCFullTraceObject)
					{
						return (int) _sortSequence * (((TRCFullTraceObject)obj1).getCalls()-((TRCFullTraceObject)obj2).getCalls());
					}
					return 0;
				}			  				  				 				   				  
			}
			
			return 0;
		
		}
		
		public int compare(Viewer viewer, Object e1, Object e2) {
				
			if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
				return comparePackageLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
				return compareClassLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
				return compareMethodLevel(viewer, e1, e2);
			else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
				return compareInstanceLevel(viewer, e1, e2);			
			
			return 0;
		}
	}

	public class ExecutionStatisticFilter extends StatisticFilter{
		
		public ExecutionStatisticFilter() {
			super();
		}
		
		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			String compareText = "";

			switch (getLevel()) {
				case PACKAGE_LEVEL : //Package level
					if (!(element instanceof TRCPackage))
						return true;
					if (((TRCPackage) element).getName().equals(""))
						compareText =
							TraceUIPlugin.getString("DEFAULT_PACKAGE");
					else
						compareText =
							((TRCPackage) element).getName();
					break;
				case CLASS_LEVEL : //class level
					if (!(element instanceof TRCClass))
						return true;
					compareText = ((TRCClass) element).getName();
					break;

				case METHOD_LEVEL : //method level
					if (!(element instanceof TRCMethod))
						return true;
					compareText = ((TRCMethod) element).getName();
					break;
					
				case INSTANCE_LEVEL : // instance level
					if(element instanceof TRCClass)   
						compareText = ((TRCClass)element).getName();					
					break;
			}

			if (_noPattern)
				return true;

			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {

					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());

				}
			}
			return flag;
		}
	}
	
	public boolean isEmpty()
	{
		return PerftraceUtil.getMaximumTime(_page.getMOFObject()) <= 0;
	}
	
	protected String getDefaultColumnsTemplatePackageLevel()
	{
		return	TraceUIPlugin.getString("STR_ST_PACKAGE") 	+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
			+ TraceUIPlugin.getString("")			+ ":1:"
			    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":14,"				
			+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":2:"
			  + String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:100,"
		    + TraceUIPlugin.getString("STR_ST_CPU_TIME_AVERAGE")		+ ":3:" 
				+ String.valueOf(ColumnData.IS_VISIBLE ) + ":right:100,"			  
			+ TraceUIPlugin.getString("STR_ST_CPU_INHTIME")		+ ":4:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_STACK_TIME") 	+ ":5:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_STACK_INHTIME") 	+ ":6:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100,"			
			+ TraceUIPlugin.getString("STR_ST_CALLS") 		+ ":7:"
			   + String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE) + ":right:50,"
			+ TraceUIPlugin.getString("STR_ST_INHCALLS") 		+ ":8:"
			  + String.valueOf(ColumnData.SUPPORTS_DELTA) + ":right:100";

	}

	protected String getDefaultColumnsTemplateClassLevel()
	{
		return TraceUIPlugin.getString("STR_ST_CLASS_NAME")
		+ ":0:"
		+ String.valueOf(
			ColumnData.NONDELETABLE
				| ColumnData.IS_VISIBLE
				| ColumnData.NONMOVABLE)
		+ ":200,"
		+ TraceUIPlugin.getString("")
		+ ":1:"
	    + String.valueOf(ColumnData.NONDELETABLE
	    					| ColumnData.IS_VISIBLE
							| ColumnData.NONMOVABLE
							| ColumnData.NONRESIZABLE
							| ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS)
		+ ":14,"		
		+ TraceUIPlugin.getString("STR_ST_PACKAGE")
		+ ":2:"
		+ String.valueOf(ColumnData.IS_VISIBLE)
		+ ":left:100,"
		+ TraceUIPlugin.getString("STR_ST_CPU_TIME")
		+ ":3:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)				
		+ ":right:100,"
	    + TraceUIPlugin.getString("STR_ST_CPU_TIME_AVERAGE")
		+ ":4:" 
		+ String.valueOf(ColumnData.IS_VISIBLE )
		+ ":right:100,"			
		+ TraceUIPlugin.getString("STR_ST_CPU_INHTIME")
		+ ":5:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
		+ ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_STACK_TIME")
		+ ":6:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)								
		+ ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_STACK_INHTIME")
		+ ":7:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)				
		+ ":right:100,"				
		+ TraceUIPlugin.getString("STR_ST_CALLS")
		+ ":8:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA | ColumnData.IS_VISIBLE)								
		+ ":right:50,"
		+ TraceUIPlugin.getString("STR_ST_INHCALLS")
		+ ":9:"
		+ String.valueOf(ColumnData.SUPPORTS_DELTA)								
		+ ":right:100";
	}
	
	protected String getDefaultColumnsTemplateMethodLevel()
	{
		return TraceUIPlugin.getString("STR_ST_METHOD_NAME")			+ ":0:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
		+ TraceUIPlugin.getString("")			+ ":1:"
		    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":14,"		
		+ TraceUIPlugin.getString("STR_ST_CLASS_NAME")	+ ":2:"
			+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
		+ TraceUIPlugin.getString("STR_ST_PACKAGE") + ":3:"
			+ String.valueOf(ColumnData.IS_VISIBLE)+ ":left:100,"					
		+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":4:" 
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_CPU_TIME_AVERAGE")		+ ":5:" 
			+ String.valueOf(ColumnData.IS_VISIBLE ) + ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_STACK_TIME")	+ ":6:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:100,"
		+ TraceUIPlugin.getString("STR_ST_CALLS")			+ ":7:"
			+ String.valueOf(ColumnData.IS_VISIBLE | ColumnData.SUPPORTS_DELTA) + ":right:55";
	}
	
	protected String getDefaultColumnsTemplateInstanceLevel()
	{
		return TraceUIPlugin.getString("STR_ST_CLASS_NAME") 	+ ":0:" 
		      	+ String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE) + ":left:200,"
			+ TraceUIPlugin.getString("")			+ ":1:"
			    + String.valueOf(ColumnData.NONDELETABLE | ColumnData.IS_VISIBLE | ColumnData.NONMOVABLE | ColumnData.NONRESIZABLE | ColumnData.NOT_VISIBLE_CHOOSE_COLUMNS) + ":14,"		
		   	+ TraceUIPlugin.getString("STR_ST_PACKAGE") 		+ ":2:"
		   		+ String.valueOf(ColumnData.IS_VISIBLE) + ":left:100,"
			+ TraceUIPlugin.getString("STR_ST_CPU_TIME")		+ ":3:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_STACK_TIME") 	+ ":4:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:100,"
			+ TraceUIPlugin.getString("STR_ST_CALLS") 		+ ":5:"
				+ String.valueOf(ColumnData.IS_VISIBLE) + ":right:50";
	}	
	
	public String getDefaultColumnsTemplate()
	{	
		String columns = super.getDefaultColumnsTemplate();
		
		if (columns == null)
		{
			if (getLevel() == INSTANCE_LEVEL)
				columns = getDefaultColumnsTemplateInstanceLevel();
			else
				columns = "";
		}
	
		return columns;
	}	
	
	/**
	 * @param i:the level of information to display (package/class or method)
	 */
	public void setLevel(int i) {
		_level = i;
	}	
	
	public int getLevel()
	{
		return _level;
	}

	public String getColumnsPreferencesKey()
	{
		setLevel(getLevel());

		if (getLevel() == ExecutionStatisticView.PACKAGE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsPack60";
		else if (getLevel() == ExecutionStatisticView.CLASS_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsClass60";
		else if (getLevel() == ExecutionStatisticView.METHOD_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsMethod60";
		else if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			return PREFERENCE_KEY_PREFIX + "ExecStatsInstance60";
		
		return "";
	}
	
	public IContentProvider getContentProvider()
	{
		return new ExecutionStatisticContentProvider();
	}
	
	public LabelProvider getTableLabelProvider()
	{
		return new ExecutionStatisticLabelProvider(this);
	}
	
	protected StatisticSorter getViewerSorterInstance()
	{
		return new ExecutionStatisticSorter();
	}
	
	protected StatisticFilter getFilterInstance()
	{
		return new ExecutionStatisticFilter();
	}
	
	protected String getViewTypeStr()
	{
		return org.eclipse.hyades.trace.views.adapter.internal.TraceConstants.EXECUTION_STATS_VIEW2;
	}
	
	public void updateButtons() {
		super.updateButtons();
			
		((ExecutionStatisticPage2) getTraceViewerPage()).percentMode().setChecked(isShowPercent());
		if (getLevel() == ExecutionStatisticView.INSTANCE_LEVEL)
			((ExecutionStatisticPage2) getTraceViewerPage()).deltaColumns().setChecked(false);
		else
			((ExecutionStatisticPage2) getTraceViewerPage()).deltaColumns().setChecked(showingDeltaColumns());		
	}		
	
	protected void showPercentUpdate()
	{
		if(isShowPercent())
		{
			_totalCalls = PerftraceUtil.getTotalCalls(_page.getMOFObject());
			if (_totalCalls==0)_totalCalls=1;			
			_totalInst = PerftraceUtil.getTotalInstances(_page.getMOFObject());
			if (_totalInst==0)_totalInst=1;				
			_activeInst = PerftraceUtil.getActiveInstances(_page.getMOFObject());
			if (_activeInst==0)_activeInst=1;			
			_totalSize = PerftraceUtil.getTotalSize(_page.getMOFObject());
			if (_totalSize==0)_totalSize=1;			
			_activeSize = PerftraceUtil.getActiveSize(_page.getMOFObject());
			if (_activeSize==0)_activeSize=1;			
			_maxTime = PerftraceUtil.getMaximumTime(_page.getMOFObject());
			if (_maxTime==0)_maxTime=1;				
		}		
	}
	
	protected void searchInLevel(TableTree table,
			TableTreeItem item,
			TRCPackage pack,
			TRCClass cls,
			String parentClsName,
			TRCMethod meth)
	{
		
		super.searchInLevel(table, item, pack, cls, parentClsName, meth);
		
		switch (getLevel()) {
			case INSTANCE_LEVEL :
				if (meth != null)
					searchInClassLevel(
							table,
							item,
							cls,
							parentClsName,
							null);
				break;
		}
	}	
}
