/**********************************************************************
 * Copyright (c) 2003-2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.util.ArrayList;
import org.eclipse.swt.widgets.*;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;

import org.eclipse.jface.viewers.*;
import org.eclipse.jface.action.IMenuManager;

import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.models.trace.*;

import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticPage;
import org.eclipse.hyades.trace.views.adapter.internal.MultiLevelStatisticViewer;
import org.eclipse.hyades.trace.views.util.internal.*;


public abstract class MultiLevelStatisticView extends StatisticView {
	
	//information can be displayed in three levels:
	// 1: package/class/method level (default)
	// 2: class/method level 
	// 3: method level
	public static final int PACKAGE_LEVEL = 1;
	public static final int CLASS_LEVEL = 2;
	public static final int DEFAULT_LEVEL = PACKAGE_LEVEL;
	
	public static String PREFERENCE_KEY_PREFIX = "org.eclipse.hyades.trace.views.multilevel.";

	/**
	 *  Add a sorter to the coverage view
	 */
	public abstract class MultiLevelStatisticSorter extends StatisticSorter {

		public MultiLevelStatisticSorter() {
			super();
		}

		/**
		 * compare between too coverage elements
		 */
		public abstract int compare(Viewer viewer, Object e1, Object e2);
	}

	/**
	 * Gives the table tree viewer the mlserage elements to display 
	 */
	public abstract class MultiLevelStatisticContentProvider implements ITreeContentProvider {

		public void dispose() {
		}

		public Object getParent(Object element) {

			return null;
		}

		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
		}

		/**
		 * return the list of elements to display in the table tree
		 */
		public abstract Object[] getElements(Object inputElement);

		/**
		 * return the children of the coverage element
		 */
		public Object[] getChildren(Object element) {
			return new ArrayList().toArray();
		}

		/**
		 * test if the coverage element has children
		 */
		public boolean hasChildren(Object element) {
			return (getChildren(element) != null);
		}
	}

	/**
	 * provides the table tree viewer: text and images
	 * for each coverage element to display
	 */
	public abstract class MultiLevelStatisticLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		protected StatisticView _viewer;

		public MultiLevelStatisticLabelProvider(StatisticView viewer) {
			super();
			_viewer = viewer;
		}

		/**
		 * return for the mlserage element to display the corresponding text
		 */
		public abstract String getColumnText(Object obj, int col);
	}

	/**
	 * Implements a table tree viewer	
	 */
	class MultiLevelStatisticTreeViewer extends TableTreeViewer {
		public MultiLevelStatisticTreeViewer(Composite parent) {
			super(parent);
		}

		public MultiLevelStatisticTreeViewer(TableTree table) {
			super(table);
		}

		public void expandItem(TableTreeItem item) {
			item.setExpanded(true);
			createChildren(item);
		}
	}

	public MultiLevelStatisticView(Composite parent, TraceViewerPage page) {
		super(parent, page);
		_viewerFilter = getFilterInstance();
	}

	public abstract boolean isEmpty();
	public abstract String getColumnsPreferencesKey();

	protected abstract String getDefaultColumnsTemplatePackageLevel();
	protected abstract String getDefaultColumnsTemplateClassLevel();
	
	protected abstract StatisticFilter getFilterInstance();
	public abstract IContentProvider getContentProvider();	
	public abstract LabelProvider getTableLabelProvider();	
	protected abstract StatisticSorter getViewerSorterInstance();
	
	public abstract int getLevel();
	public abstract void setLevel(int i);	
	
    protected abstract String getViewTypeStr();	
	
	/**
	 * Create the table tree widget
	 */
	protected Composite createTable(Composite parent, int options) {
		return new TableTree(parent, options);
	}

	protected TableTreeViewer getTableTreeViewer(TableTree table)
	{
		return new MultiLevelStatisticTreeViewer(table);
	}
	
	protected void openSourceForSelection(ISelection selection)
	{
		if (selection != null) {
			TraceViewerPage page = getTraceViewerPage();
			if (page != null) {
				final MultiLevelStatisticViewer viewer = (MultiLevelStatisticViewer)page.getTraceViewer();
				if (viewer != null) {
					BusyIndicator.showWhile(Display.getDefault(), new Runnable() {
						public void run() {
							viewer.openSource().run();
						}
					});
				}
			}
		}
	}

	/**
	 * return the table widget
	 */
	public Table getTable() {
		return ((TableTree) getTableViewer().getControl()).getTable();

	}
	
	/**
	 * Try to see if the selected object is candidate for openSource
	 */
	protected boolean canOpenSourceOnThisObject() {
		ITraceSelection _model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (_model.size() == 0 || _model.getFirstElement() == null) {
			return false;
		}
		Object firstElement = _model.getFirstElement();
		if (firstElement instanceof TRCClass &&
			((TRCClass)firstElement).getSourceInfo() != null &&
			((TRCClass)firstElement).getSourceInfo().getLocation().length() > 0) {
			return true;
		} else if (firstElement instanceof TRCMethod) {
			return true;
		}
		return false;
	}
	
	public String getDefaultColumnsTemplate()
	{
		String columns = null;

		if (getLevel() == 0
			|| getLevel() == PACKAGE_LEVEL) {
			//In the package level the class names column and the package column are hidden				
			columns = getDefaultColumnsTemplatePackageLevel();

		} else if (
			getLevel() == CLASS_LEVEL) {
			//In the class level the class names column is hidden
			columns = getDefaultColumnsTemplateClassLevel();
		}
		
		return columns;
	}	

	/**
	 * The same element must be selected in all the statistics views
	 */
	protected void handleSelectionEvent() {
		ITraceSelection model =
			UIPlugin.getDefault().getSelectionModel(_page.getMOFObject());
		if (model.size() > 0) {
			Object sel = model.getFirstElement();
			if (sel != null) {
				select(sel);
			}
		}
	}

	/**
	 * select the element (in the coverage table) which is selected in the other 
	 * statistics views
	 */
	private void select(Object obj) {
		TRCClass cls = null;
		TRCPackage pack = null;
		TRCMethod meth = null;
		String parentClsName = null;

		TableTree table = (TableTree) getTableViewer().getControl();

		//Summary element
		TableTreeItem[] items = table.getItems();
		if (items.length != 0) // if we have data
			{
			//((MultiLevelStatisticTreeViewer) getTableViewer()).expandItem(items[0]);

			if (obj instanceof TRCPackage)
				pack = (TRCPackage) obj;
			else if (obj instanceof TRCClass) {
				cls = (TRCClass) obj;
				pack = cls.getPackage();
			} else if (obj instanceof TRCMethod) {
				meth = (TRCMethod) obj;
				cls = meth.getDefiningClass();
				pack = cls.getPackage();
			}
			
			searchInLevel(table,
						getItemsRoot(table),
						pack,
						cls,
						parentClsName,
						meth);
		}
	}
	
	protected TableTreeItem getItemsRoot(TableTree table)
	{
		return null;
	}

	/**
	 * search the element to select in the package level table
	 */
	protected void searchInPackageLevel(
		TableTree table,
		TableTreeItem item,
		TRCPackage pack,
		TRCClass cls,
		String parentClsName,
		TRCMethod meth) {
		TableTreeItem pacItem = null;

		TableTreeItem[] pacItems = null;
		
		if (item != null)
			pacItems = item.getItems();
		else
			pacItems = table.getItems();

		pacItem = searchItem(pacItems, pack);

		if (pacItem != null) {
			if (cls == null) { //Package selected
				table.setSelection(new TableTreeItem[] { pacItem });
				return;
			}
			
			((MultiLevelStatisticTreeViewer)getTableViewer()).expandItem(pacItem);

			//TRCClass elements
			searchInClassLevel(
				table,
				pacItem,
				cls,
				parentClsName,
				meth);
			return;
		}
		else if (item != null)
		{
			table.setSelection(new TableTreeItem[] { item });
			return;
		}		
		
		table.deselectAll();
	}

	/**
	 * search the element to select in the class level table
	 */
	protected void searchInClassLevel(
		TableTree table,
		TableTreeItem item,
		TRCClass cls,
		String parentClsName,
		TRCMethod meth) {
		TableTreeItem[] classItems = null;
		TableTreeItem classItemBis = null;

		if (item != null)
			classItems = item.getItems();
		else
			classItems = table.getItems();		
		
		classItemBis = searchItem(classItems, cls);

		if (classItemBis != null) {
			if (meth == null) { // TRCClass selected
				table.setSelection(new TableTreeItem[] { classItemBis });
				return;
			}
			
			((MultiLevelStatisticTreeViewer)getTableViewer()).expandItem(classItemBis);			
			
			searchInMethodLevel(table, classItemBis, meth);
			return;
		}
		else if (item != null)
		{
			table.setSelection(new TableTreeItem[] { item });	
			return;			
		}
		
		table.deselectAll();
	}
	
	/**
	 * search the element to select in the method level table
	 *
	 */
	protected void searchInMethodLevel(
		TableTree table,
		TableTreeItem item,
		TRCMethod meth) {
		TableTreeItem[] methodItems = null;
		
		if (item != null)
			methodItems = item.getItems();
		else
			methodItems = table.getItems();			

		TableTreeItem methodItem = searchItem(methodItems, meth);

		if (methodItem != null) {
			table.setSelection(new TableTreeItem[] { methodItem });
			return;
		}
		else if (item != null)
		{
			table.setSelection(new TableTreeItem[] { item });		
			return;			
		}

		table.deselectAll();
	}
	
	
	protected void searchInLevel(TableTree table,
			TableTreeItem item,
			TRCPackage pack,
			TRCClass cls,
			String parentClsName,
			TRCMethod meth)
	{
		switch (getLevel()) {
		case PACKAGE_LEVEL :
			searchInPackageLevel(
				table,
				item,
				pack,
				cls,
				parentClsName,
				meth);
			break;
		case CLASS_LEVEL :
			if (cls != null)
				searchInClassLevel(
					table,
					item,
					cls,
					parentClsName,
					meth);
			break;
		}
		
	}
	
	protected Object getItemModelData(TableTreeItem item)
	{
		return item.getData();
	}
	
	protected TableTreeItem searchItem(TableTreeItem[] items, Object obj) {
		TableTreeItem item = null;
		for (int j = 0; j < items.length; j++) {
			item = items[j];

			if (item.getData() != null &&
					obj == getItemModelData(item))
				return item;
		}
		return null;
	}		

	/**
	 * Update the "model selection" when an element is selected in the table
	 */
	public void updateModelSelection() {

		ISelection selection = getTableViewer().getSelection();
		if(selection != null && !selection.isEmpty())
		{
			Object sel = ((IStructuredSelection)selection).getFirstElement();
			notifyViewSelectionChanged(this,sel);
		}
	}

	/**
	 * if the selection event is sent by the active table: discard the event
	 * else handle it
	 */
	public void handleViewSelectionChangedEvent(ViewSelectionChangedEvent event) {
		Object  source = event.getSource();
		if(source!=this){	
			handleSelectionEvent();
		}

	}

	/**
	 * Called when the context menu is about to open.
	 * @see IFillMenuTarget#fillContextMenu
	 */
	public void menuAboutToShow(IMenuManager menu) {

		menu.add(getUpdateAction());
		menu.add(fSeparator);
		resetChooseColumnsAction();
		menu.add(
			getChooseColumnsAction(
				getColumnDataList(),
				getColumnsPreferencesKey()));
	}

	/**
	 * Redraw the table to display the three levels
	 * (package level, class level and method level)
	 * 
	 */
	public void redrawTable() {
		getTable().setRedraw(false);

		ArrayList list = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());
		resetColumns(list);
		_currentColumns = list;
		getTable().setRedraw(true);
		
		refresh();
	}

	/**
	 * handle the selection event on:
	 * - A column for sorting
	 * - An element of the table tree or
	 * - on the "Case-sensitive" button
	 */
	public void widgetSelected(org.eclipse.swt.events.SelectionEvent arg0) {
		super.widgetSelected(arg0);
		if (arg0.widget instanceof TableColumn) {
			expandFirstElement();
			handleSelectionEvent();
		}
	}

	/**
	 * Filter the elements
	 */
	protected void filterNames() {
		super.filterNames();
		expandFirstElement();
		handleSelectionEvent();
	}

	/**
	 * dispose the active columns and create others
	 */
	protected void resetColumns(ArrayList list) {
		super.resetColumns(list);
		expandFirstElement();
		handleSelectionEvent();
	}

	/**
	 * reset the choose column Action
	 */
	protected void resetChooseColumnsAction() {
		_chooseColumnAction = null;
	}

	public void dispose() {
		super.dispose();
	}

	public void updateButtons() {
		((MultiLevelStatisticPage) getTraceViewerPage()).updateButtons(getLevel());
		((MultiLevelStatisticPage) getTraceViewerPage()).openSource().setEnabled(canOpenSourceOnThisObject());
	}	
}
