/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.views.internal;

import java.io.*;
import java.util.ArrayList;
import java.util.StringTokenizer;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.trace.*;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.hyades.trace.views.actions.internal.OpenSourceAction;
import org.eclipse.hyades.trace.views.adapter.internal.TraceConstants;
import org.eclipse.hyades.trace.views.util.internal.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.ViewForm;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;

/*
* CONTEXT_ID stvw0001 for filter entry field of statistics views
*/
/*
* CONTEXT_ID stvw0002 for filter case sensitive checkbox of statistics views
*/

/**
 * @version 	1.0
 * @author Mike Wulkan
 * StatisticView is the abstract base class for all of the statistics style views.
 * 
 */
public abstract class StatisticView implements
                                      IViewSelectionChangedListener
									  , SelectionListener
									  , ControlListener
{
	protected ArrayList tmpList;
	
	protected StructuredViewer _viewer;
	protected TableLayout _tableLayout = null;
	protected Composite _viewContainer;
	protected Composite _filterContainer;
	protected ViewForm _dataPane;
	protected Composite _viewComp;
	protected Text _filterText;
	protected Button _caseControl;
	protected boolean _initialized = false;
	protected TraceViewerPage _page;
	protected StatisticFilter _viewerFilter;
	protected StatisticSorter _viewerSorter;
	protected ArrayList _currentColumns = null;
	protected boolean _newSelection = false;
	protected boolean _isShowPercent = false;
	protected boolean _deltaColumns = false;
	
	protected boolean _refresh = false;

	protected Action _updateAction;
	protected Action _chooseColumnAction;
	protected Action _showPercent;
	protected Action _deltaColumnsAction;	
	protected Action _chooseColumnsAction;
	protected Action _printColumnsAction;
	protected Separator fSeparator;
		
	protected double _maxTime;
	protected int    _totalInst;
	protected int    _totalCalls;
	protected int    _activeInst;
	protected long   _totalSize;
	protected long   _activeSize;
	
	//extention points
	protected ArrayList _listOfColumExtension = new ArrayList();
	protected String _templateOfColumExtension="";
	
	public abstract class StatisticSorter extends ViewerSorter {
		protected TableColumn _sortColumn;
		protected int _sortSequence = -1;
		protected StatisticTableColumnInfo _info;
		protected int _pos = 0;

		public StatisticSorter() {
			super();
		}
		
		public void setSortedColumn(TableColumn newSortColumn) {
			_sortColumn = newSortColumn;

			// Get the currently designated sort column and turn remove its sorted indicator
			TableColumn currentColumn = getCurrentSortColumn(newSortColumn.getParent());
			if (currentColumn != null) {
				String text = currentColumn.getText();
				if (text.startsWith(">") || text.startsWith("<"))
					text = text.substring(1);
				if (currentColumn == newSortColumn) {
					_sortSequence = -_sortSequence;
				}
				else {
					_sortSequence = -1;
				}
				currentColumn.setText(text);
				StatisticTableColumnInfo.getStatisticTableColumnInfo(currentColumn).setSortColumn(false);
			}
			else {
				_sortSequence = 1;
			}

			// Add the sorted indicator to the newly sorted column
			String text = newSortColumn.getText();
			
			_info = StatisticTableColumnInfo.getStatisticTableColumnInfo(newSortColumn);
            _pos = _info.getColumnData().getInitalPos();			
            
			newSortColumn.setText(((_sortSequence > 0) ? ">" : "<") + text);
			StatisticTableColumnInfo.getStatisticTableColumnInfo(newSortColumn).setSortColumn(true);
		}
		
		protected int getSortSequence()
		{
			return _sortSequence;
		}
		
		protected void setSortSequence(int sortSequence)
		{
			_sortSequence = sortSequence;
		}

		private TableColumn getCurrentSortColumn(Table table) {
			for (int idx = 0; idx < table.getColumnCount(); idx++) {
				if (StatisticTableColumnInfo.getStatisticTableColumnInfo(table.getColumn(idx)).isSortColumn()) {
					return table.getColumn(idx);
				}
			}
			return null;
		}

		public TableColumn resetSortColumn(Table table) {
			_sortColumn = getCurrentSortColumn(table);
			return _sortColumn;
		}
	}

	public class StatisticFilter extends ViewerFilter {
		protected String _pattern;
		protected String _prefix = "";
		protected String _suffix = "";
		protected ArrayList _textList;
		protected boolean _caseSensitive = true;
		protected boolean _exactMatch;
		protected boolean _noPattern = true;

		public StatisticFilter() {
			super();
		}

		public void setPattern(String pattern, boolean caseSensitive) {
			_caseSensitive = caseSensitive;
			_pattern = _caseSensitive ? pattern : pattern.toLowerCase();
			if (pattern == "*" || pattern == "") {
				_noPattern = true;
			}
			else {
				_noPattern = false;
			}

			_exactMatch = (_pattern.indexOf("*") == -1);

			StringTokenizer st = new StringTokenizer(_pattern, "*");
			_textList = new ArrayList();
			while (st.hasMoreTokens()) {
				_textList.add(st.nextToken());
			}

			if (_pattern.startsWith("*")) {
				_prefix = "";
			}
			else if (_textList.size() > 0) {
				_prefix = (String) _textList.get(0);
				_textList.remove(0);
			}

			if (_pattern.endsWith("*")) {
				_suffix = "";
			}
			else if (_textList.size() > 0) {
				_suffix = (String) _textList.get(_textList.size() - 1);
				_textList.remove(_textList.size() - 1);
			}
		}

		public boolean select(Viewer viewer, Object parent, Object element) {
			
			boolean flag = true;
			
			if (_noPattern)
				return true;
			
			String compareText = "";
			if(element instanceof TRCPackage)
			{
			   compareText = ((TRCPackage)element).getName();
			   if(compareText.equals(""))
			     compareText = TraceUIPlugin.getString("DEFAULT_PACKAGE");
			}						
			else if(element instanceof TRCClass)   
			   compareText = ((TRCClass)element).getName();
			else if(element instanceof TRCMethod)   
			   compareText = ((TRCMethod)element).getName()+((TRCMethod)element).getSignature();
			else if(element instanceof TRCObject)  
			{
				TRCObject obj = (TRCObject) element;
				compareText = obj.getIsA().getName();
			   		
			}
			   
			if (!_caseSensitive) {
				compareText = compareText.toLowerCase();
			}
			if (_exactMatch) {
				return compareText.compareTo(_prefix) == 0;
			}
			if (_prefix != "") {
				flag = compareText.startsWith(_prefix);
			}
			if (flag && _suffix != "") {
				flag = compareText.endsWith(_suffix);
			}
			if (flag) {
				for (int k = 0; k < _textList.size(); k++) {
					String str1 = (String) _textList.get(k);

					int index = compareText.lastIndexOf(str1);
					if (index == -1) {
						flag = false;
						break;
					}

					compareText = compareText.substring(index + str1.length());
				}
			}
			return flag;
		}
		
	}
	
	//-------------------------------
	// UpdateAction popup action
	//-------------------------------
	class UpdateAction extends Action {
		public UpdateAction(String name) {
			super(name);
			org.eclipse.ui.help.WorkbenchHelp.setHelp(UpdateAction.this, TraceUIPlugin.getPluginId() + ".uvew0000");

		    PDPluginImages.setImageDescriptors(this,PDPluginImages.T_LCL,PDPluginImages.IMG_UPDATEVIEWS);					

		}

		/**
		 * Invoked when an action occurs.
		 * Argument context is the Window which contains the UI from which this action was fired.
		 * This default implementation prints the name of this class and its label.
		 */
		public void run() {
			
			ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
			event.setSource(_page.getMOFObject());
			event.setType(ProfileEvent.UPDATE_MODEL);
			
			UIPlugin.getDefault().notifyProfileEventListener(event);
		}
	}

	//-------------------------------
	// ChooseColumnsAction popup action
	//-------------------------------
	class ChooseColumnsAction extends Action {
		private ArrayList _cols;
		private String _key;
		public ChooseColumnsAction(String name, ArrayList cols, String key) {
			super(name);
			_cols = cols;
			_key = key;
			org.eclipse.ui.help.WorkbenchHelp.setHelp(ChooseColumnsAction.this, TraceUIPlugin.getPluginId() + ".ccol0000");
			TracePluginImages.setImageDescriptors(this, TracePluginImages.T_LCL, TracePluginImages.IMG_CHOOSECOLUMNS);
		}

		/**
		 * Invoked when an action occurs.
		 * Argument context is the Window which contains the UI from which this action was fired.
		 * This default implementation prints the name of this class and its label.
		 */
		public void run() {
			ColumnsDialog dialog =
				new ColumnsDialog(getControl().getShell(), TraceUIPlugin.getString("STR_ST_COL_SETTINGS"), null, _cols, getDefaultColumnsTemplate());
			dialog.open();

			if (dialog.getReturnCode() == Window.OK) {
				ColumnData.setColumns(dialog.getData(), _cols, _key);
				resetColumns(_cols);
				refresh();
			}
		}
	}   
		class PrintColumnsAction extends Action {
		public PrintColumnsAction(String name) {
			super(name);
		}

		public void run() {
			PrintColumnsDialog dialog =
				new PrintColumnsDialog(getControl().getShell(), TraceUIPlugin.getString("STR_ST_PRINT_COL"), null);
			dialog.open();

			if (dialog.getReturnCode() == Window.OK) {
				printColumnsToHTML(dialog.getFileName());
			}
		}
	}


	//-------------------------------
	// Show percent popup action
	//-------------------------------
	class ShowPercentAction extends Action {
		public ShowPercentAction(String name) {
			super(name);

			org.eclipse.ui.help.WorkbenchHelp.setHelp(ShowPercentAction.this, TraceUIPlugin.getPluginId() + ".clst0003");
		}

		public void setChecked(boolean value) {
			super.setChecked(value);
		}

		/**
		 * Invoked when an action occurs.
		* Argument context is the Window which contains the UI from which this action was fired.
		 * This default implementation prints the name of this class and its label.
		 */
		public void run() {
			handlePercentChanged();
		}
	}

    /**
     * 
     */
	class DeltaColumnsAction extends Action {
		public DeltaColumnsAction(String name) {
			super(name);
			org.eclipse.ui.help.WorkbenchHelp.setHelp(DeltaColumnsAction.this, TraceUIPlugin.getPluginId() + ".ccol0000");
		}

		public void setChecked(boolean value) {
			super.setChecked(value);
		}
		
		public void run() {
			handleDeltaChanged();
		}
	}

	protected StatisticView(Composite parent, TraceViewerPage page) {
		_page = page;
		_currentColumns = ColumnData.createColumnData(getColumnsPreferencesKey(), getDefaultColumnsTemplate());

		//parse the columns extension
 		ArrayList list=parseExtensions();
 		if ((list!=null)&&(list.size()!=0))
 			_currentColumns.addAll(list);
		
		createControl(parent, _currentColumns);
	    int perc = TraceUIPlugin.getDefault().getPreferenceStore().getInt(buildPreferenceStoreSring(TraceConstants.PERCENTAGE));
	    if(perc == 1)
	        _isShowPercent = true;
	    
	    int delt = TraceUIPlugin.getDefault().getPreferenceStore().getInt(buildPreferenceStoreSring(TraceConstants.DELTA));
	    if(delt == 1)
	        _deltaColumns = true;


		initialize();
		
		tmpList = new ArrayList();
		fSeparator = new Separator();

	}

	/**
	* Attaches a contextmenu listener to the table
	*/
	private void setMenuListener(IMenuListener menuListener) {
		MenuManager menuMgr = new MenuManager();
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(menuListener);
		Menu menu = menuMgr.createContextMenu(_viewer.getControl());

		_viewer.getControl().setMenu(menu);
	}

	/*
	 * @see SelectionListener#widgetSelected(SelectionEvent)
	 */
	public void widgetSelected(org.eclipse.swt.events.SelectionEvent arg0) {
		if (arg0.widget instanceof TableColumn) {
			_viewerSorter.setSortedColumn((TableColumn) arg0.widget);
			_viewer.setSorter(_viewerSorter);
			_viewer.refresh();
		}
		else if (arg0.widget instanceof Table || arg0.widget instanceof TableTree) {
			updateModelSelection();
		}
		else if (arg0.widget instanceof Button) {
			if (arg0.widget == _caseControl) {
				filterNames();
			}
		}

		updateButtons();
	}


	public abstract Table getTable();
	protected abstract IContentProvider getContentProvider();
	protected abstract void handleSelectionEvent();
	protected abstract Composite createTable(Composite parent, int options);
	protected abstract String getColumnsPreferencesKey();
	protected abstract String getDefaultColumnsTemplate();
	protected abstract TableTreeViewer getTableTreeViewer(TableTree table);

	protected void openSourceForSelection(ISelection selection)
	{
		OpenSourceAction action = new OpenSourceAction();

		action.selectionChanged(action,	selection);
		action.run();
	}
	
	protected StructuredViewer createTableViewer(Composite table) {
		TableTreeViewer tv = getTableTreeViewer((TableTree) table);
		tv.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				ISelection selection = event.getSelection();

				if (selection != null && !selection.isEmpty()) {
					TableTreeViewer tv = (TableTreeViewer)getTableViewer();
	                Object node = ((IStructuredSelection)selection).getFirstElement();
					
					if(tv.isExpandable(node))
					{
						tv.setExpandedState(node, !tv.getExpandedState(node));
					}
					else
					{
						openSourceForSelection(selection);
					}
				}
			}
		});
		return (StructuredViewer) tv;
	}
	
	
	protected Composite getViewComponent() {
		return _viewComp;
	}

	protected TraceViewerPage getTraceViewerPage() {
		return _page;
	}

	protected ViewForm getDataPane() {
		return _dataPane;
	}

	public void showNewSelection() {
		if (_newSelection) {
			getTable().showSelection();
			_newSelection = false;
		}
	}

	public boolean showingDeltaColumns() {
		return _deltaColumns; 
	}

	protected void setNewSelection() {
		_newSelection = true;
	}

	public Action getUpdateAction() {
		if (_updateAction == null) {
			_updateAction = new UpdateAction(UIPlugin.getResourceString("REFRESH_VIEWS"));
		}
		return _updateAction;
	}

	public Action getDeltaColumnsAction() {
		if (_deltaColumnsAction == null) {
			_deltaColumnsAction = new DeltaColumnsAction(TraceUIPlugin.getString("STR_ST_SHOW_DELTA_COLUMNS"));
		}
		return _deltaColumnsAction;
	}
	
	public Action getChooseColumnsAction(ArrayList cols, String key) {
		_chooseColumnAction = new ChooseColumnsAction(TraceUIPlugin.getString("STR_ST_CHOOSE_COL"), cols, key);

		return _chooseColumnAction;
	}
	
	public Action getPrintColumnsAction() {
		if (_printColumnsAction == null) {
			_printColumnsAction = new PrintColumnsAction(TraceUIPlugin.getString("STR_ST_PRINT_COL"));
		}
		return _printColumnsAction;
	}
	
	public Action getShowPercentAction() {
		if (_showPercent == null) {
			_showPercent = new ShowPercentAction(TraceUIPlugin.getString("SHOW_PERCENT"));
		}
		return _showPercent;
	}
	
	public boolean isShowPercent() {
		return _isShowPercent;
	}
	
	public Control getControl() {
		return _viewContainer;
	}
	
	public ArrayList getColumnDataList() {
		return _currentColumns;
	}

	public StructuredViewer getTableViewer() {
		return _viewer;
	}
	
	public StatisticSorter getViewerSorter() {
		return _viewerSorter;
	}
	
	public StatisticFilter getViewerFilter() {
		return _viewerFilter;
	}

	public TableLayout getTableLayout() {
		if (_tableLayout == null)
			_tableLayout = new TableLayout();
		return _tableLayout;
	}

	protected abstract void updateButtons();

	public abstract void updateModelSelection();
	
	/*
	 * @see SelectionListener#widgetDefaultSelected(SelectionEvent)
	 */
	public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent arg0) {
	}

	/*
	 * @see ControlListener#controlMoved(ControlEvent)
	 */
	public void controlMoved(ControlEvent arg0) {
	}

	/*
	 * @see ControlListener#controlResized(ControlEvent)
	 */
	public void controlResized(ControlEvent arg0) {
		/* Get the client area for the shell */
		Rectangle dataBounds = getViewComponent().getClientArea();
		getDataPane().setBounds(dataBounds.x, dataBounds.y, dataBounds.width, dataBounds.height);

		// If the table column was resized, update the ColumnData array with the new size
		// and save the new column definitions to the preferences.
		if (arg0.getSource() instanceof TableColumn) {
			TableColumn col = (TableColumn) arg0.getSource();
			StatisticTableColumnInfo columnInfo = (StatisticTableColumnInfo) col.getData();
			ColumnData.setColumns(getColumnDataList(), getColumnsPreferencesKey());
		}
	}

	/*
	 * @see IMenuListener#menuAboutToShow(IMenuManager)
	 */
	public void menuAboutToShow(IMenuManager arg0) {
	}

	/*
	 * This method handles a selection event
	 */
	public void refresh() {
		
		//showPercentUpdate();
			
		if(!getTableViewer().getControl().isDisposed())	
			update();
	}
	
	/*
	 * This method handles a selection event
	 */
	public void selectionChanged() {
		
			handleSelectionEvent();
		
	}
	

	/**
	 * This method applies a new filter pattern to the table data.
	 */
	protected void filterNames() {
		String pattern = _filterText.getText().trim();
		_viewerFilter.setPattern(pattern, _caseControl.getSelection());
		
		(_viewer.getControl()).setRedraw(false);
		
		_viewer.refresh();
		
		(_viewer.getControl()).setRedraw(true);		
	}

	
	public void updateUI(ArrayList cols) {
		update();
		updateButtons();
	}


	protected boolean _firstTime = true;
	
	protected abstract StatisticSorter getViewerSorterInstance();
	
	protected void showPercentUpdate()
	{
	}
	
	protected void setLinesVisible(Table table)
	{
	}
	
	protected void firstTimeUpdate()
	{
		_viewer.addFilter(_viewerFilter);
		_firstTime = false;
		Table table = getTable();
		setLinesVisible(table);
		TableColumn firstColumn = table.getColumn(updateTableGetColumnNumber());
		
		_viewerSorter = getViewerSorterInstance();
		
		_viewerSorter.setSortedColumn(firstColumn);
		_viewer.setSorter(_viewerSorter);
	}
	
	protected int updateTableGetColumnNumber()
	{
		return 0;
	}
	
	protected void postUpdateEvents()
	{
        handleSelectionEvent();
	}	
	
	protected void expandFirstElement()
	{
	}	
	
	public void update() {
		
		if (_firstTime) {
			firstTimeUpdate();
	        
			if (_deltaColumns)
	        	resetColumns(_currentColumns);  			
		}
		
		// set the input of the viewer		
		showPercentUpdate();

        getTable().setRedraw(false);		

        getTableViewer().setInput(_page.getMOFObject());

        getTable().redraw();
		expandFirstElement();		
        getTable().setRedraw(true);
        
        postUpdateEvents();
	}

	/**
	* This method is called as a result of changing the selected columns.
	* It takes the new column definition list and adjusts the columns of the
	* table to match this list.
	*/
	protected void resetColumns(ArrayList list) {
		String sortedColumnName=""; //store sorted column
		int sortSequence = _viewerSorter.getSortSequence(); //store sorted sequence
	
		Table table = getTable();
		table.setRedraw(false);
	
		// Dispose all the existing columns
		for (int idx = table.getColumnCount(); idx > 0; idx--) {
			TableColumn col = table.getColumn(table.getColumnCount() - 1);
			StatisticTableColumnInfo staticTblColInfo = (StatisticTableColumnInfo)(col.getData());
			if (staticTblColInfo.isSortColumn())
			{
				sortedColumnName = staticTblColInfo.getColumnData().name();				
			}
			col.dispose();
		}
	
		for (int idx = 0; idx < list.size(); idx++) {
			ColumnData data = (ColumnData) list.get(idx);
			if (data.visible()) {
				StatisticTableColumnInfo staticTblColInfo  = new StatisticTableColumnInfo(this, data, _deltaColumns);
				if (data.name().equals(sortedColumnName)){
					staticTblColInfo.setSortColumn(true);
				}
			}
		}
		if (_viewerSorter.resetSortColumn(getTable()) == null) {
			_viewer.setSorter(null);
		}
		else {
			_viewerSorter.setSortSequence(sortSequence*-1);
			_viewerSorter.setSortedColumn(_viewerSorter.resetSortColumn(getTable()));
			_viewer.setSorter(_viewerSorter);
		}
		
		table.setRedraw(true);
	}

	private void createControl(Composite parent, ArrayList cols) {
		GridLayout layout;
		GridData data;

		_viewContainer = new Composite(parent, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 2;
		_viewContainer.setLayout(layout);
		_viewContainer.setLayoutData(GridUtil.createFill());

		_filterContainer = new Composite(_viewContainer, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 3;
		_filterContainer.setLayout(layout);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		_filterContainer.setLayoutData(data);

		createFilterControl(_filterContainer);

		_viewComp = new Composite(_viewContainer, SWT.NULL);
		layout = new GridLayout();
		layout.numColumns = 1;
		_viewComp.setLayout(layout);
		data = GridUtil.createFill();
		data.horizontalSpan = 2;
		_viewComp.setLayoutData(data);

		_dataPane = new ViewForm(_viewComp, SWT.NULL);
		Control table = createTableViewer(_dataPane, cols);
		_dataPane.setContent(table);

		_dataPane.setLayoutData(GridUtil.createFill());

		_viewContainer.addControlListener(this);
	}
	
	private void createFilterControl(Composite parent) {
		Label filter = new Label(parent, SWT.NULL);
		filter.setText(TraceUIPlugin.getString("STR_ST_FILTER"));

		_filterText = new Text(parent, SWT.BORDER);
		_filterText.setLayoutData(GridUtil.createHorizontalFill());
		_filterText.addKeyListener(new KeyAdapter() {
			public void keyReleased(KeyEvent key) {
				if (key.character == 13) //ENTER
					filterNames();
			}
		});

		_caseControl = new Button(parent, SWT.CHECK);
		_caseControl.setLayoutData(GridUtil.createHorizontalFill());
		_caseControl.setText(TraceUIPlugin.getString("STR_ST_CASE_SENSITIVE"));
		_caseControl.setSelection(true);
		_caseControl.addSelectionListener(this);

		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_filterText,
			TraceUIPlugin.getPluginId() + ".stvw0001");
		org.eclipse.ui.help.WorkbenchHelp.setHelp(
			_caseControl,
			TraceUIPlugin.getPluginId() + ".stvw0002");
	}
	
	private Control createTableViewer(Composite parent, ArrayList cols) {
		Composite table = createTable(parent, SWT.SINGLE | SWT.FULL_SELECTION);

		// Create the TableTree Viewer
		_viewer = createTableViewer(table);

		for (int idx = 0; idx < cols.size(); idx++) {
			ColumnData data = (ColumnData) cols.get(idx);
			if (data.visible()) {
				new StatisticTableColumnInfo(this, data, _deltaColumns);
			}
		}

		Table tbl = getTable();
		tbl.setLinesVisible(true);
		tbl.setHeaderVisible(true);
		tbl.setLayout(getTableLayout());
		tbl.setLayoutData(new GridData(GridData.FILL_BOTH));

		// add the content provider
		_viewer.setContentProvider(getContentProvider());

		// add the label provider
		_viewer.setLabelProvider(getTableLabelProvider());

		if (table instanceof TableTree) {
			((TableTree) table).addSelectionListener(this);
		}
		else if (table instanceof Table) {
			((Table) table).addSelectionListener(this);
		}

		UIPlugin.getDefault().addViewSelectionChangedListener(this);
        
		return _viewer.getControl();
	}

	public void dispose() {
		
		UIPlugin.getDefault().removeViewSelectionChangedListener(this);
		
		_page = null;
	}

	private void initialize() {
		if (_initialized)
			return;

		_initialized = true;

	}
	
	public abstract LabelProvider getTableLabelProvider();

    public void handlePercentChanged()
	{			
		_isShowPercent = !_isShowPercent;
		TraceUIPlugin.getDefault().getPreferenceStore()
			.setValue(buildPreferenceStoreSring(TraceConstants.PERCENTAGE), (_isShowPercent?1:0));
		
		refresh();
		
		updateButtons();
	}
    
    protected abstract String getViewTypeStr();
    
    public void handleDeltaChanged()
    {
    	_deltaColumns = !_deltaColumns;

    	TraceUIPlugin.getDefault().getPreferenceStore()
		.setValue(buildPreferenceStoreSring(TraceConstants.DELTA), (_deltaColumns?1:0));
    	
    	
    	resetColumns(_currentColumns);
    	refresh();
    }
    
    protected String buildPreferenceStoreSring(String str)
    {
    	return getViewTypeStr() + "." + str;
    }

	private void printColumnsToHTML(String fileName) {
		Writer writer = null;
		try {
			File file = new File(fileName);
			OutputStream outputStream = new FileOutputStream(file.toString(), false);
			writer = new OutputStreamWriter(outputStream, "UTF8");
		} catch (IOException ioException) {
			ioException.printStackTrace();
		}

		String caption = _page.getTraceViewer().getTitle();
		EObject element = _page.getMOFObject();
		if (element != null && element instanceof TRCAgent) {
			
			String time = String.valueOf(((TRCAgent) element).getStartTime());
			if(time != null)
			   caption += " (" + time + ")";
		}

		final String newLine = System.getProperties().getProperty("line.separator");
		StringBuffer buffer = new StringBuffer();
		buffer.append("<html>" + newLine);
		buffer.append("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">" + newLine
		              + "<head>" + newLine
		              + "<STYLE TYPE=\"text/css\">"  + newLine
		              + "p, table, td, th {  font-family: arial, helvetica, geneva; font-size: 10pt}" + newLine
		              + "body {  font-family: arial, helvetica, geneva; font-size: 10pt; clip:   rect(   ); margin-top: 5mm; margin-left: 3mm}" + newLine
		              + "</STYLE>" + newLine
		              + "</head>" + newLine);
		buffer.append("<body text=\"#000000\" bgcolor=\"#ffffff\">" + newLine);
		buffer.append("<table border=0 cellspacing=2 cellpadding=2 WIDTH=\"100%\">" + newLine);
		buffer.append("<caption align=top><B>" + newLine);
		buffer.append(caption + newLine);
		buffer.append("</B></caption>" + newLine);

		Table table = getTable();

		buffer.append("<tr>" + newLine);
		for (int idx = 0; idx < table.getColumnCount(); idx++) {
			String columnText = table.getColumn(idx).getText();

			if (columnText.indexOf("<") == 0)
				columnText = "&lt;" + columnText.substring(1);
			else if (columnText.indexOf(">") == 0)
				columnText = "&gt;" + columnText.substring(1);

			buffer.append("<th align=left>" + columnText + "</th>" + newLine);
		}
		buffer.append("</tr>" + newLine);

        boolean color = true;
		for (int kdx = 0; kdx < table.getItemCount(); kdx++) {
			
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">" + newLine);
			else   
				buffer.append("<tr>" + newLine);
				
		    color = !color;		
			for (int ndx = 0; ndx < table.getColumnCount(); ndx++) {
				String itemText = table.getItem(kdx).getText(ndx);

				buffer.append("<td align=left>" + itemText + "</td>" + newLine);
			}
			buffer.append("</tr>" + newLine);
		}
		buffer.append("</table>" + newLine);
		buffer.append("</body>" + newLine);
		buffer.append("</html>" + newLine);
		try {
			writer.write(buffer.toString());
			writer.flush();
			writer.close();
		} catch (Exception exception) {
			exception.printStackTrace();
		}
	}
	
	/**
 	 * This method should be implemented in the childs 
 	 * @return
 	 */
 	protected ArrayList parseExtensions(){return null;}
	
 	public void notifyViewSelectionChanged(Object source, Object selection) {
		UIPlugin.getDefault().getSelectionModel(_page.getMOFObject()).add(selection);
			
		ViewSelectionChangedEvent event = UIPlugin.getDefault().getViewSelectionChangedEvent();
		event.setSource(source);
		UIPlugin.getDefault().notifyViewSelectionChangedListener(event);
	}
}
