/**********************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.HyadesConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.ProcessTreeItem;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * @author znensi
 */
public class ProfileAttachDelegate implements ILaunchConfigurationDelegate
{
	public void launch(ILaunchConfiguration conf, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException {
		try {
			monitor.beginTask("", 4);
			monitor.subTask(UIPlugin.getResourceString("LAUNCHING_MESSAGES_ON_PROGRESS_VIEW"));
			boolean success = ProfileLaunchUtil.performProfilingTypesLaunch(conf);
			monitor.worked(1);
			
			if (!success) {
				monitor.setCanceled(true);
				return;
			}

			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			
			String hostName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getInt(HyadesConstants.LOCALHOST_PORT));
			String projectName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(TraceConstants.TRACE_PROJECT_NAME));
			String monitorName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(TraceConstants.TRACE_MONITOR_NAME));
			String profileFile = null;
			if (conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false))
				profileFile = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String)null);
				
			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			ArrayList filters = manager.getFilters(conf);
			Vector options = manager.getOptions(conf);
			Vector agents = new Vector(getAgents(conf));
			monitor.worked(1);
			
			/*
			 * #59138: Check if the agents listed in the launch config
			 * are still valid. If not, display an error message.
			 */
			List agentList = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, (List)null);
	
			// count the agents
			int agentCount = 0;
			if (agentList != null) {
				Iterator iter = agentList.iterator();
				while (iter.hasNext()) {
					String agentString = (String)iter.next();
					agentCount += new StringTokenizer(agentString, ":,").countTokens() - 1;
				}
			}
			
			if (agentCount != agents.size()) {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						Shell parent = UIPlugin.getActiveWorkbenchShell();
						IStatus status = new Status(Status.WARNING, UIPlugin.getPluginId(), IResourceStatus.WARNING, UIPlugin.getResourceString("ERROR_DLG_INVALID_AGENTS_REASON"), null);
						ErrorDialog.openError(parent, UIPlugin.getResourceString("TRACE_MSG"), UIPlugin.getResourceString("LAUNCH_ERROR_"), status);
					}
				});
				monitor.setCanceled(true);
				return;
			}
			monitor.worked(1);
			
			PDCoreUtil.attach(hostName, agents, options, filters, projectName, monitorName, String.valueOf(port), profileFile, launch);
			
			Display.getDefault().asyncExec(new Runnable() {
	        	public void run() {
	        		try{
	        			if (UIPlugin.getDefault().getPreferenceStore().getBoolean(TraceConstants.PROFILE_TIPS)) {
		        			MessageDialogWithToggle.openInformation(UIPlugin.getActiveWorkbenchShell(), UIPlugin.getResourceString("PROFILING_TIP_TITLE"), UIPlugin.getResourceString("ATTACH_INFO_MESSAGE"), UIPlugin.getResourceString("ATTACH_MESSAGE_CHECKBOX_MSG"), false, UIPlugin.getDefault().getPreferenceStore(), TraceConstants.PROFILE_TIPS);
	        			}
	        		} catch (Exception exc) {
	        			exc.printStackTrace();
	        		}
	        	}
	        });
			monitor.worked(1);
		}
		catch (CoreException e) {
			monitor.setCanceled(true);
			throw e;
		}
	}
	
	static List getNodeConfiguration(ILaunchConfiguration conf)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();

		try
		{
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, "localhost");
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(HyadesConstants.LOCALHOST_PORT));
			Enumeration enum = PDCoreUtil.getNodeConfiguration(host, port); 
			List list = new ArrayList();
			if (enum != null)
			{
				while (enum.hasMoreElements())
					list.add(enum.nextElement());
			}
			//if the enum is null, an empty list will be returned
			return list;
		}
		catch (CoreException e)
		{
			HyadesUIPlugin.logError(e);
			return null;
		}
	}
	
	public static class AgentAttribute
	{
		private String _name;
		private String _type;
		
		public AgentAttribute(Agent agent)
		{
			_name = agent.getName();
			_type = agent.getType();
		}
		
		public AgentAttribute(String agentAttributeString)
		{
			StringTokenizer tok = new StringTokenizer(agentAttributeString, ";");
			
			_name = tok.nextToken();
			_type = tok.nextToken();
		}
		
		public String getName()
		{
			return _name;
		}
		
		public String getType()
		{
			return _type;
		}
		
		public String getAttributeString()
		{
			return getName() + ";" + getType();
		}
	}
	
	public static class ProcessAttribute
	{
		private String _pid;
		private Vector _agentAttributes;
		
		public ProcessAttribute(Process process)
		{
			try {
				_pid = process.getProcessId();

				_agentAttributes = new Vector();
	
				Enumeration agentItems = process.listAgents();
				while (agentItems.hasMoreElements())
					_agentAttributes.addElement(new AgentAttribute((Agent) agentItems.nextElement()));
			}
			catch (InactiveProcessException e)
			{
				HyadesUIPlugin.logError(e);
			}
		}
		
		public ProcessAttribute(String processAttributeString)
		{
			StringTokenizer tok = new StringTokenizer(processAttributeString, ":");
			_pid = tok.nextToken();
			
			
			_agentAttributes = new Vector();
			tok = new StringTokenizer(tok.nextToken(), ",");

			while (tok.hasMoreTokens())
				_agentAttributes.addElement(new AgentAttribute(tok.nextToken()));
		}
		
		public String getPID()
		{
			return _pid;
		}
		
		public String getAttributeString()
		{
			StringBuffer buf = new StringBuffer();

			buf.append(_pid);
			buf.append(':');

			for (int i = 0; i < _agentAttributes.size(); i++)
			{
				AgentAttribute agentAttr = (AgentAttribute) _agentAttributes.elementAt(i);
				buf.append(agentAttr.getAttributeString());

				// agent delimiter
				if (i < (_agentAttributes.size()-1))
					buf.append(',');
			}	
			return buf.toString();
		}
		
		public int numAgents()
		{
			return _agentAttributes.size();
		}
		
		public Vector getAgentAttributes()
		{
			return _agentAttributes;
		}
		
	}
	
	public static List getAttributeProcesses(ILaunchConfiguration conf)
	{
		try
		{
			List processes = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, (List)null);
			
			if (processes == null)
				return new ArrayList();
			
			Iterator iter = processes.iterator();

			ArrayList list = new ArrayList();			
			
			while (iter.hasNext())
				list.add(new ProcessAttribute((String)iter.next()));
			
			return list;
		}
		catch (Exception e)
		{
			HyadesUIPlugin.logError(e);
		}
		
		return new ArrayList();
	}
	
	public static List getAttributeAgents(ILaunchConfiguration conf)
	{
		Iterator iter = getAttributeProcesses(conf).iterator();
		
		ArrayList list = new ArrayList();
		
		while (iter.hasNext())
		{
			ProcessAttribute pa = (ProcessAttribute)iter.next();
			list.addAll(pa.getAgentAttributes());
		}
		return list;
	}	

	public static List getAgents(ILaunchConfiguration conf) {
		return getAgents(conf, getNodeConfiguration(conf));
	}
	
	static List getAgents(ILaunchConfiguration conf, List nodes)
	{
		List agents = new ArrayList();

		if (nodes != null) {
			try
			{
				Iterator iter = nodes.iterator();
				List list = getAttributeProcesses(conf);
				if (list != null)
				{
					Map map = createMap(list);
	
					Vector v = new Vector();
					while (iter.hasNext())
					{
						Process process = (Process)iter.next();
						String[] selected = (String[])map.get(process.getProcessId());
						if (selected != null)
						{
							for (int i = 0; i < selected.length; ++i)
								agents.add(process.getAgent(selected[i]));
						}
					}
				}
			}
			catch (Exception e)
			{
				HyadesUIPlugin.logError(e);
			}
		}

		return agents;
	}

	// parse and organize the agents in a more accessible data structure
	public static Map createMap(List list)
	{
		Map map = new HashMap();
		
		Iterator iter = list.iterator();
		while (iter.hasNext())
		{
			ProcessAttribute pa = (ProcessAttribute) iter.next();

			Vector agentAttributes = pa.getAgentAttributes();
			String[] agents = new String[agentAttributes.size()];

			for (int i = 0; i < agents.length; ++i)
				agents[i] = ((AgentAttribute)agentAttributes.elementAt(i)).getName();

			map.put(pa.getPID(), agents);
		}		
		
		return map;
	}
}

