/*******************************************************************************
 * Copyright (c) 2004 Hyades project. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Common
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.lang.reflect.InvocationTargetException;
import java.util.Vector;

import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.DeferredProcessAdapter;
import org.eclipse.hyades.trace.ui.internal.launcher.ProcessWrapperRunnable;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.progress.DeferredTreeContentManager;

public class CTree {

	private Label _label;

	private Tree _tree;

	private Composite groupWidget;

	private TreeViewer _viewer;

	private CTreeContentProvider _contentProvider;

	private CTreeLabelProvider _labelProvider;

	private final static Object[] EMPTY_ARRAY = new Object[0];

	public class CTreeContentProvider implements ITreeContentProvider {

		private ILaunchConfiguration _conf;

		private DeferredTreeContentManager tManager;

		public void setLaunchConfiguration(ILaunchConfiguration conf) {
			_conf = conf;
		}

		public ILaunchConfiguration getLaunchConfiguration() {
			return _conf;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof ProcessTreeItem) {
				ProcessTreeItem pti = (ProcessTreeItem) parentElement;

				if (tManager != null) {

					pti.setAdapter(new DeferredProcessAdapter(this, pti,
							getTreeViewer()));
					Object[] children = tManager.getChildren(pti);
					if (children != null)
						return children;
				}
				return fetchChildren(pti);
			} else if (parentElement instanceof ProfilingProcessList) {
				return ((ProfilingProcessList) parentElement).getProcesses();
			}

			return EMPTY_ARRAY;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {
			if (element instanceof ProcessTreeItem)
			{
				return ((ProcessTreeItem)element).getParent();
			}
			
			return null;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {

			if (element instanceof ProcessTreeItem) {
				ProcessTreeItem pti = (ProcessTreeItem) element;

				if (pti.getData() == null || pti.getData() instanceof Process)
					return true;
				else
					return false;
			} else if (element instanceof ProfilingProcessList) {
				return ((ProfilingProcessList) element).getProcessCount() > 0;
			}

			return false; // the "Update ..." placeholder has no children
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
		 *      java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			if (viewer instanceof AbstractTreeViewer) {
				tManager = new DeferredTreeContentManager(this,
						(AbstractTreeViewer) viewer);
			}
		}

		public Object[] fetchChildren(Object parent) {
			IRunnableContext context = UIPlugin.getActiveWorkbenchWindow();
			ProcessWrapperRunnable runnable = new ProcessWrapperRunnable(
					parent, _conf);
			try {
				context.run(true, true, runnable);
			} catch (InvocationTargetException e) {
				e.printStackTrace();
				return EMPTY_ARRAY;
			} catch (InterruptedException e) {
				e.printStackTrace();
				return EMPTY_ARRAY;
			}

			return runnable.getChildren();
		}
	}

	public class CTreeLabelProvider extends LabelProvider {

		public String getText(Object element) {
			if (element instanceof ProcessTreeItem)
				return ((ProcessTreeItem) element).getLabelText();
			else if (isPendingUpdate(element))
				return UIPlugin.getResourceString("STR_PENDING_ITEM");

			return "";
		}

		private boolean isPendingUpdate(Object element) {
			return element instanceof IWorkbenchAdapter;
		}

		public Image getImage(Object element) {
			if (element instanceof ProcessTreeItem)
				return ((ProcessTreeItem) element).getImage();

			return super.getImage(element);
		}
	}

	public CTree(Composite parent, String text) {
		// Create the grid.
		groupWidget = new Composite(parent, SWT.NULL);

		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		groupWidget.setLayout(layout);
		GridData data = GridUtil.createFill();
		data.heightHint = 200;
		data.widthHint = 250;
		groupWidget.setLayoutData(data);

		_label = new Label(groupWidget, SWT.NULL);
		_label.setText(text);
		_tree = new Tree(groupWidget, SWT.MULTI | SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL);

		_viewer = new TreeViewer(_tree);

		_viewer.setContentProvider(getContentProvider());
		_viewer.setLabelProvider(getLabelProvider());

		_tree.setLayoutData(GridUtil.createFill());
	}

	protected IContentProvider getContentProvider() {
		if (_contentProvider == null)
			_contentProvider = new CTreeContentProvider();
		return _contentProvider;
	}

	protected ILabelProvider getLabelProvider() {
		if (_labelProvider == null)
			_labelProvider = new CTreeLabelProvider();
		return _labelProvider;
	}

	protected TreeViewer getTreeViewer() {
		return _viewer;
	}

	public void initializeTree(ILaunchConfiguration conf) {
		_contentProvider.setLaunchConfiguration(conf);
		getTreeViewer().setInput(new ProcessTreeItem(null, null));
	}

	public void resetTree(ProfilingProcessList processList) {
		getTreeViewer().setInput(processList);
	}

	private Tree getTree() {
		return getTreeViewer().getTree();
	}

	public void setRedraw(boolean redraw) {
		getTree().setRedraw(redraw);
	}

	public void enable(boolean enable) {
		getTree().setEnabled(enable);
		_label.setEnabled(enable);
	}

	public Object[] getSelectedItems() {
		Vector selectedItemsV = new Vector();
		
		ISelection iSelection = getTreeViewer().getSelection();
		
		if (!(iSelection instanceof StructuredSelection))
			return new Object[0];
		
		Object [] selection = ((StructuredSelection)iSelection).toArray();
		ProcessTreeItem pti;
		
		for (int i = 0; i < selection.length; i++) {
			
			if (selection[i] instanceof ProcessTreeItem)
			{
				pti = (ProcessTreeItem) selection[i];
				
				if (pti.getData() instanceof Process) {
					selectedItemsV.addElement(pti);
				} else if (pti.getData() instanceof Agent) {
					if (!selectedItemsV.contains(pti.getParent()))
						selectedItemsV.addElement(pti.getParent());
				}
			}
			else
			{
				selectedItemsV.addElement(selection[i]);
			}
		}

		return selectedItemsV.toArray();
	}

	public Object[] getItems() {
		TreeItem []items =  getTree().getItems();
		
		Vector itemsV = new Vector();
		
		for(int i = 0; i < items.length; i++)
		{
			if (items[i].getData() instanceof ProcessTreeItem)
			{
				if (((ProcessTreeItem)items[i].getData()).getData() instanceof Process)
					itemsV.addElement(items[i].getData());				
			}
			else
			{
				itemsV.addElement(items[i].getData());
			}

		}
		return itemsV.toArray();
	}

	public void clearSelection() {
		getTree().deselectAll();
	}

	public void setSelectionFromData(Object[] data) {
		StructuredSelection sSelection = new StructuredSelection(data);

		getTreeViewer().setSelection(sSelection);
	}

	public void addSelectionListener(SelectionListener listener) {
		getTree().addSelectionListener(listener);
	}

	public void addPaintListener(PaintListener listener) {
		getTree().addPaintListener(listener);
	}

	public boolean hasWidget(Widget widget) {
		return getTree() == widget;
	}

	public int getItemCount() {
		return getItems().length;
	}
}