/*******************************************************************************
 * Copyright (c) 2004 Hyades project. All rights reserved. This program and the
 * accompanying materials are made available under the terms of the Common
 * Public License v1.0 which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: IBM - Initial API and implementation
 ******************************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.*;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.ui.*;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileAttachDelegate;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;

public class DoubleCTree extends AbstractChangeable {

	Composite _group;

	Composite groupWidget;

	private CTree sourceList;

	private CTree targetList;

	private ProfilingProcessList sourceProcessList;

	private ProfilingProcessList targetProcessList;

	private boolean _needToResetAgentLists = false;

	private boolean _treesLoaded = false;

	public Label targetListLabel;

	public Label sourceListLabel;

	public Button add;

	public Button remove;

	public Button addAll;

	public Button removeAll;

	private WizardPage _wizardPage;

	private ILaunchConfiguration _conf;

	class TreePaintListener implements PaintListener {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.eclipse.swt.events.PaintListener#paintControl(org.eclipse.swt.events.PaintEvent)
		 */
		public void paintControl(PaintEvent e) {
			resetAgentLists();
		}

	}

	class TreeSelListener implements SelectionListener {
		public void widgetSelected(SelectionEvent e) {
			sourceProcessList.initializeFromProcesses(sourceList.getItems());

			//allow only one list to have selection
			if (sourceList.hasWidget(e.widget)
					&& sourceList.getSelectedItems().length != 0)
				targetList.clearSelection();
			else if (targetList.hasWidget(e.widget)
					&& targetList.getSelectedItems().length != 0)
				sourceList.clearSelection();

			changed();
		}

		public void widgetDefaultSelected(SelectionEvent e) {
			sourceProcessList.initializeFromProcesses(sourceList.getItems());

			if (sourceList.hasWidget(e.widget)) {
				Object[] items = sourceList.getSelectedItems();
				if (items.length > 0) {
					addAgentsToTargetList(items);
					changed();
				}
			} else if (targetList.hasWidget(e.widget)) {
				Object[] items = targetList.getSelectedItems();
				if (items.length > 0) {
					removeAgentsFromTargetList(items);
					changed();
				}
			}
		}
	}

	class ButtonListener implements Listener {

		public void handleEvent(Event e) {
			sourceProcessList.initializeFromProcesses(sourceList.getItems());

			if (e.widget == add) {
				addAgentsToTargetList(sourceList.getSelectedItems());
			} else if (e.widget == addAll) {
				addAgentsToTargetList(sourceList.getItems());
			} else if (e.widget == remove) {
				removeAgentsFromTargetList(targetList.getSelectedItems());
			} else if (e.widget == removeAll) {
				removeAgentsFromTargetList(targetList.getItems());
			}

			changed();
		}
	}

	protected void addAgentsToTargetList(Object[] items) {
		targetList.setRedraw(false);
		sourceList.setRedraw(false);

		Vector dataToBeSelected = new Vector();

		for (int i = 0; i < items.length; i++) {
			if (!vaildItemtoAdd(items[i]))
				continue;

			dataToBeSelected.addElement(items[i]);
			targetProcessList.addProcess(items[i]);
			sourceProcessList.removeProcess(items[i]);
		}

		sourceList.resetTree(sourceProcessList);
		targetList.resetTree(targetProcessList);

		sourceList.clearSelection();

		if (dataToBeSelected.size() > 0)
			targetList.setSelectionFromData(dataToBeSelected.toArray());

		targetList.setRedraw(true);
		sourceList.setRedraw(true);

		if ((_wizardPage != null) && getSelectedItems().length > 0)
			_wizardPage.setPageComplete(true);
	}

	protected void removeAgentsFromTargetList(Object[] items) {
		targetList.setRedraw(false);
		sourceList.setRedraw(false);

		for (int i = 0; i < items.length; i++) {
			targetProcessList.removeProcess(items[i]);
			sourceProcessList.addProcess(items[i]);
		}

		targetList.clearSelection();
		if ((_wizardPage != null) && getSelectedItems().length > 0)
			_wizardPage.setPageComplete(true);

		sourceList.resetTree(sourceProcessList);
		targetList.resetTree(targetProcessList);

		targetList.setRedraw(true);
		sourceList.setRedraw(true);

	}

	private boolean vaildItemtoAdd(Object item) {
		if (item == null || item instanceof IWorkbenchAdapter)
			return false;
		else
			return true;
	}

	public Object[] getTargetItems() {
		return targetList.getItems();
	}

	public void removeAll() {
		sourceProcessList.initializeFromProcesses(new Object[0]);
		targetProcessList.initializeFromProcesses(new Object[0]);

		sourceList.setRedraw(false);
		targetList.setRedraw(false);

		sourceList.resetTree(sourceProcessList);
		targetList.resetTree(targetProcessList);

		sourceList.setRedraw(true);
		targetList.setRedraw(true);

	}

	public void setWizardPage(WizardPage wizardPage) {
		_wizardPage = wizardPage;
	}

	public DoubleCTree(Composite parent, int style, String groupText,
			String sourceTextID, String targetTextID) {
		_treesLoaded = false;

		sourceProcessList = new ProfilingProcessList();
		targetProcessList = new ProfilingProcessList();

		//_group = new Group(parent, style);
		_group = new Composite(parent, style);
		//_group.setText(groupText);

		GridLayout layout = new GridLayout();
		layout.verticalSpacing = -10;
		layout.horizontalSpacing = 0;
		_group.setLayout(layout);
		_group.setLayoutData(GridUtil.createFill());

		createDoubleTree(_group, sourceTextID, targetTextID);

		//add listeners
		add.addListener(SWT.Selection, new ButtonListener());
		remove.addListener(SWT.Selection, new ButtonListener());
		addAll.addListener(SWT.Selection, new ButtonListener());
		removeAll.addListener(SWT.Selection, new ButtonListener());
		//double-click action
		sourceList.addSelectionListener(new TreeSelListener());
		targetList.addSelectionListener(new TreeSelListener());
		///////////////////

		sourceList.addPaintListener(new TreePaintListener());
	}

	/**
	 * Create double list components
	 */
	public void createDoubleTree(Composite parent, String sourceTextID,
			String targetTextID) {
		GridData spec;
		_conf = null;

		// Create the grid.
		groupWidget = parent;

		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		groupWidget.setLayout(layout);

		sourceList = new CTree(groupWidget, sourceTextID);

		// Create the grid.
		Composite buttonWidget = new Composite(groupWidget, SWT.NULL);
		GridLayout blayout = new GridLayout();
		blayout.numColumns = 1;
		buttonWidget.setLayout(blayout);

		Label label = new Label(buttonWidget, SWT.PUSH);
		label.setText("");
		spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);

		add = new Button(buttonWidget, SWT.PUSH);
		add.setText(UIPlugin.getResourceString("LEFT_TO_RIGHT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		add.setLayoutData(spec);

		remove = new Button(buttonWidget, SWT.PUSH);
		remove.setText(UIPlugin.getResourceString("RIGHT_TO_LEFT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		remove.setLayoutData(spec);

		new Label(buttonWidget, SWT.NONE);

		addAll = new Button(buttonWidget, SWT.PUSH);
		addAll.setText(UIPlugin.getResourceString("LEFT_ALL_TO_RIGHT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		addAll.setLayoutData(spec);

		removeAll = new Button(buttonWidget, SWT.PUSH);
		removeAll.setText(UIPlugin.getResourceString("RIGHT_ALL_TO_LEFT"));
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		removeAll.setLayoutData(spec);

		Label label1 = new Label(buttonWidget, SWT.PUSH);
		label1.setText("");
		spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);

		targetList = new CTree(groupWidget, targetTextID);
	}

	/**
	 * Insert the method's description here. Creation date: (11/08/2000 2:31:59
	 * PM)
	 * 
	 * @return com.ibm.swt.widgets.Control
	 */
	public Control getControl() {
		return _group;
	}

	public Object[] getSelectedItems() {
		return targetList.getSelectedItems();
	}

	public Object[] getSelectedItemsForProfiling() {
		return targetList.getItems();
	}	

	public Composite getWidget() {
		return groupWidget;
	}

	public void initializeLaunchConfiguration(ILaunchConfiguration conf) {
		_conf = conf;
	}

	public void initialize() {
		sourceList.initializeTree(_conf);

		sourceProcessList.initializeFromProcesses(sourceList.getItems());
		targetProcessList.initializeFromProcesses(targetList.getItems());
		_needToResetAgentLists = true;
	}

	private boolean listContains(Object[] objects, Object obj) {
		for (int i = 0; i < objects.length; i++) {
			if (obj.equals(objects[i]))
				return true;
		}
		return false;
	}

	public java.util.List getAttributeProcesses() {
		if (_conf != null) {
			try {
				return ProfileAttachDelegate.getAttributeProcesses(_conf);
			} catch (Exception e) {
				
			}
		}

		return new ArrayList();
	}

	private Object[] getTargetPIDList() {
		if (treesLoaded())
			return targetProcessList.getPIDs();
		else
			return (ProfileAttachDelegate.createMap(getAttributeProcesses()))
					.keySet().toArray();
	}

	private Object[] getTargetSelectedPIDList() {
		if (treesLoaded()) {
			ProfilingProcessList selectedAgents = new ProfilingProcessList();
			selectedAgents.initializeFromProcesses(getSelectedItems());
			return selectedAgents.getPIDs();
		} else
			return (ProfileAttachDelegate.createMap(getAttributeProcesses()))
					.keySet().toArray();
	}

	public boolean treesLoaded() {
		return _treesLoaded;
	}

	public void resetAgentLists() {
		if (_needToResetAgentLists
				&& (sourceList.getItemCount() == 0 || (sourceList
						.getItemCount() >= 1 && !(sourceList.getItems()[0] instanceof IWorkbenchAdapter)))) {
			_needToResetAgentLists = false;

			sourceProcessList.initializeFromProcesses(sourceList.getItems());

			Object[] PIDList = getTargetPIDList();
			Object[] selectedPIDList = getTargetSelectedPIDList();

			if (PIDList.length > 0) {
				//EC: refresh button was clicked with selected agent(s)
				//EC: this need special care as refresh button shouldn't
				// 'reset' the 2 list, but update it instead.

				Vector processesInTarget = new Vector();
				Vector dataToBeSelected = new Vector();

				for (int i = 0; i < PIDList.length; i++) {
					Object agentFromSource = sourceProcessList
							.getProcessForPID(PIDList[i].toString());
					if (agentFromSource != null) {
						processesInTarget.addElement(agentFromSource);
						sourceProcessList.removeProcess(agentFromSource);

						if (listContains(selectedPIDList, PIDList[i].toString()))
							dataToBeSelected.addElement(agentFromSource);
					}
				}

				targetProcessList.initializeFromProcesses(processesInTarget
						.toArray());

				sourceList.setRedraw(false);
				targetList.setRedraw(false);

				sourceList.resetTree(sourceProcessList);
				targetList.resetTree(targetProcessList);

				if (dataToBeSelected.size() > 0)
					targetList.setSelectionFromData(dataToBeSelected.toArray());

				sourceList.setRedraw(true);
				targetList.setRedraw(true);
			}

			_treesLoaded = true;
			changed();
		}
	}
}