/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.wizard;

import java.util.Comparator;
import java.util.List;

import org.eclipse.hyades.security.util.GridUtil;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileOverviewUI;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetType;
import org.eclipse.hyades.trace.ui.internal.util.ProfilingTypesPageContent;
import org.eclipse.hyades.trace.ui.internal.util.Sorter;
import org.eclipse.hyades.trace.ui.internal.util.TreeContentProvider;
import org.eclipse.hyades.trace.ui.internal.util.TreeLabelProvider;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetTypeGroup;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

public class ProfileSetWizardPage2  extends WizardPage implements IProfileSetWizardPage, ICheckStateListener, ISelectionChangedListener {

	private boolean pageComplete;
	private ProfilingTypesPageContent dc;
	private String profilingSetName;
	private CheckboxTreeViewer treeViewer;
	private TreeContentProvider  provider;
	private ProfilingSetsManagerCopy copy;
	
	private TraceProfileOverviewUI _overviewUI;	
	
	public ProfileSetWizardPage2(String pageId, String psName, ProfilingSetsManagerCopy managerCopy, TraceProfileOverviewUI overviewUI) {			
		super(pageId);
		profilingSetName = psName;
		copy = managerCopy;
		setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_UI_WZ_EDITPROFSET));
		_overviewUI = overviewUI;		
	}
			
	public boolean isPageComplete() {
		return pageComplete;
	}
	
	public boolean canFlipToNextPage() {
		return pageComplete;
	}
	
	public void dispose()
	{
		dc.dispose();
		super.dispose();			
	}
	
	public void createControl(Composite parent) {
		
		Composite result = (Composite) new Composite(parent, SWT.NONE);
		
		GridLayout layout;
		GridData data;

		layout = new GridLayout();
		layout.numColumns = 2;
		result.setLayout(layout);
		data = GridUtil.createFill();
		result.setLayoutData(data);
		
		data = GridUtil.createFill();
		
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		
		provider = new TreeContentProvider(_overviewUI);
		treeViewer = new CheckboxTreeViewer(result);
		treeViewer.setContentProvider(provider);
		treeViewer.setSorter(new ViewerSorter() {
			private Comparator c = new Sorter(); 
			public int compare(Viewer viewer, Object e1, Object e2) {
				return c.compare(e1, e2);
			}
		});
		
		Object[] profileTypeGroups = _overviewUI.getManagerCopy().getProfilingTypesGroups().values().toArray();
		Object[] profilingTypes = _overviewUI.getManagerCopy().getProfilingTypes().values().toArray();
		ImageDescriptor[] descriptorList = new ImageDescriptor[profileTypeGroups.length + profilingTypes.length];
		
		//profiling groups icons
		for (int i = 0; i < profileTypeGroups.length; i ++) {
			descriptorList[i] = ((IProfilingSetTypeGroup)profileTypeGroups[i]).getIcon();
		}
		
		//profiling type icons
		for (int j = 0; j < profilingTypes.length; j++)
		{
			IProfilingSetType pType = (IProfilingSetType)profilingTypes[j];
			descriptorList[j + profileTypeGroups.length] = pType.getImage();
		}
		treeViewer.setLabelProvider(new TreeLabelProvider(descriptorList));
		treeViewer.setInput("");
		treeViewer.getTree().setLayout(layout2);
		treeViewer.getTree().setLayoutData(GridUtil.createVerticalFill());
		
		IProfilingSetType[] array = new ProfilingSetType[_overviewUI.getManagerCopy().getProfilingTypes().size()];
		dc = new ProfilingTypesPageContent(result, SWT.NONE);
		dc.createContent((IProfilingSetType[])_overviewUI.getManagerCopy().getProfilingTypes().values().toArray(array), copy);
		dc.setVisible(true);
		
		treeViewer.addCheckStateListener(this);
		treeViewer.addSelectionChangedListener(this);
		
		setControl(result);
	
		setPageComplete(isPageComplete());
	}
	
	public void setVisible(boolean visible)
	{
		super.setVisible(visible);
		
		if (visible)
			updateWizardPageMsgsBtns();
	}
	
	private void setChecked(IProfilingSetType ptype)
	{
		if(ptype == null)
			return;
		
		treeViewer.expandToLevel(ptype, 0);
		treeViewer.setChecked(ptype, true);
		
		// Get all of this ProfileType's parent's children. If all of them are now checked,
		// the check our parent.
		Object parent = provider.getParent(ptype);
		Object[] siblings = provider.getChildren(parent);
		boolean foundUnchecked = false;
		for(int i = 0; i < siblings.length; i ++) {
			if (treeViewer.getChecked(siblings[i]) == false) {
				foundUnchecked = true;
			}
		}
		
		treeViewer.setChecked(parent, true);
		treeViewer.setGrayed(parent, foundUnchecked);
	}
	
	public void checkStateChanged(CheckStateChangedEvent event){
		Object checkedElement = event.getElement();

		if (checkedElement instanceof IProfilingSetType) {
			treeViewer.setSelection(new StructuredSelection(checkedElement));
			Object parent = provider.getParent(checkedElement);
			Object[] siblings = provider.getChildren(parent);
			boolean foundChecked = false;
			boolean foundUnchecked = false;
			for(int i = 0; i < siblings.length; i ++) {
				if (treeViewer.getChecked(siblings[i]) == true) {
					foundChecked = true;
				}
				else {
					foundUnchecked = true;
				}
			}
			treeViewer.setChecked(parent, foundChecked);
			treeViewer.setGrayed(parent, foundChecked && foundUnchecked);
		}
		else if (checkedElement instanceof IProfilingSetTypeGroup) {
			treeViewer.setSelection(new StructuredSelection(checkedElement));
			treeViewer.setGrayed(checkedElement, false);
			treeViewer.setSubtreeChecked(checkedElement, event.getChecked());
		}
		updateWizardPage();
	}

	public void selectionChanged(SelectionChangedEvent event) {
		updateWizardPage();
		IStructuredSelection selection = (IStructuredSelection) event.getSelection();
		if (selection.getFirstElement() instanceof IProfilingSetType) {
			updateGui((IProfilingSetType)selection.getFirstElement());
		} 
		else if (selection.getFirstElement() instanceof IProfilingSetTypeGroup){
			dc.showGroupDetails((IProfilingSetTypeGroup)selection.getFirstElement());
		}else {
			dc.showEmptyDetails();
		}
	}
	

	private void updateGui(IProfilingSetType newProfileType) {
		Control newControl = newProfileType.getProfilingType().getControl(profilingSetName);
		Control currentControl = dc.getCurrentlyDisplayingControl();
		if (newControl.equals(currentControl) == false) {
			dc.showDetailsFor(newControl);
		}
	}
	
	
	private void updateWizardPageMsgsBtns()
	{
		if (!pageComplete)
			setErrorMessage(UIPlugin.getResourceString("STR_EDIT_PROFILINGSET_DESC_NOT_SELECTED"));
		else
			setErrorMessage(null);
		
		getWizard().getContainer().updateButtons();				
	}

	/**
	 * Updates the Next/Back buttons on the wizard.
	 * The buttons will be enabled if one of the following conditions are true:
	 * - if at least one of the checked elements use filters.
	 * - if the first selected valid element uses filters.
	 *
	 * Otherwise, the buttons will be disabled.
	 */
	private void updateWizardPage() {
		Object[] checkedElementList = treeViewer.getCheckedElements();

		// Get the list of checked elements, then filter out all of the ones that aren't
		// ProfileType objects. For each of the ones that ARE profile type objects, if any
		// of them have filters turned on, then enable the next button and return.
		for(int i = 0; i < checkedElementList.length; i ++) {
			if (checkedElementList[i] instanceof IProfilingSetType) {
				IProfilingSetType nextProfile = (IProfilingSetType) checkedElementList[i];
				if (nextProfile.isUseFilters()) {
					pageComplete = nextProfile.isUseFilters();
					updateWizardPageMsgsBtns();
					return;
				}
			}
		}

		// This condition has been commented out as a result of bugzilla 56785.
		
		// If none of the checked elements have filters turned on, then check that the currently
		// selected element has filters turned on. If it does, then enable the next button. Otherwise,
		// disable the next button.
//		IStructuredSelection selection = (IStructuredSelection) treeViewer.getSelection();
//		Iterator i = selection.iterator();
//		while(i.hasNext() == true) {
//			Object currentIteration = i.next();
//			if (currentIteration instanceof IProfilingSetType) {
//				IProfilingSetType firstSelectedProfile = (IProfilingSetType) currentIteration;
//				pageComplete = firstSelectedProfile.isUseFilters();
//				getWizard().getContainer().updateButtons();
//				return;
//			}
//		}
		
		// If none of the conditions hold, disable the buttons.
		pageComplete = false;
		updateWizardPageMsgsBtns();
	}		
	
	public void initializeFrom(ProfilingSetsManagerCopy copy) {
		IProfilingSet set = copy.getDefaultSet();
		if(set == null)
		    return;
		
		List types = set.getProfilingTypes();
		for(int idx=0; idx<types.size(); idx++)
		{
			String type = types.get(idx).toString();
			
			Object ptype = copy.getProfilingTypes().get(type);
			if(type != null)
				setChecked((IProfilingSetType)ptype);
		}
		
		/*
		 * #69174: Select the topmost checked profiling type
		 * and show its options.
		 */
		Object[] checked = treeViewer.getCheckedElements();
		if (checked != null) {
			for (int i=0;i<checked.length;++i) {
				if (checked[i] instanceof IProfilingSetType) {
					ISelection selection = new StructuredSelection(checked[i]);
					treeViewer.setSelection(selection, true);
					break;
				}
			}
		}
		
		/*
		 * #60091: Send the getControl() signal consistently whenever
		 * the profiling type's UI is shown.
		 */
		IStructuredSelection sel = (IStructuredSelection)treeViewer.getSelection();
		if (sel != null && !sel.isEmpty()) {
			Object obj = sel.getFirstElement();
			if (obj != null && obj instanceof IProfilingSetType) {
				IProfilingType type = ((IProfilingSetType)obj).getProfilingType();
				if (type != null) {
					type.getControl(profilingSetName);
				}
			}
		}
	}
	
	public void setProfilingSetName(String name)
	{
		profilingSetName = name;
	}
	
	public boolean performApply(ProfilingSetsManagerCopy copy)
	{
		//update profiling types
		IProfilingSet set = copy.getDefaultSet();
		
		set.getProfilingTypes().clear();
		Object[] selections = treeViewer.getCheckedElements();
		for(int idx=0; idx<selections.length; idx++)
		{
			Object type = selections[idx];
			if(type != null && type instanceof IProfilingSetType)
			{
				set.getProfilingTypes().add(((IProfilingSetType)type).getId());
			}
		}
		
		return dc.performApply(copy);
	}
}
