package org.eclipse.hyades.logging.commons;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogConfigurationException;
import org.apache.commons.logging.LogFactory;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Implementation of the Apache Commons
 * <code>org.apache.commons.logging.LogFacotry</code> abstract class which represents a
 * proxy factory specific for <code>org.eclipse.hyades.logging.commons.Logger</code> instances.
 * <p>
 * This proxy factory is intended for use in multiple class loader environments where  
 * one or more different types of loggers and their factories are required for each 
 * separate class loader, all utilizing the same Apache Commons binaries.  Alternatively, 
 * users may configure the default <code>org.apache.commons.logging.LogFactory</code> implementation
 * to use the generate <code>org.eclipse.hyades.logging.commons.Logger</code> instances but all 
 * class loaders will be forced to use the <code>org.eclipse.hyades.logging.commons.Logger</code> instances.
 * <p>
 * An instance of this class will be returned from the static <code>getFactory()</code>
 * API on the default <code>org.apache.commons.logging.LogFactory</code> implementation. The
 * <code>org.apache.commons.logging.LogFactory</code> implementation uses the
 * following discovery process to resolve the configured
 * <code>org.apache.commons.logging.LogFactory</code> implementation within the caller's class
 * loader:
 * <p>
 * 1) Look for a system property named
 * <code>org.apache.commons.logging.LogFactory</code>.
 * <p> 
 * For example,
 * 
 * <code>...-Dorg.apache.commons.logging.LogFactory=org.eclipse.hyades.logging.commons.LoggerFactory...</code>
 * 
 * - or -
 * 
 * <code>System.setProperty("org.apache.commons.logging.LogFactory","org.eclipse.hyades.logging.commons.LoggerFactory");</code>
 * <p>
 * 2) Use the JDK 1.3 JAR Services Discovery mechanism (see http://java.sun.com/j2se/1.3/docs/guide/jar/jar.html)
 * to look for a resource named <code>META-INF/services/org.apache.commons.logging.LogFactory</code> whose first 
 * line is assumed to contain the desired class name.
 * <p> 
 * For example (<code>META-INF/services/org.apache.commons.logging.LogFactory</code>),
 * 
 * <code>org.eclipse.hyades.logging.commons.LoggerFactory</code>
 * <p>
 * 2) Look for a properties file named <code>commons-logging.properties</code> visible in the application class path, 
 * with a property named <code>org.apache.commons.logging.LogFactory</code> defining the desired implementation class name. 
 * <p> 
 * For example,
 * 
 * <code>org.apache.commons.logging.LogFactory=org.eclipse.hyades.logging.commons.LoggerFactory</code>
 * <p>
 * NOTE: Although this proxy factory provides functionality for setting/getting/removing configuration attributes, 
 * no configuration attributes are used internally.  As such, setting any configuration attributes will have no effect
 * aside from containment purposes.
 * <p>
 * NOTE: The Apache Commons logging classes must be on the CLASSPATH at run-time
 * to utilize this logging class
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @author Eric Labadie
 * @version August 13, 2004
 * @since August 13, 2004
 * @see org.apache.commons.logging.LogFactory
 * @see org.eclipse.hyades.logging.commons.Logger
 */
public class LoggerFactory extends LogFactory {

    /**
     * Private store of configuration attributes for this
     * <code>org.eclipse.hyades.logging.commons.LoggerFactory</code> instance.
     */
    private Hashtable attributes = null;

    /**
     * Private store of <code>org.eclipse.hyades.logging.commons.Logger</code> instances.
     */
    private Hashtable loggers = null;

    /**
     * No-argument constructor.
     */
    public LoggerFactory() {

        attributes = new Hashtable();

        loggers = new Hashtable();
    }

    /**
     * Set a configuration attribute with the specified non-null name.
     * <p>
     * Passing a <code>null</code> name results in no action.
     * <p>
     * Passing a <code>null</code> value is equivalent to calling
     * <code>#removeAttribute(String)</code>.
     * <p>
     * 
     * @param name
     *            The non-null name of the configuration attribute.
     * @param value
     *            The value of the configuration attribute, or <code>null</code>
     *            to remove the named configuration attribute.
     * @see org.apache.commons.logging.LogFactory#setAttribute(java.lang.String,
     *      java.lang.Object)
     */
    public void setAttribute(String name, Object value) {

        if (name != null) {

            if (value == null) {
                attributes.remove(name);
            } else {
                attributes.put(name, value);
            }
        }
    }

    /**
     * Removes a configuration attribute with the specified non-null name.
     * <p>
     * Passing a <code>null</code> name results in a no-operation.
     * <p>
     * 
     * @param name
     *            The non-null name of the configuration attribute to be
     *            removed.
     * @see org.apache.commons.logging.LogFactory#removeAttribute(java.lang.String)
     */
    public void removeAttribute(String name) {

        if (name != null) {
            attributes.remove(name);
        }
    }

    /**
     * Returns the configuration attribute with the specified non-null name, or
     * <code>null</code> if no such named attribute exists.
     * <p>
     * Passing a <code>null</code> name results in a <code>null</code>
     * return value.
     * <p>
     * 
     * @param name
     *            Name of the requested configuration attribute.
     * @return The requested configuration attribute, or <code>null</code> if
     *         no such named attribute exists
     * @see org.apache.commons.logging.LogFactory#getAttribute(java.lang.String)
     */
    public Object getAttribute(String name) {

        if (name != null) { return (attributes.get(name)); }

        return null;
    }

    /**
     * Returns an array containing the names of all currently defined
     * configuration attributes. If there are no configuration attributes, a
     * zero length array is returned.
     * <p>
     * 
     * @return The array of configuration attribute names, otherwise a zero
     *         length array.
     * @see org.apache.commons.logging.LogFactory#getAttributeNames()
     */
    public String[] getAttributeNames() {

        Set attributeNames = attributes.keySet();

        return ((String[]) (attributeNames.toArray(new String[attributeNames.size()])));
    }

    /**
     * Convenience API which returns an instance of a named
     * <code>org.eclipse.hyades.logging.commons.Logger</code> based on the
     * name of the parameter class instance.
     * <p>
     * An instance of a named
     * <code>org.eclipse.hyades.logging.commons.Logger</code> is created if no
     * named instance current exists or all instances have been released.
     * <p>
     * Once a named <code>org.eclipse.hyades.logging.commons.Logger</code> is
     * created, the instance is cached for future calls to retrieve the same
     * named <code>org.eclipse.hyades.logging.commons.Logger</code>.
     * <p>
     * The name of the <code>org.eclipse.hyades.logging.commons.Logger</code>
     * uniquely identifies an instance of an
     * <code>org.eclipse.hyades.logging.commons.Logger</code>. All subsequent
     * calls will return the same instance of the named
     * <code>org.eclipse.hyades.logging.commons.Logger</code>.
     * <p>
     * 
     * @param classInstance
     *            Class instance used to derive the name of the returned
     *            <code>org.eclipse.hyades.logging.commons.Logger</code>
     *            instance.
     * @return A named <code>org.eclipse.hyades.logging.commons.Logger</code>
     *         instance.
     * @exception LogConfigurationException
     *                if the named
     *                <code>org.eclipse.hyades.logging.commons.Logger</code>
     *                instance could not be created.
     * @see org.apache.commons.logging.LogFactory#getInstance(java.lang.Class)
     * @see org.eclipse.hyades.logging.commons.LoggerFactory#getInstance(String)
     */
    public Log getInstance(Class classInstance) throws LogConfigurationException {
        return getInstance(classInstance.getName());
    }

    /**
     * Returns an instance of a non-null named
     * <code>org.eclipse.hyades.logging.commons.Logger</code> based on the
     * parameter name.
     * <p>
     * Passing a <code>null</code> logger name results in a <code>null</code>
     * return value.
     * <p>
     * An instance of a named
     * <code>org.eclipse.hyades.logging.commons.Logger</code> is created if no
     * named instance current exists or all instances have been released.
     * <p>
     * Once a named <code>org.eclipse.hyades.logging.commons.Logger</code> is
     * created, the instance is cached for future calls to retrieve the same
     * named <code>org.eclipse.hyades.logging.commons.Logger</code>.
     * <p>
     * The name of the <code>org.eclipse.hyades.logging.commons.Logger</code>
     * uniquely identifies an instance of an
     * <code>org.eclipse.hyades.logging.commons.Logger</code>. All subsequent
     * calls will return the same instance of the named
     * <code>org.eclipse.hyades.logging.commons.Logger</code>.
     * <p>
     * 
     * @param loggerName
     *            The non-null name of the returned
     *            <code>org.eclipse.hyades.logging.commons.Logger</code>
     *            instance.
     * @return A named <code>org.eclipse.hyades.logging.commons.Logger</code>
     *         instance, otherwise <code>null</code> if the parameter logger
     *         name is <code>null</code>.
     * @exception LogConfigurationException
     *                if the named
     *                <code>org.eclipse.hyades.logging.commons.Logger</code>
     *                instance could not be created.
     * @see org.apache.commons.logging.LogFactory#getInstance(java.lang.String)
     */
    public Log getInstance(String loggerName) throws LogConfigurationException {

        if (loggerName != null) {

            if (loggers.containsKey(loggerName)) { return (((Log) (loggers.get(loggerName)))); }

            Logger logger = new Logger(loggerName);

            loggers.put(loggerName, logger);

            return logger;
        }

        return null;
    }

    /**
     * Releases all cached
     * <code>org.eclipse.hyades.logging.commons.Logger</code> instances
     * created by this
     * <code>org.eclipse.hyades.logging.commons.LoggerFactory</code> instance.
     * <p>
     * All cached <code>org.eclipse.hyades.logging.commons.Logger</code> are
     * de-registered (e.g.
     * <code>org.eclipse.hyades.logging.commons.Logger#finalize()</code>
     * before being released.
     * <p>
     * 
     * @see org.apache.commons.logging.LogFactory#release()
     */
    public void release() {

        Iterator loggerNames = loggers.keySet().iterator();

        while (loggerNames.hasNext()) {
            ((Logger) (loggers.remove(loggerNames.next()))).finalize();
        }
    }
}