/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.sample.loganalyzer;

import java.lang.reflect.InvocationTargetException;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.logging.parsers.ApacheAccessLogParser;
import org.eclipse.hyades.logging.parsers.ApacheErrorLogParser;
import org.eclipse.hyades.logging.parsers.Parser;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.sample.HyadesSamplePlugin;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.internal.util.CoreUtil;
import org.eclipse.hyades.ui.internal.util.StringUtil;
import org.eclipse.hyades.ui.internal.wizard.exampleproject.CreationWizardPage;
import org.eclipse.hyades.ui.internal.wizard.exampleproject.ZipAdjusterCreationWizard;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;

/**
 * This wizard class creates two pages that specifes the project to create the sample files
 * and the error log file, access log file and symptom database used in the log analyzer sample.
 * 
 * @author sleeloy
 */
public class LogAnalyzerCreationWizard 
extends ZipAdjusterCreationWizard
{
	LogAnalyzerCreationPage pageOne = null;
	LogAnalyzerSamplePage pageTwo = null;
	
	private ResourceBundle resourceBundle;
	
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.ZipAdjusterCreationWizard#addZipExtensions()
	 */
	protected void addZipExtensions()
	{
		addZipExtension("trcpxmi");
		addZipExtension("trcaxmi");
		addZipExtension("trcdbxmi");
		
		setResourceBundle(HyadesSamplePlugin.getInstance().getResourceBundle());
	}
	
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizard#logError(java.lang.String)
	 */
	protected void logError(String error)
	{
		HyadesSamplePlugin.logError(error);
	}
	/**
	 * @see com.ibm.etools.ac.samples.internal.wizard.exampleproject.CreationWizard#createWizardPage(java.lang.String, org.eclipse.core.runtime.IConfigurationElement)
	 */
	protected CreationWizardPage createWizardPage(
		String suggestedName,
		IConfigurationElement configurationElement) {
		if (suggestedName.equals("newProject1")){
			pageTwo = new LogAnalyzerSamplePage(this, suggestedName, configurationElement, pageOne);
			return pageTwo;
		}
		else{
			pageOne = new LogAnalyzerCreationPage(this, suggestedName, configurationElement);
			return pageOne;
		}
	}

	/**
	 * This method correlates an apache log file based on time.  The aco parameter
	 * contains the agent that represents the access log. 
	 * 
	 * @param aco - the agent that represents the access log.
	 * @return true if correlation was sucessful, otherwise false is returned.
	 */
	public boolean correlateAccessLog(AnalyzeCorrelateOperation aco ){
//		EObject emfObject = aco.getAgent().getAgentProxy();
		
		CorrelateLogAction runnable = null;
//			
//		runnable = new CorrelateLogAction();
//		runnable.setMofObject(emfObject);
//		
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(runnable);
//		try
//		{
//			getContainer().run(false, true, op);
//		}
//		catch (InvocationTargetException e)
//		{
//			handleException(e.getTargetException());
//			return false;
//		}
//		catch (InterruptedException e)
//		{
//			return false;
//		}
//		runnable.dispose();
		
//		//correlate monitor
		TRCMonitor monitor = aco.getTRCMonitor();
		runnable = new CorrelateLogAction();
		runnable.setMofObject(monitor);
		
		op = new WorkspaceModifyDelegatingOperation(runnable);
		try
		{
			getContainer().run(false, true, op);
		}
		catch (InvocationTargetException e)
		{
			handleException(e.getTargetException());
			return false;
		}
		catch (InterruptedException e)
		{
			return false;
		}
		runnable.dispose();
//		
//		Iterator iter = monitor.getNodes().iterator();
//		while (iter.hasNext()){
//			emfObject = (EObject)iter.next();
//			runnable = new CorrelateLogAction();
//			runnable.setMofObject(emfObject);
//		
//			op = new WorkspaceModifyDelegatingOperation(runnable);
//			try
//			{
//				getContainer().run(false, true, op);
//			}
//			catch (InvocationTargetException e)
//			{
//				handleException(e.getTargetException());
//				return false;
//			}
//			catch (InterruptedException e)
//			{
//				return false;
//			}
//			runnable.dispose();
//			
//		}
			
		
		return true;
		
	}		
	
	/**
	 * This method analyzes an apache error log file based on time.  The aco parameter
	 * contains the agent that represents the error log. 
	 * 
	 * @param aco - the agent that represents the error log.
	 * @return true if analysis was sucessful, otherwise false is returned.
	 */
	public boolean analyzeErrorLog(String projectName, String fileName, String symptomDBPath, AnalyzeCorrelateOperation aco  ){
		
		AnalyzeLogAction runnable = null;
		EObject emfObject = aco.getAgent().getAgentProxy();
			
		runnable = new AnalyzeLogAction();
		runnable.setSymptomDBPath(symptomDBPath);
		runnable.setProjectName(projectName);
		runnable.setSDBFileName(fileName);
		runnable.setMofObject(emfObject);
		
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(runnable);
		try
		{
			getContainer().run(false, true, op);
		}
		catch (InvocationTargetException e)
		{
			handleException(e.getTargetException());
			return false;
		}
		catch (InterruptedException e)
		{
			return false;
		}
									
		return true;
		
	}		

   /**
	* This method parses a given log file and returns a AnalyzeCorrelateOperation
	* object that represents the parsed log file. 
    * 
    * @param logFile - the path of the log file to parse.
    * @param parser - the parser to use to parse the log file.
    * @param name - the name of the imported log file
    * @return AnalyzeCorrelateOperation that represents the parsed log file.
    */
    public AnalyzeCorrelateOperation importLogFile(String logFile, Parser parser, String name){
		INavigator fViewer = null;
		AnalyzeCorrelateOperation runnable = null;
			
		try {
			IWorkbenchPage page = UIPlugin.getActivePage();

			fViewer = ((INavigator) (page.showView(PDPerspective.ID_LOG_NAVIGATOR_VIEW)));
			
		} catch (Exception e) {
//			errorMessage = LogUIPlugin.getResourceString("IMPORT_LOG_FILE_DEST_ERROR_");

			return null;
		}
			
		runnable = new AnalyzeCorrelateOperation(parser, name);
		runnable.setFViewer(fViewer);
		runnable.setLogFilePath(logFile);
		runnable.setProjectName(pageOne.getProjectName());
		runnable.setTcMonitor("LogCorrelationAnalyzerSample_TCMonitor");
		
		IRunnableWithProgress op = new WorkspaceModifyDelegatingOperation(runnable);
		try
		{
			getContainer().run(false, true, op);
		}
		catch (InvocationTargetException e)
		{
			handleException(e.getTargetException());
			return null;
		}
		catch (InterruptedException e)
		{
			return null;
		}
		runnable.dispose();
		return runnable;
    }
	/**
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */
	protected boolean finishWork() {
		
			AnalyzeCorrelateOperation acoLog = importLogFile(pageTwo.getLogPath(), new ApacheAccessLogParser(), HyadesSamplePlugin.getString("LOG_FILE_LABEL"));
			
			AnalyzeCorrelateOperation acoErrorLog = importLogFile(pageTwo.getErrorLogPath(), new ApacheErrorLogParser(), HyadesSamplePlugin.getString("ERRORLOG_FILE_LABEL"));
			if (pageTwo.getAnalyzeLogButton().getSelection())
				analyzeErrorLog(pageTwo.getProjectName(), pageTwo.getSDBFileName(), pageTwo.getSymptomDBFilePath(), acoErrorLog);				
	
			if (pageTwo.getCorrelateLogButton().getSelection()){
				correlateAccessLog(acoLog);
			}
			return true;
	}
	
	/**
	 * Displays an error message box given a specific exception.
	 * 
	 * @param throwable
	 */
	private void handleException(Throwable throwable)
	{
		logError(CoreUtil.getStackTrace(throwable));

		String[] options = {IDialogConstants.OK_LABEL};
		MessageDialog dialog = new MessageDialog(getShell(), getString("W_ERROR"), null, getString("ERR_UNABLE_CREATE"), MessageDialog.ERROR, options, 0);
	}
	
	/**
	 * Returns the value associated with a key in the resource bundle of this
	 * wizard's plugin definition.  If it is not possible to retrieve the value then 
	 * the key is returned.
	 * 
	 * <p>Subclasses may overwrite.
	 *  
	 * @param key
	 * @return String.
	 */
	protected String getString(String key)
	{
		if(resourceBundle != null)
		{
			try
			{
				return resourceBundle.getString(key);
			}
			catch (MissingResourceException e)
			{
			}
		}
			
		return key;
	}
	
	/**
	 * Returns the string value associate to a given key.  The strings defined 
	 * in <code>args</code> replaces the %n (where n>=1) variables defined in the 
	 * file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 */
	protected String getString(String key, String[] args)
	{
		String ret = getString(key);
		for (int i = args.length; i > 0; i--)
			ret = StringUtil.replace(ret, ("%"+i), args[i-1]);
		
		return ret;
	}
	
	
	/**
	 * Returns the resource bundle used by this wizard.
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		return resourceBundle;
	}

	/**
	 * Sets the resource bundle to be used by this wizard
	 * @param bundle
	 */
	public void setResourceBundle(ResourceBundle bundle)
	{
		resourceBundle = bundle;
	}


	/**
	 * @return
	 */
	public LogAnalyzerSamplePage getPageTwo() {
		return pageTwo;
	}

}
