/*******************************************************************************
 * Copyright (c) 2003, 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.datapool.internal.dialog;

import java.text.MessageFormat;
import java.text.NumberFormat;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolVariable;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolSuggestedType;
import org.eclipse.hyades.test.ui.datapool.DatapoolPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.DatapoolUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

/**
 * Dialog to collect information about a variable/column that is being
 * inserted or edited by the datapool table.  
 * 
 * @author psun
 *
 */
public class DatapoolColumnDialog extends Dialog implements KeyListener, SelectionListener{

	private static final String TAG_VARIABLES = "variables"; //$NON-NLS-1$
	
	private static final int HEIGHT = 120;
	private static final int WIDTH = 300;
	private Shell shell = null;
	private IDatapool datapool = null;
	private String title = null;
	private IDatapoolVariable selectedVariable = null;
	private IDatapoolVariable previousVariable = null;

	private Text nameField = null;
	private Text typeField = null;
	private Color nameFieldForeground = null;
	private Color typeFieldForeground = null;
	private Combo insertionVariables = null;
	private Label nameErrorLabel;
	private Label typeErrorLabel;

	private String name = null;
	private String type = new String();
	private String insertionVariableID = null;
	private String insertionVariableName = null;
	private int defaultNameCounter = 1;
	
	public DatapoolColumnDialog(Shell parentShell, 
									     IDatapool datapool,
									     IDatapoolVariable selectedVariable,
									     IDatapoolVariable previousVariable,
									     String title)
	{
		super(parentShell);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
		this.shell = parentShell;
		this.datapool = datapool;
		this.title = title; 
		this.selectedVariable = selectedVariable;
		this.previousVariable = previousVariable;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent)
	{
		Control control = super.createContents(parent);
		enableOK();
		return control;
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent)
	{
		getShell().setText(title);
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		superComposite.setLayoutData(gridData);		

		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		superComposite.setLayout(gridLayout);

		Label nameLabel = new Label(superComposite, SWT.NONE);
		nameLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_NAME")); //$NON-NLS-1$

		nameField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		nameField.setLayoutData(gridData);
		nameField.addKeyListener(this);
		nameFieldForeground = nameField.getForeground();

		Label typeLabel = new Label(superComposite, SWT.NONE);
		typeLabel.setText(DatapoolPlugin.getResourceString("DATA_COL_DLG_TYPE")); //$NON-NLS-1$

		typeField = new Text(superComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		typeField.setLayoutData(gridData);
		typeField.addKeyListener(this);
		typeFieldForeground = typeField.getForeground();
	
		IDatapoolVariable[] variables = DatapoolUtil.getInstance().getVariables(datapool, null);
		if(selectedVariable != null)
		{
			name = selectedVariable.getName();
			nameField.setText(name);
			nameField.selectAll();
			IDatapoolSuggestedType suggestedType = selectedVariable.getSuggestedType();
			if(suggestedType != null)
			{
				type = suggestedType.getSuggestedClassName();
				if(type != null)
				{
					typeField.setText(type);
					typeField.setSelection(type.length() + 1);
				}
			}
		}
		String[] variableLabels = getVariableLabels(variables);

		Label insertionLabel = new Label(superComposite, SWT.NONE);
		if(title.equals(DatapoolPlugin.getResourceString("DATA_COL_DLG_TITLE_INS"))) //$NON-NLS-1$
		{
			insertionLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_INS")); //$NON-NLS-1$
			name = createDefaultName();
			while(!DatapoolUtil.getInstance().isVariableNameUnique(datapool, name, null))
			{
				name = createDefaultName();
			}
			nameField.setText(name);
			nameField.selectAll();
		}
		else
			insertionLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_MOVE")); //$NON-NLS-1$

		insertionVariables = new Combo(superComposite, SWT.DROP_DOWN | SWT.READ_ONLY);
		insertionVariables.setItems(variableLabels);
		insertionVariables.setData(TAG_VARIABLES, variables);
	    insertionVariables.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		insertionVariables.addSelectionListener(this);
		
		if(variables.length > 0 && previousVariable != null)
		{
			int selectIndex = findVariableIndexInCombo(variables, previousVariable);
			insertionVariables.select(selectIndex + 1);
			insertionVariableName = variables[selectIndex].getName();
			insertionVariableID = variables[selectIndex].getId();
		}
		else
		{
			insertionVariables.select(0);
			insertionVariableName = new String();
			insertionVariableID = new String();
		}
		
		nameErrorLabel = new Label(superComposite, SWT.NONE);
		// set text to error labels in order to get the proper layout. 
		nameErrorLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_ERROR_INDICATOR") +
								DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_NAME_NOT_VALID")); //$NON-NLS-1$
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 2;
		nameErrorLabel.setLayoutData(gridData);
		nameErrorLabel.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
		nameErrorLabel.setVisible(false);

		typeErrorLabel = new Label(superComposite, SWT.NONE);
		typeErrorLabel.setText(DatapoolPlugin.getResourceString("DATA_DLG_ERROR_INDICATOR") + 
								DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_TYPE_NOT_VALID")); //$NON-NLS-1$
		gridData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		gridData.horizontalSpan = 2;
		typeErrorLabel.setLayoutData(gridData);
		typeErrorLabel.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
		typeErrorLabel.setVisible(false);

		superComposite.pack();
		return superComposite;
	}

	private boolean enableOK()
	{
		boolean enableButton = true;
		String nameErrorText = new String();
		Display display= Display.getDefault();
		Color color= display.getSystemColor(SWT.COLOR_RED);
	
		String type = typeField.getText();
		if(type != null)
		{
			boolean isTypeValid = DatapoolUtil.getInstance().isVariableTypeValid(type);
			if(!isTypeValid)
			{
				enableButton = false;				
				typeField.setForeground(color);				
			}
			else
			{				
				typeField.setForeground(typeFieldForeground);
			}			
			typeErrorLabel.setVisible(!enableButton);
		}
		
		String name = nameField.getText();
		if(name == null || name.length() == 0)
		{
			enableButton = false;
			nameErrorLabel.setVisible(false);
		}
		if(name != null && name.length() > 0)
		{
			boolean isNameUnique = DatapoolUtil.getInstance().isVariableNameUnique(datapool, name, selectedVariable);
			boolean isNameValid = DatapoolUtil.getInstance().isVariableNameValid(name);
			if(isNameUnique && isNameValid)
			{		
				nameField.setForeground(nameFieldForeground);
				nameErrorLabel.setVisible(false);
			}
			else
			{
				enableButton = false;				
				nameField.setForeground(color);
				nameErrorText = DatapoolPlugin.getResourceString("DATA_DLG_ERROR_INDICATOR");
				if(!isNameUnique)
				{
					nameErrorText += DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_NAME_NOT_UNIQUE"); //$NON-NLS-1$
				}
				else if(!isNameValid)
				{
					nameErrorText += DatapoolPlugin.getResourceString("DATA_COL_DLG_ERROR_NAME_NOT_VALID"); //$NON-NLS-1$
				}
				nameErrorLabel.setText(nameErrorText);
				nameErrorLabel.setVisible(true);
			}
		}
		
		getButton(IDialogConstants.OK_ID).setEnabled(enableButton);
		return enableButton;
	}
	
	private String[] getVariableLabels(IDatapoolVariable[] variables)
	{
		if(variables == null)
			return null;
		int listSize = variables.length;
		String[] returnList = new String[listSize + 1];
		String insertLabelPattern = DatapoolPlugin.getResourceString("DATA_DLG_INSERT_VARIABLE_FORMAT");
		for(int i = 0; i < listSize; i++)
		{
			IDatapoolVariable variable = variables[i];
			if(variable == null)
				continue;
			String name = variable.getName();			
			if(i == 0)
			{
				Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_BEFORE"), name}; //$NON-NLS-1$
				returnList[i] = MessageFormat.format(insertLabelPattern, messageElements);  //$NON-NLS-1$
			}
			Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_DLG_AFTER"), name}; //$NON-NLS-1$
			returnList[i + 1] = MessageFormat.format(insertLabelPattern, messageElements); //$NON-NLS-1$
		}
		if(listSize == 0)
		{
			returnList[0] = DatapoolPlugin.getResourceString("DATA_DLG_FIRST_ELEMENT"); //$NON-NLS-1$
		}
		return returnList;
	}

	public String getName()
	{
		return name;
	}

	public String getType()
	{
		return type;
	}
	
	public String getInsertionVariableName()
	{
		return insertionVariableName;
	}

	public String getInsertionVariableID()
	{
		return insertionVariableID;
	}

	private int findVariableIndexInCombo(IDatapoolVariable[] variables, IDatapoolVariable variable)
	{
		if(variables == null || variable == null)
			return -1;
		for(int i = 0; i < variables.length; i++)
		{
			if(variables[i].equals(variable))
				return i;	
		}
		return -1;
	}

	// ISelectionListener

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		Combo insertVariableCombo = (Combo)e.widget;
		int selectionIndex = insertVariableCombo.getSelectionIndex();
		IDatapoolVariable[] variables = (IDatapoolVariable[])insertVariableCombo.getData(TAG_VARIABLES);
		if(selectionIndex > 0 && selectionIndex <= variables.length)
		{
			IDatapoolVariable variable = variables[selectionIndex - 1];
			insertionVariableName = variable.getName();
			insertionVariableID = variable.getId();
		}
		else
		{
			insertionVariableName = new String();
			insertionVariableID = new String();
		}
	}

	/* 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) 
	{
		return;	
	}

	// IKeyListener

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) 
	{		
		return;
	}

	/* 
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) 
	{
		if(nameField != null)
			name = nameField.getText();
		if(typeField != null)
			type = typeField.getText();
		enableOK();
	}

	private String createDefaultName()
	{
		Object[] messageElements = {DatapoolPlugin.getResourceString("DATA_VARIABLE_NAME"), NumberFormat.getInstance().format(defaultNameCounter)}; //$NON-NLS-1$
		defaultNameCounter++;
		return MessageFormat.format(DatapoolPlugin.getResourceString("DATA_VARIABLE_NAME_FORMAT"), messageElements); //$NON-NLS-1$
	}	
}
