/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.logging.adapter.Adapter;
import org.eclipse.hyades.logging.adapter.AdapterException;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.core.SerializationException;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.parsers.LogParserException;
import org.eclipse.hyades.logging.parsers.Parser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * This class is used as in interface between the UI and the generic adapter.
 * The UI side is expecting an implementor of IParser and will call setUserInput() and parse(log).
 * This class will implement the plumbing to map the two methods above to those in the generic adapter.
 */
public class StaticParserWrapper extends Parser {
	private Adapter adapter;
	private String parserName;
	private String agentName;
	private String key;
	private String originalConfigFile = null;
	
//	protected Plugin thisPlugin;
//	protected ResourceBundle resourceBundle;
	protected String newDirectory = null;
	protected String newFileName = null;
	protected String currentPlugin;

	// Hash table to store all active loggers
	private static Hashtable activeWrappers = new Hashtable();

	public StaticParserWrapper() {
		// Generate a key for this instance
		key = String.valueOf(System.currentTimeMillis());
		adapter = new Adapter();
		parserName = new String(Messages.getString("HyadesGAUnknownNameTag"));
		agentName = new String(Messages.getString("HyadesGAUnknownNameTag"));

		currentPlugin = "org.eclipse.hyades.logging.adapter.config";
	}

	public static Log getLogInstance(String logKey) {
		return (Log)activeWrappers.remove(logKey);
	}
	
	/**
	 * @see org.eclipse.hyades.logging.parsers.IParser.parse(org.apache.commons.logging.Log)
	 */
	public void parse(Log argLogger) throws LogParserException {
		try {
			activeWrappers.put(key, argLogger);
			adapter.start(false,false);
		} catch (AdapterException e) {
		}
	}

	public void setUserInput(Hashtable table) throws LogParserException {
		// Pass the instance key to the parser
		table.put("loggerKey", key);

		String newConfigFile = getNewConfigFile(table);
		adapter.setContextConfigPath(newConfigFile);
		adapter.setComponentConfigPath(newConfigFile);
	}
	
	/**
	 * Since this is not a real parser we don't need to implement the getName() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.getName()
	 */
	public String getName() {

		return agentName;
	}

	// 
	/**
	 * Since this is not a real parser we don't need to implement the parseNext() method
	 * 
	 * @see org.eclipse.hyades.logging.parsers.IParser.parseNext()
	 */
	public CommonBaseEvent[] parseNext() throws LogParserException {
		return null;
	}

	// Since this is not a real parser we don't need to implement the getVersion() method
	public String getVersion() {
		return null;
	}

	/**
	 * Method to get an input stream to the config file
	 * @param fileName - config file name
	 * @param table - table of user input parameters
	 * @return InputStream
	 * @throws LogParserException 
	 */	
	private InputStream getConfigInputStream(String fileName, Hashtable table) throws LogParserException
	{
		String fileSeparator = System.getProperty("file.separator");
		// Get input stream for config file relative to the current directory
		try {
			// If we have the wrong file separator in the config file path then
			if (fileName.indexOf('/') != -1 && fileSeparator != "/") {
				// Replace file separators
				fileName = fileName.replace('/', fileSeparator.charAt(0));
			}
			// Remove ./
			if (fileName.startsWith("." + fileSeparator)) {
				fileName = fileName.substring(2);
			}
			return new FileInputStream(fileName);
		} catch (FileNotFoundException exc) {
			// The user may have specified a location for the config file so use it to try to find the config file
			String fileName2 = (String) table.get(Messages.getString("HyadesGAConfigFileRootAttributeName")) + fileSeparator + fileName;
			try {
				// Since we have a new config file path change the global variable
				originalConfigFile = fileName2;
				return new FileInputStream(fileName2);
			}  catch (FileNotFoundException exc2) {
				//
				// Bug 62317
				// Try to resolve the config file using the command line list of directories
				//
				String config_paths = (String)table.get("config_path"); // get the paths containing the adapter files
				// Check to ensure config_path was set in the input table
				// If it is not then throw an appropriate exception.
				if (config_paths != null && config_paths.length() != 0) {
					StringTokenizer strtok = new StringTokenizer(config_paths, File.pathSeparator, false); // tokenize the path
					while(strtok.hasMoreTokens()) {
						String config_path = strtok.nextToken();
						File f = new File(config_path, fileName); // look for the adapter file using the path
						if(f.exists()) {
							try {
								table.remove("config_path"); // This is no longer required
								return new FileInputStream(f.getAbsolutePath());
							} catch (FileNotFoundException e) {
								// Should not hit here since we've already checked the existance
							}
						}
					}
				}
				// Bug 62317 ends

				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_", fileName));
			} catch (SecurityException exc2) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", fileName));
			}
		} catch (SecurityException exc) {
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", fileName));
		}		
	}

	/**
	 * Method to create a new config file based on an existing config file and 
	 * a hash table of configuration parameter values
	 * @param table - hash table of configuration parameters
	 * @return String - name of new config file
	 * @throws LogParserException 
	 */		
	private String getNewConfigFile(Hashtable table) throws LogParserException
	{
		Document doc = null;

		// Resolve the version number of the log in order to load the correct config file
		String version = (String) (table.get(Messages.getString("HyadesGALogVersionTagName")));
		String config_file = null;
		
		// If no version attribute given, use default
		if(version == null)
		{
			version = Messages.getString("HyadesGADefaultLogVersionTagName");
		}

		// Get the config file defined for this version
		config_file =  (String) (table.get(version));
		
		if(config_file == null)
		{
			// Cannot find the corresponding config file, use default
			config_file = (String) (table.get(Messages.getString("HyadesGADefaultLogVersionTagName")));
			
			// If the default (which should be) is not defined, throw an exception
			if(config_file == null) {
				throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_No_Config_File_ERROR_"));
			}
		}

		// Get the input stream for the config file

		InputStream inStream = null;
		String configFileDirectory = null;
		originalConfigFile = config_file;  // set original config file
		
		try {
			// Try to get the input stream using the plugin if we're running in eclipse

			Boolean append = Boolean.TRUE;

			Class pathClass = Class.forName("org.eclipse.core.runtime.Path");
			Constructor pathConstructor = pathClass.getConstructor(new Class[] {config_file.getClass()});
			Object path = pathConstructor.newInstance(new Object[] {config_file});
			
			Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");			
			Class pluginClass = Class.forName("org.eclipse.core.runtime.Plugin");
			
			Object plugin = platformClass.getMethod("getPlugin", new Class[] { currentPlugin.getClass()}).invoke(null, new Object[] { currentPlugin });
			
			inStream = (InputStream)(pluginClass.getMethod("openStream", new Class[] {Class.forName("org.eclipse.core.runtime.IPath"), boolean.class}).invoke(plugin, new Object[] {path, append}));
			
			/* Get the directory of the config file */
			/* bugzilla 70110 - Use Plaform.asLocalURL() to resolve the relative URL returned by Plugin.find() */
			URL url = (URL)(pluginClass.getMethod("find", new Class[] {Class.forName("org.eclipse.core.runtime.IPath")}).invoke(plugin, new Object[] {path})) ;
			File f = new File(((URL)(platformClass.getMethod("asLocalURL", new Class[] { url.getClass()}).invoke(null, new Object[] { url }))).getFile()) ;
			configFileDirectory = f.getParent();			
		} catch	(ClassNotFoundException e) {
			// We must be running outside of eclipse so try to find the config
			// file relative to the current directory
			
			inStream = getConfigInputStream(config_file, table);
			configFileDirectory = new File(config_file).getParentFile().getAbsolutePath();
		} catch	(InvocationTargetException e) {
			// We may be running outside of eclipse but the eclipse classes may be in
			// the path so try to find the config file relative to the current directory
			// To Do:  The exception should be logged, however, in case we are running inside
			// eclipse and there was an error from the invoked method calls.
			
			inStream = getConfigInputStream(config_file, table);
			// getConfigInputStream may modify originalConfigFile due to a user input parameter so use it to get the config file directory
			configFileDirectory = new File(originalConfigFile).getParentFile().getAbsolutePath();
		} catch (Exception e) {
			// Some other exception occurred - throw a LogParserException
			throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Open_ERROR_", config_file), e);
		}

		String logFilePath = ((String)(table.get(Messages.getString("HyadesGAInputLogFilenameTag"))));
	    File logFile = null;

	    if((logFilePath != null) && (logFilePath.trim().length() > 0)){
		    logFile = new File(logFilePath.trim());
	    }

		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(inStream);
		} catch (SAXException e) {
			return null;
		} catch (IOException e) {
			return null;
		} catch (ParserConfigurationException e) {
			return null;
		} catch (Exception e) {
			return null;
		}
		/**
		 * Get the sensor config - there should only be one sensor - and fill in
		 * the properties and attributes with the values from the input hash table
		 */
		NodeList sensorList = doc.getElementsByTagName(Messages.getString("HyadesGASensorTagName"));
		int sensorCount = sensorList.getLength();
		for (int i = 0; i < sensorCount; ++i)
		{
			Element sensorElement = (Element) sensorList.item(i);
			
			// We should always have a sensor instance but check in case an invalid
			// config file is used.
			if (sensorElement != null) {
				Element sensorNode;
				Element sensorPropertyElement;
				Element sensorTypeInstance = null;
				String propValue;
				String propName;
				String newConverterCmd = null;

				Element directoryPropertyElement = null;
				Element fileNamePropertyElement = null;
				
				// Get the sensor children (properties or sensor type instances)
				NodeList sensorNodes = sensorElement.getChildNodes();
				for (int k = 0; k < sensorNodes.getLength(); k++) {
					if (sensorNodes.item(k).getNodeType() == Node.ELEMENT_NODE) {
						sensorNode = (Element) sensorNodes.item(k);
						// Process the sensor property
						if (sensorNode.getTagName().equals(Messages.getString("HyadesGAPropertyElementTagName"))) {
							sensorPropertyElement = sensorNode;

							propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"));
							propValue = sensorPropertyElement.getAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"));
							
							if (propName.equals(Messages.getString("HyadesGAdirectoryAttributeName"))) {
								// save property element so it can be set after converter processing
								directoryPropertyElement = sensorPropertyElement;
								/* There might not be a log file if the converter command generates a file.
								 * If there is a log file then set the directory property
								 */
								if (logFile != null) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), logFile.getParent());
								}									
							}
							else if (propName.equals(Messages.getString("HyadesGAfileNameAttributeName"))) {
								// save property element so it can be set after converter processing
								fileNamePropertyElement = sensorPropertyElement;
								/* There might not be a log file if the converter command generates a file.
								 * If there is a log file then set the fileName property
								 */
								if (logFile != null) {
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), logFile.getName());
								}									
							}
							else if (propName.equals(Messages.getString("HyadesGAconverterCmdAttributeName"))) {
								if (propValue != null && propValue.length() > 0) {
									if (newConverterCmd == null) {
										newConverterCmd = modifyConverter(propValue, configFileDirectory, table, logFile);
									}
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), newConverterCmd);
								}
							}
							else if (propName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName"))) {
								parserName = propValue;							
							}
							else {
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), propValue.trim());
								}
							}
						}
						// Process the sensor type instance
						else if (sensorNode.getTagName().equals(Messages.getString("HyadesGAStaticParserSensorTagName")) ||
								 sensorNode.getTagName().equals(Messages.getString("HyadesGASingleFileSensorTagName"))) {
							sensorTypeInstance = sensorNode;
							// Get the sensor properties for this sensor type instance
							NodeList sensorPropertyList = sensorTypeInstance.getElementsByTagName(Messages.getString("HyadesGASensorPropertyElementTagName"));
							int sensorPropertyCount = sensorPropertyList.getLength();
							for(int j = 0; j < sensorPropertyCount; ++j)
							{
								sensorPropertyElement = (Element) sensorPropertyList.item(j);
								propName = sensorPropertyElement.getAttribute(Messages.getString("HyadesGASensorPropertyNameAttributeName"));
								propValue = (String)table.get(propName);
								if(propValue != null)
								{
									sensorPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), propValue.trim());
								}
							}

							// Get the sensor type attributes
							NamedNodeMap sensorTypeAttributeList = sensorTypeInstance.getAttributes();
							int sensorTypeAttributeCount = sensorTypeAttributeList.getLength();
				
							for(int j = 0; j < sensorTypeAttributeCount; ++j)
							{
								Node sensorTypeAttribute = sensorTypeAttributeList.item(j);
								String attrName = sensorTypeAttribute.getNodeName();
								// Modify the attribute based on user input
								if(attrName.equals(Messages.getString("HyadesGAdirectoryAttributeName")))
								{
									/* There might not be a log file if the converter command generates a file.
									 * If there is a log file then set the directory attribute
									 */
									if (logFile != null) {
										sensorTypeAttribute.setNodeValue(logFile.getParent());
									}									
								}
								else if(attrName.equals(Messages.getString("HyadesGAfileNameAttributeName")))
								{
									/* There might not be a log file if the converter command generates a file.
									 * If there is a log file then set the fileName attribute
									 */
									if (logFile != null) {
										sensorTypeAttribute.setNodeValue(logFile.getName());
									}
								}
								else if(attrName.equals(Messages.getString("HyadesGAstaticParserClassAttributeName")))
								{
									parserName = sensorTypeAttribute.getNodeValue();
								}
								else if(attrName.equals(Messages.getString("HyadesGAconverterCmdAttributeName")))
								{
									if (newConverterCmd == null) {
									   newConverterCmd = modifyConverter(sensorTypeAttribute.getNodeValue(), configFileDirectory, table, logFile);
									}
									sensorTypeAttribute.setNodeValue(newConverterCmd);
								}
								else
								{
									propValue = (String)table.get(attrName);
									if(propValue != null)
									{
										sensorTypeAttribute.setNodeValue(propValue.trim());
									}
								}
							}
						}
					}
				}

				// Get the sensor attributes
				NamedNodeMap sensorAttributeList = sensorElement.getAttributes();
				int sensorTypeAttributeCount = sensorAttributeList.getLength();
				
				for(int k = 0; k < sensorTypeAttributeCount; ++k)
				{
					Node sensorTypeAttribute = sensorAttributeList.item(k);
					String attrName = sensorTypeAttribute.getNodeName();
					
					// Modify the attribute based on user input
					propValue = (String)table.get(attrName);
					if(propValue != null && propValue.length() > 0)
					{
						sensorTypeAttribute.setNodeValue(propValue.trim());
					}
				}

				// Check if the directory of the log file to be parsed was updated by modifyConverter 				
				if (newDirectory != null) {
					// If so update it in the config file
					if (directoryPropertyElement != null) {
						directoryPropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), newDirectory);
					}
					if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAdirectoryAttributeName"));
						sensorAttribute.setNodeValue(newDirectory);					
					}
				}
				
				// Check if the name of the log file to be parsed was updated by modifyConverter 				
				if (newFileName != null) {
					// If so update it in the config file
					if (fileNamePropertyElement != null) {
						fileNamePropertyElement.setAttribute(Messages.getString("HyadesGASensorPropertyValueAttributeName"), newFileName);
					}
					if (sensorTypeInstance != null) {
						// Set the attribute
						Node sensorAttribute = sensorTypeInstance.getAttributeNode(Messages.getString("HyadesGAfileNameAttributeName"));
						sensorAttribute.setNodeValue(newFileName);					
					}	
				}
			}
		}

		/**
		 * Add the logger key to the outputter so the the outputter can write the the UI
		 * logger directly - there should only be one outputter.
		 */
		NodeList outputterList = doc.getElementsByTagName(Messages.getString("HyadesGAOutputterTagName"));
		int outputterCount = outputterList.getLength();
		for (int i = 0; i < outputterCount; ++i) {
			Element outputterElement = (Element) outputterList.item(i);
			Element outputterProperty = doc.createElement(Messages.getString("HyadesGAPropertyElementTagName"));
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyValueAttributeName"), key);
			outputterProperty.setAttribute(Messages.getString("HyadesGAPropertyNameAttributeName"),Messages.getString("HyadesGAStaticParserLoggerKeyName"));
			outputterElement.appendChild(outputterProperty);
		}


		// Save the resulting xml

		// Create a temporary file for the new adapter configuration
		File newFile = null;
		
		try {
			
			// Try to create a temporary file in the plugin's workspace directory (e.g. <workspace>/.metadata/plugins/org.eclipse.hyades.logging.adapter.config) for the new adapter configuration

			Class platformClass = Class.forName("org.eclipse.core.runtime.Platform");
			
			Method getPluginStateLocationMethod = platformClass.getMethod("getPluginStateLocation",new Class[]{Class.forName("org.eclipse.core.runtime.Plugin")});
			
			Object iPathObject =  getPluginStateLocationMethod.invoke(null,new Object[]{AdapterConfigPlugin.getPlugin()});
			
			Class iPathClass = iPathObject.getClass();
			
			Method toOSStringMethod = iPathClass.getMethod("toOSString",null);
			
			newFile = new File(((String)(toOSStringMethod.invoke(iPathObject,null))),"GLA" + System.currentTimeMillis() + ".adapter");
		}
		catch (Exception e) {
			
			try {		
				newFile = File.createTempFile("GLA", ".adapter");
			}
			catch (Exception e2){
				// If a temporary file cannot be generated in the OS tmp directory
				// try creating it in the same directory as the current config file.
				File oldFile = new File(config_file);
				String tmpFileName = new String("GLA" + System.currentTimeMillis() + ".adapter");
				newFile = new File(oldFile.getParent(), tmpFileName);
			}
		}
		
		// Ensure the temporary file gets deleted
		newFile.deleteOnExit();
		
		//Save the configuration in the temporary file:
		try {
			XmlUtility.serialize(doc, newFile);            
        } 
		catch (SerializationException s) {
		    throw new LogParserException(Messages.getString("HyadesGA_CBE_Adapter_Config_File_Save_ERROR_",newFile.getAbsolutePath()),s);
        }

		return newFile.getAbsolutePath();
	}

	/**
	 * Method to perform any modifications to the converter string based on user
	 * input from the UI
	 * @param converter  - converter string to be updated
	 * @param configFileDirectory  - the absolute path to the directory of the config file 
	 * @param table  - of user input values
	 * @param logFile - log file to be parsed
	 * @return String - new converter string
	 */
	protected String modifyConverter(String converter, String configFileDirectory, Hashtable table, File logfile) {
		return converter;
	}
}
