/**********************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.internal.execution.testgen.ui.wizards;

import java.io.IOException;
import java.util.ArrayList;
import java.util.StringTokenizer;
import org.eclipse.core.internal.resources.File;
import org.eclipse.core.internal.resources.Folder;
import org.eclipse.core.internal.resources.Project;
import org.eclipse.core.internal.resources.WorkspaceRoot;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.execution.testgen.TestgenPlugin;
import org.eclipse.hyades.internal.execution.testgen.ui.TestgenUIUtility;
import org.eclipse.hyades.internal.execution.testgen.ui.TestgenUIUtility.TestGenUIException;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.internal.ide.dialogs.ResourceSorter;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

/**
 * Abstract wizard page which can be extended by any wizard page which wants to provide controls for selecting file location
 * and specifying file name.  It provides the directory structure in a tree and allows filename to be entered. It keeps track of the 
 * last selected file for any extender and restores that selection the next time the extender is instantiated
 * @author Ernest Jessee
 */
public abstract class FileLocationSelectionWizardpage extends WizardPage implements ITestGenWizardPage
{
	private IStructuredSelection selection;

	protected TreeViewer tree;

	private File selectedFile;
	
	private IContainer selectedContainer;
	
	private Object previousSelection = null;
	
	private Text selectedFileName;

	public abstract String getFileExtension();
	
	public abstract String getFileNameLabel();
	
	public abstract String getTitle();
	
	public abstract String getDescription();

	protected Listener modifyListener = new Listener() {
		public void handleEvent(Event e) {
			if (e.widget == selectedFileName)
			{
				boolean bValid = isFileNameValid(selectedFileName.getText());
				boolean bContainer = (selectedContainer != null);
				setPageComplete(bValid && bContainer);
				if (!bValid)
				{
					setMessage(TestgenPlugin.getResourceString("FileLocationSelectionWizardpage.INVALID_FILENAME"), WARNING);			
				}
				else if (!bContainer)
					setMessage(TestgenPlugin.getResourceString("FileLocationSelectionWizardpage.INVALID_CONTAINER"), WARNING);
				else
					setMessage(null);

			}
		}
	};		
	
	class TreeSelectionChangedListener implements ISelectionChangedListener
	{
		

		public void selectionChanged(SelectionChangedEvent event)
		{
			IStructuredSelection selection = (IStructuredSelection)event.getSelection();
			Object primarySelection = selection.getFirstElement();
			if(selection.size()>1)
			{				
				tree.setSelection(new StructuredSelection(primarySelection));
			}
			
			if(File.class.isInstance(primarySelection))
			{
				selectedFile = (File)primarySelection;
				IContainer selectionParent = selectedFile.getParent();
				if(IFolder.class.isInstance(selectionParent) || IProject.class.isInstance(selectionParent))
				{
					selectedContainer = (IContainer)selectionParent;
					selectedFileName.setText(selectedFile.getName());
					//setPageComplete(true); //handled in modifyListener
					
				}
			}
			else if(IContainer.class.isInstance(primarySelection))
			{
				selectedContainer = (IContainer)primarySelection;
				selectedFile = null;
				
				setPageComplete(selectedFileName.getText().equals(""));
			}
			
				
		}

}
	
	class TreeFilter extends ViewerFilter
	{
		public boolean select(Viewer viewer,Object parentElement,Object element)
		{
			if(File.class.isInstance(element))
			{
				File file = (File)element;
				if (file.getFileExtension() == null)
					return false;
				if(file.getFileExtension().compareToIgnoreCase(getFileExtension())==0)
					return true;
				else	
					return false;
			}	
			else
			{	
				if (IProject.class.isInstance(element))   //62169
				{
					IProject proj = (IProject)element;
					return proj.isOpen();
				}
				return true;
				
			}
		}

	}
	
	
	
	public FileLocationSelectionWizardpage()
	{
		super(FileLocationSelectionWizardpage.class.getName());
	}

	
	
	
	
	public void saveSettings()
	{	
		String path = getValidFileName();
		
		TestgenPlugin.getDefault().getPreferenceStore().setValue(getClass().getName(),path);
			
	}

	private String getValidFileName()
	{
		String path =selectedContainer.getFullPath().toString()+"/"+selectedFileName.getText(); //$NON-NLS-1$
		
		//TODO more filename validation
		if(!path.endsWith("."+getFileExtension())) //$NON-NLS-1$
			path+="."+getFileExtension(); //$NON-NLS-1$
		return path;
	}
	

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite grandParent)
	{
		setTitle(getTitle());
		
		setDescription(getDescription()); 
		
		
		try
		{
					
			Font font = grandParent.getFont();
			Composite parent = new Composite(grandParent, SWT.NULL);
			
			TestgenUIUtility.assignGridLayout(parent,1);
			TestgenUIUtility.layoutControlInGrid(parent,1,GridData.HORIZONTAL_ALIGN_FILL);
			
			setControl(parent);
			
			WorkbenchHelp.setHelp(parent,getContextHelpID());
			
			tree = TestgenUIUtility.createTreeViewer(parent, font);
			
			tree.setContentProvider(new WorkbenchContentProvider());
			
			tree.setLabelProvider(new DecoratingLabelProvider(new WorkbenchLabelProvider(),PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator()));
			
			tree.setInput(ResourcesPlugin.getWorkspace().getRoot());
			tree.setSorter(new ResourceSorter(ResourceSorter.NAME));
			
			tree.addFilter(new TreeFilter());
			
			ArrayList selectedObjectList = new ArrayList();
			
			String fileName = getPreviouslySelectedObject(selectedObjectList);
			
			if(fileName!=null && selectedObjectList.size()>0 &&IContainer.class.isInstance(selectedObjectList.get(0)))
			{
				selectedContainer = (IContainer)selectedObjectList.get(0);
			}
			else if(fileName!=null && selectedObjectList.size()>0 && File.class.isInstance((selectedObjectList.get(0))))
			{
				selectedFile = (File)selectedObjectList.get(0);
				selectedContainer = (IContainer)selectedFile.getParent();
			}
				
			if(fileName!=null)					
				tree.setSelection(new StructuredSelection(selectedObjectList));
			
			tree.addSelectionChangedListener(new TreeSelectionChangedListener());
			
			TestgenUIUtility.layoutControlInGrid(tree.getControl(),1,GridData.FILL_BOTH|GridData.GRAB_VERTICAL);
			
			Canvas canvas = new Canvas(parent,SWT.NULL);
			TestgenUIUtility.layoutControlInGrid(canvas,1,GridData.HORIZONTAL_ALIGN_FILL);
			TestgenUIUtility.assignGridLayout(canvas,2);
			Label fileNameLabel = TestgenUIUtility.createLabel(canvas,getFileNameLabel(),null);
			selectedFileName = TestgenUIUtility.createEditControl(canvas,"",null); //$NON-NLS-1$
			
			if(fileName!=null)
				selectedFileName.setText(fileName);
			
			selectedFileName.addListener(SWT.Modify, modifyListener);
			
			TestgenUIUtility.layoutControlInGrid(fileNameLabel,1,GridData.HORIZONTAL_ALIGN_END);
			TestgenUIUtility.layoutControlInGrid(selectedFileName,1,GridData.HORIZONTAL_ALIGN_FILL|GridData.GRAB_HORIZONTAL);
			autoSelectProject();
			
			setPageComplete(true);
			
			
			
						
		}
		catch (TestGenUIException e)
		{
			e.printStackTrace();
		}
		
		
	}

	private String getPreviouslySelectedObject(ArrayList objectList)
	{
		String selection=null;
		String projectName=null;
		
		
		selection = TestgenPlugin.getDefault().getPreferenceStore().getString(getClass().getName());
		if(selection=="") //$NON-NLS-1$
			return null;
		
		WorkspaceRoot root = (WorkspaceRoot)ResourcesPlugin.getWorkspace().getRoot();
		
		selection = selection.substring(1,selection.length());
		
		StringTokenizer pathTokenizer = new StringTokenizer(selection,"/"); //$NON-NLS-1$
		
		projectName=pathTokenizer.nextToken();
		
		if(projectName==null)
			return null;
		
		Project project = (Project)root.getProject(projectName);
		if(!project.exists())
			return null;
		
		File file= null;
		Folder folder = null;
		Folder childFolder = null;
		String objectName = null;
		while(pathTokenizer.hasMoreTokens())
		{
			objectName = pathTokenizer.nextToken();
			
			if(folder == null)
			{
				folder = (Folder)project.getFolder(objectName);
				if(!folder.exists())
				{
					file = (File)project.getFile(objectName);
					if(file.exists())
					{
						objectList.add(file);
						return objectName;
					}
						
				}
				continue;
					
			}
			else
			{
				childFolder = (Folder)folder.getFolder(objectName);
				if(childFolder.exists())
				{
					folder = childFolder;
					continue;
				}
				else
				{
					file = (File)folder.getFile(objectName);
					break;
				} 	
				
			}
				
			
		}
		

		if(file.exists())
			objectList.add(file);
		else if(folder.exists())
			objectList.add(folder);
		else
			objectList.add(project);
			
		return objectName;
		
	}
		

	
	

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.IWizardPage#isPageComplete()
	 */
	public boolean isPageComplete()	
	{		
		if(isFileNameValid(selectedFileName.getText()) && selectedContainer!=null)
			return true;
		else		
			return false;
		 
		
	}

	public String getSelectedFilePath()
	{	
		return getValidFileName();
	}

	protected void autoSelectProject()
	{
		if (selectedContainer == null )
		{ 
			IProject[] projects = org.eclipse.core.resources.ResourcesPlugin.getWorkspace().getRoot().getProjects();
			if (projects.length > 0)
			{
			
				Object[] elementArray = tree.getVisibleExpandedElements();
				//List list = new List();
				//list.add(elementArray[0]);
				if (elementArray.length > 0)
					((StructuredViewer)tree).setSelection(new StructuredSelection(elementArray[0]));	
			}
	
		}
	}
	
	protected boolean isFileNameValid(String filename)
	{
		//file is later converted to an EMF URI object, so it must be a valid URI
		//thus no URI reserved characters are allowed
		if (filename.length() < 1)
			return false;
		final char[] RESERVED_CHARS = {';', '/' , '?', ':', '@', '&' , '=' , '+', '$' , ',',
				'<', '>', '#', '%', '"', ' ', //delimiters
				'{', '}', '|', '\\', '^', '[', ']', (char)39 /* ' */,  //'unwise'
				'!',    //just because?
				'*'}; //these are allowed in URLs but not in filenames
		
		for (int i = 0; i < RESERVED_CHARS.length; i++)
		{
			if (filename.indexOf(RESERVED_CHARS[i]) > -1)
				return false;
		}
		return true;
		//TODO: DBCS??
/*		java.io.File file = new java.io.File(filename);
		if (file.exists())
			return true;
		try {
			boolean b = file.createNewFile();
			b = b && file.exists();
			if (b)
				file.delete();
			return b;
		}
		catch (IOException ioe) {
			return false;
		}
*/		
	}
}
