/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.logging.adapter.formatters;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.logging.adapter.AdapterInvalidConfig;
import org.eclipse.hyades.logging.adapter.ICBEPropertyConstants;
import org.eclipse.hyades.logging.adapter.IDirectedGraph;
import org.eclipse.hyades.logging.adapter.IFormatter;
import org.eclipse.hyades.logging.adapter.impl.ProcessUnit;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.core.Guid;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ConfigureSituation;
import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.CreateSituation;
import org.eclipse.hyades.logging.events.cbe.DependencySituation;
import org.eclipse.hyades.logging.events.cbe.DestroySituation;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FeatureSituation;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.StopSituation;
import org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl;

/**
 * @author smith
 *
 * This Formatter class processes a HashMap to generate a CBE.
 * The HashMap key is the CBE property name string and the value
 * is the CBE property value. 
 * 
 * The HashMap key is a sequence of period delimited property name
 * string tokens representing the path to a specific CBE property.
 * We need a path to represent the CBE property because
 * CBE's have a class hierarchy.  For example, the CBE has a Reporter Component
 * ID and a Source Component ID property.  Both of these are ComponentIdentification
 * objects.  So, for example, to represent the location property of the Source
 * Component ID the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_SOURCECOMPONENTID.COMPONENTID_LOCATION
 * Some CBE properties are arrays so we can represent an element of an array
 * with an array index.  For example contextDataElements is an array of
 * ContextDataElements.  To represent the name property of the second ContextDataElement
 * in the array the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_CONTEXTDATAELEMENT[2].CONTEXTDATAELEMENT_NAME
 * A special case is the children attribute of ExtendedDataElement. It is a
 * collection of ExtendedDataElements which aren't ordered.  Therefore, the
 * name of the ExtendedDataElement will be used to represent the child ExtendedDataElement.
 * For example, to represent the type property of the child ExtendedDataDlement with name 
 * "ErrorCode" in the third ExtendedDataElement in the CBE extendedDataElements 
 * array the HashMap key would be:
 * COMMONBASEEVENT.COMMONBASEEVENT_EXTENDEDDATAELEMENT[3].EXTENDEDDATAELEMENT_CHILD.ErrorCode.EXTENDEDDATAELEMENT_TYPE
 *  
 */
public class CBEFormatter extends ProcessUnit implements IFormatter, ICBEPropertyConstants {

	/**
	 * Constant to indicate to use a built-in value
	 */
	private final static String BUILTIN = "##BUILTIN";
	/**
	 * Factory for creating events
	 */
	private static EventFactory eventFactory = new SimpleEventFactoryHomeImpl().getEventFactory("org.eclipse.hyades.logging.adapter.formatters.CBEFormatter");

	/**
	 * IP address of the machine parsing the log file. 
	 */
	private String localHostId = null;

	/**
	 * Format of the IP address of the machine parsing the log file.  This
	 * field will be set either to the String "IPV4" or the String "IPV6".
	 */
	private String localHostIdFormat = null;

	// Local host constants
	private final static String HOST_ID_FORMAT_IPV4 = "IPV4";
	private final static String HOST_ID_FORMAT_IPV6 = "IPV6";
	private final static String NONE = "None";
	private final static String UNKNOWN_HOST = "Unknown Host";


	/**
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#processEventItems(java.lang.Object[])
	 */
	public Object[] processEventItems(Object[] msgs) {
		/* If we have no messages return */
		if(msgs==null || msgs.length==0) {
			return null;
		}
		return processDirectedGraphs(msgs);
	}

	/**
	 * Process an array of messages
	 * @param msgs An array of HashMap objects representing the data in a group of messages
	 * @return CommonBaseEvent[] The CBE's generated from the HashMaps passed in
	 */
	private CommonBaseEvent[] processDirectedGraphs(Object[] msgs) {
		
		
		/* Create our CBE's that will be ouputs of this procesing.  There is one CBE per message */
		CommonBaseEvent[] cbes = new CommonBaseEvent[msgs.length];
		
		// For each message, process the list of rule matches
		for (int k = 0; k < msgs.length; k++) {
			
			/* Keep a cache of all elements that can have multiple entries. This is required
			 * as some types will have multiple rules that apply to them.  We need to find the element
			 * if previous rules have already been matched.  The scope of the cache is strictly the
			 * duration of the processing step for this CBE instance.
			 */ 
			HashMap cache=new HashMap(); 
			
			
			// Get the nodes of the graph
			Iterator ni= ((List)msgs[k]).iterator();
		
			CommonBaseEvent cbe = eventFactory.createCommonBaseEvent();
			
	        //Set the event's globalInstanceId property with a new GUID:
			cbe.setGlobalInstanceId(Guid.generate());

			// Iterate over the nodes of the graph
			while (ni.hasNext()) {
				IDirectedGraph  graphEntry = (IDirectedGraph)ni.next();
	
				// Get the path
				List path=graphEntry.getPath();
			
				try {		
						
					if (((String)path.get(0)).equals(COMMONBASEEVENT)) {
						
						if (((String)path.get(2)).equals(COMMONBASEEVENT_EXTENSIONNAME)) {
							cbe.setExtensionName((String)graphEntry.getValue());
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_LOCALINSTANCEID)) {
							String value = (String)graphEntry.getValue();
							
							if (value.equals(BUILTIN)) {
								value = getLocalHostId() + System.currentTimeMillis() + Integer.toString(msgs[k].hashCode());
							}
							cbe.setLocalInstanceId(value);
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_GLOBALINSTANCEID)) {
							String value = (String)graphEntry.getValue();
							
							if (value.equals(BUILTIN)) {
								value = (new Guid().toString());
							}
							cbe.setGlobalInstanceId(value);
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_CREATIONTIME)) {
							String strValue;
							Long lValue;
							try {
								strValue = (String)graphEntry.getValue();
								cbe.setCreationTime(strValue);
							}
							catch (ClassCastException e) {
								/* time returned by parser may be in milliseconds */
								lValue = (Long)graphEntry.getValue();
								cbe.setCreationTimeAsLong(lValue.longValue());
							}
							catch (Exception e) {
								throw e;
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SEVERITY)) {
							cbe.setSeverity(Short.parseShort((String)graphEntry.getValue()));
						}						
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_PRIORITY)) {
							cbe.setPriority(Short.parseShort((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_MSG)) {
							String message=(String)graphEntry.getValue();
							
							/* RKD:  if our message is greater the 1024 bytes in length we will chunk it up into extended 
							 * data elements to contain the extra info.  If the 1024 is ever removed from the schema we will
							 * be able to remove this portion from the processing
							 */
							int chunks=message.length()/1024;
							if(chunks==0) {
								cbe.setMsg(message);
							}
							else {
								cbe.setMsg(message.substring(0, 1024));
								for(int i=0; i<chunks;i++) {
									int offset=1024*i;
									String subChunk=message.substring(offset, offset+1024);
									String edeName="message_"+i;
									/* Get our IExtendedDataElement or create a new one */
									ExtendedDataElement ede=(ExtendedDataElement)cache.get(edeName);
									if(ede==null) {
										ede = eventFactory.createExtendedDataElement();	
										cbe.addExtendedDataElement(ede);
										cache.put(edeName, ede);
									}
									ede.setName(edeName);
									ede.setType("string");
									ede.setValues(new String[]{subChunk});
								}
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_REPEATCOUNT)) {
							cbe.setRepeatCount(Short.parseShort((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_ELAPSEDTIME)) {
							cbe.setElapsedTime(Long.parseLong((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SEQUENCENUMBER)) {
							cbe.setSequenceNumber(Long.parseLong((String)graphEntry.getValue()));
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_VERSION)) {
							cbe.setVersion((String)graphEntry.getValue());
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_OTHERDATA)) {
							cbe.addAny((String)graphEntry.getValue());                        
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_MSGDATAELEMENT)) {
						
							// Get the message data element from the CBE if it exists,
							// otherwise create it.
							MsgDataElement mde = cbe.getMsgDataElement();
							if (mde == null) {
								mde = eventFactory.createMsgDataElement();
								cbe.setMsgDataElement(mde);
							}
							// Set the message data element property
							
							if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGID)) {
								mde.setMsgId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGIDTYPE)) {
								mde.setMsgIdType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGID)) {
								mde.setMsgCatalogId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGTOKENS)) {
								
								String msgCatTok = (String)graphEntry.getValue();
								
								// If there is a message catalog token string then add it to the msgCatalogToken array
								if (msgCatTok != null && msgCatTok.length() > 0) { 
									mde.addMsgCatalogTokenAsString(msgCatTok);
								}

							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOGTYPE)) {
								mde.setMsgCatalogType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGCATALOG)) {
								mde.setMsgCatalog((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(MSGDATAELEMENT_MSGLOCALE)) {
								mde.setMsgLocale((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
							    
							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_MsgDataElement_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4)});							    
							    
							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_CONTEXTDATAELEMENTS)) {
							
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_ASSOCIATEDEVENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);


							/* Get our IExtendedDataElement or create a new one */
							ContextDataElement cde=(ContextDataElement)cache.get(key);
							if(cde==null) {
								cde = eventFactory.createContextDataElement();
								cbe.addContextDataElement(cde);	
								cache.put(key, cde);
							}

							if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_CONTEXTID)) {
								// contextId and contextValue properties are mutually exclusive so
								// check for existence of contextValues before setting contextId
								if (cde.getContextValue() != null) {		
								    
								    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

								    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Duplicate_ContextDataElement_WARN_");
								    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4),CONTEXTDATAELEMENT_CONTEXTVALUE});							    

								    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

								    log(event);
								}
								else {
									cde.setContextId((String)graphEntry.getValue());
								}
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_TYPE)) {
								cde.setType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_NAME)) {
								cde.setName((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(CONTEXTDATAELEMENT_CONTEXTVALUE)) {
								// contextId and contextValue properties are mutually exclusive so
								// check for existence of contextId before setting contextValue
								if (cde.getContextId() != null) {

								    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

								    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Duplicate_ContextDataElement_WARN_");
								    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4),CONTEXTDATAELEMENT_CONTEXTID});							    

								    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

								    log(event);
								}
								else {									
									cde.setContextValue((String)graphEntry.getValue());
								}
							}
							else {
								// error - incorrect name
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_ContextDataElement_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4)});							    

							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_REPORTERCOMPONENTID) || ((String)path.get(2)).equals(COMMONBASEEVENT_SOURCECOMPONENTID)) {
							// Get the component identification from the CBE if it exists,
							// otherwise create it
							ComponentIdentification cid;
							if (((String)path.get(2)).equals(COMMONBASEEVENT_REPORTERCOMPONENTID)) {
								cid = cbe.getReporterComponentId();
								if (cid == null) {
									cid = eventFactory.createComponentIdentification();
									cbe.setReporterComponentId(cid);
								}
							}
							else {
								cid = cbe.getSourceComponentId();
								if (cid == null) {
									cid = eventFactory.createComponentIdentification();
									cbe.setSourceComponentId(cid);
								}
							}
							
							// Set the component identification property
							
							if (((String)path.get(4)).equals(COMPONENTID_LOCATION)) {
								String value = (String)graphEntry.getValue();
								
								if (value.equals(BUILTIN)) {
									value = getLocalHostId();
								}
								cid.setLocation(value);
							}
							else if (((String)path.get(4)).equals(COMPONENTID_LOCATIONTYPE)) {
								String value = (String)graphEntry.getValue();
								
								if (value.equals(BUILTIN)) {
									value = getLocalHostIdFormat();
								}
								cid.setLocationType(value);
							}
							else if (((String)path.get(4)).equals(COMPONENTID_APPLICATION)) {
								cid.setApplication((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_EXECUTIONENVIRONMENT)) {
								cid.setExecutionEnvironment((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENT)) {
								cid.setComponent((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_SUBCOMPONENT)) {
								cid.setSubComponent((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENTIDTYPE)) {
								cid.setComponentIdType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_COMPONENTTYPE)) {
								cid.setComponentType((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_INSTANCEID)) {
								cid.setInstanceId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_PROCESSID)) {
								cid.setProcessId((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(COMPONENTID_THREADID)) {
								cid.setThreadId((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_ComponentIdentification_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4)});							    

							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}	
						}
						else if (((String)path.get(2)).equals(COMMONBASEEVENT_SITUATION)) {
	
							// Get the situation from the CBE if it exists,
							// otherwise create it.
							Situation sit = cbe.getSituation();
							if (sit == null) {
								sit = eventFactory.createSituation();
								cbe.setSituation(sit);
							}
							
							// Set the appropiate situation property
							/* bugzilla 77354
							 * situationType is an abstract class in the CBE implementation.
							 * The GLA editor pre-Hyades 3.2 created an element in the adapter file for
							 * SITUATION_SITUATIONTYPE as a parent of the actual situation type but 
							 * starting in Hyades 3.2 this element is no longer generated.
							 * Therefore the following code must handle adapter files created
							 * with and without SITUATION_SITUATIONTYPE.
							 * 
							 * Situation categoryName must match the actual situation type
							 * stored in the CBE so it will be set automatically when a situation 
							 * type is created.  Therefore a rule for categoryName is not necessary. 
							 */
							if (((String)path.get(4)).equals(SITUATION_CATEGORYNAME)) {
								sit.setCategoryName((String)graphEntry.getValue());
							}
							else if (((String)path.get(4)).equals(AVAILABLESITUATION) || 
									 (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(AVAILABLESITUATION))) {					
								// Set the situation type properites	
								AvailableSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createAvailableSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(AVAILABLESITUATION);
								}
								else if (sittype instanceof AvailableSituation){
									asit = (AvailableSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(AVAILABLESITUATION_OPERATIONDISPOSITION)) {
										asit.setOperationDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(AVAILABLESITUATION_PROCESSINGDISPOSITION)) {
										asit.setProcessingDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(AVAILABLESITUATION_AVAILABILITYDISPOSITION)) {
										asit.setAvailabilityDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
									    
									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_AvailableSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}														
								}
							}	
							else if (((String)path.get(4)).equals(REQUESTSITUATION) ||
									 (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(REQUESTSITUATION))) {
								RequestSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createRequestSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(REQUESTSITUATION);
								}
								else if (sittype instanceof RequestSituation){
									asit = (RequestSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
										asit.setSituationQualifier((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_RequestSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}														
								}												
							}
							else if (((String)path.get(4)).equals(STARTSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(STARTSITUATION))) {									
								StartSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createStartSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(STARTSITUATION);
								}
								else if (sittype instanceof StartSituation){
									asit = (StartSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
										asit.setSituationQualifier((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_StartSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}														
								}											
							}											
							else if (((String)path.get(4)).equals(STOPSITUATION) ||
									 (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(STOPSITUATION))) {
								StopSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createStopSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(STOPSITUATION);
								}
								else if (sittype instanceof StopSituation){
									asit = (StopSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SITUATIONQUALIFIER)) {
										asit.setSituationQualifier((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_StopSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    
									    
									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}														
								}													
							}
							else if (((String)path.get(4)).equals(CONNECTSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(CONNECTSITUATION))) {
								ConnectSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createConnectSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(CONNECTSITUATION);
								}
								else if (sittype instanceof ConnectSituation){
									asit = (ConnectSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(CONNECTSITUATION_SITUATIONDISPOSITION)) {
										asit.setSituationDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_ConnectSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);
	
									    log(event);
									}														
								}												
							}													
							else if (((String)path.get(4)).equals(CONFIGURESITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(CONFIGURESITUATION))) {
								ConfigureSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createConfigureSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(CONFIGURESITUATION);
								}
								else if (sittype instanceof ConfigureSituation){
									asit = (ConfigureSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_ConfigureSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(CREATESITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(CREATESITUATION))) {
								CreateSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createCreateSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(CREATESITUATION);
								}
								else if (sittype instanceof CreateSituation){
									asit = (CreateSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_CreateSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(DESTROYSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(DESTROYSITUATION))) {
						
								DestroySituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createDestroySituation();
									sit.setSituationType(asit);
									sit.setCategoryName(DESTROYSITUATION);
								}
								else if (sittype instanceof DestroySituation){
									asit = (DestroySituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(SITUATIONTYPE_SUCCESSDISPOSITION)) {
										asit.setSuccessDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_CreateSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(REPORTSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(REPORTSITUATION))) {
								ReportSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createReportSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(REPORTSITUATION);
								}
								else if (sittype instanceof ReportSituation){
									asit = (ReportSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(REPORTSITUATION_REPORTCATEGORY)) {
										asit.setReportCategory((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_ReportSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(FEATURESITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(FEATURESITUATION))) {
								FeatureSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createFeatureSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(FEATURESITUATION);
								}
								else if (sittype instanceof FeatureSituation){
									asit = (FeatureSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(FEATURESITUATION_FEATUREDISPOSITION)) {
										asit.setFeatureDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_FeatureSituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(DEPENDENCYSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(DEPENDENCYSITUATION))) {
								DependencySituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createDependencySituation();
									sit.setSituationType(asit);
									sit.setCategoryName(DEPENDENCYSITUATION);
								}
								else if (sittype instanceof DependencySituation){
									asit = (DependencySituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(DEPENDENCYSITUATION_DEPENDENCYDISPOSITION)) {
										asit.setDependencyDisposition((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_DependencySituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else if (((String)path.get(4)).equals(OTHERSITUATION) ||
									(((String)path.get(4)).equals(SITUATION_SITUATIONTYPE) && ((String)path.get(6)).equals(OTHERSITUATION))) {
								OtherSituation asit = null;
								SituationType sittype = sit.getSituationType();
								if (sittype == null) {
									asit = eventFactory.createOtherSituation();
									sit.setSituationType(asit);
									sit.setCategoryName(OTHERSITUATION);
								}
								else if (sittype instanceof OtherSituation){
									asit = (OtherSituation)sittype;
								}
								else {
									// error - incorrect situtation type
									// DNS: bugzilla 55337 - ignore this error for now  log(Messages.getString("HyadesGA_CBE_Formatter_SituationType_Warning",getUniqueID(),(String)path.get(6)),AdapterSensor.INFO_LEVEL);										
								}
								
								if (asit != null) {
									int pindex = 6;
									if (((String)path.get(4)).equals(SITUATION_SITUATIONTYPE)) {
										pindex = 8;
									}
									if (((String)path.get(pindex)).equals(SITUATIONTYPE_REASONINGSCOPE)) {
										asit.setReasoningScope((String)graphEntry.getValue());
									}
									else if (((String)path.get(pindex)).equals(OTHERSITUATION_ANYDATA)) {
										asit.setAny((String)graphEntry.getValue());
									}
									else {
										// error - incorrect situtation type property
									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_DependencySituation_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(pindex)});							    

									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
								}												
							}
							else {
								// error - incorrect name
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_SituationType_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(6)});							    

							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}
							
						}
						else if (((String)path.get(2)).startsWith(COMMONBASEEVENT_ASSOCIATEDEVENTS)) {
							
							
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_ASSOCIATEDEVENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);


							/* Get our IExtendedDataElement or create a new one */
							AssociatedEvent ae=(AssociatedEvent)cache.get(key);
							if(ae==null) {
								ae = eventFactory.createAssociatedEvent();
								cbe.addAssociatedEvent(ae);	
								cache.put(key, ae);
							}
								
	
							if (((String)path.get(4)).equals(ASSOCIATEDEVENT_ASSOCIATIONENGINEREFERENCE)) {
								// As of CBE 1.0.1 this property is a reference to an association engine so 
								// we shouldn't be getting this property from the parser
								// Log an warning message.
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_AssociatedEvent_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4)});							    

							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}
							else if (((String)path.get(4)).equals(ASSOCIATEDEVENT_ASSOCIATIONENGINEINFO)) {
		
								// Get the AssociationEngineInfo from the AssociatedEvent if it exists,
								// otherwise create it.
								AssociationEngine aei = ae.getAssociationEngineInfo();
								if (aei == null) {
									aei = eventFactory.createAssociationEngine();

									//Set the engine's id property with a new GUID:
									aei.setId(Guid.generate());

									ae.setAssociationEngineInfo(aei);
								}
								
								// Set the AssociationEngine property
								if (((String)path.get(6)).equals(ASSOCIATIONENGINE_ID)) {
									aei.setId((String)graphEntry.getValue());
								}
								else if (((String)path.get(6)).equals(ASSOCIATIONENGINE_NAME)) {
									aei.setName((String)graphEntry.getValue());
								}
								else if (((String)path.get(6)).equals(ASSOCIATIONENGINE_TYPE)) {
									aei.setType((String)graphEntry.getValue());
								}
								else {
									// error - incorrect name
								    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

								    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_AssociationEngine_WARN_");
								    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(6)});							    

								    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

								    log(event);
								}
							}
							else if (((String)path.get(4)).equals(ASSOCIATEDEVENT_RESOLVEDEVENT)) {
								ae.addResolvedEvent((String)graphEntry.getValue());
							}
							else {
								// error - incorrect name
							    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

							    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_AssociatedEvent_WARN_");
							    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),(String)path.get(4)});							    

							    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

							    log(event);
							}
						}
						else if (((String)path.get(2)).startsWith(COMMONBASEEVENT_EXTENDEDDATAELEMENTS)) {
	
							/* Extract our key that identifies this rule.  We will use COMMONBASEEVENT_EXTENDEDDATAELEMENTS + index as the key */
							String key=((String)path.get(2))+(String)path.get(3);
							
							
							/* Get our IExtendedDataElement or create a new one */
							ExtendedDataElement ede=(ExtendedDataElement)cache.get(key);
							if(ede==null) {
								ede = eventFactory.createExtendedDataElement();	
								cbe.addExtendedDataElement(ede);
								cache.put(key, ede);
							}
							
									
									
							// If we have a valid extended data element then set its property with the data
							String property=((String)path.get(4));
							int depth=4;
							while (property != null) {											
						
								if (property.equals(EXTENDEDDATAELEMENT_NAME)) {
									ede.setName((String)graphEntry.getValue());
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_TYPE)) {
									ede.setType((String)graphEntry.getValue());
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_VALUES)) {
									/* bugzilla 70133 - support for multiple values in an ExtendedDataElement */
									List valueList = ede.getValues();
									
									if (valueList == null) {
										/* set the value list */
										String[] result={(String)graphEntry.getValue()};			
										
										ede.setValues(result);
									}
									else {
										/* add value to EDE value list */
										valueList.add((String)graphEntry.getValue());
									}
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_HEXVALUE)) {
									// values and hexvalues properties are mutually exclusive so
									// add check for existence of values before setting hexvalues
									if (ede.getValues() != null) {										

									    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
									    
									    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Duplicate_ExtendedDataElement_WARN_");
									    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),property,EXTENDEDDATAELEMENT_VALUES});							    
									    
									    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

									    log(event);
									}
									else {
										ede.setHexValueAsCharArray((char[])graphEntry.getValue());
									}
									property = null;
								}
								else if (property.equals(EXTENDEDDATAELEMENT_CHILDREN)) {
									// Get the name of the child Extended Data Element
									String edeName = ((String)path.get(++depth));
									
									// Get the child ExtendedDataElement from the ExtendedDataElement if it exists,
									// otherwise create it.
									List edeArr = ede.getChildren();
									if (edeArr == null) {
										ExtendedDataElement edeTmp = eventFactory.createExtendedDataElement();
										edeTmp.setName(edeName);
										ede.addChild(edeTmp);
										ede = edeTmp;
									}
									else {
										ExtendedDataElement edeTmp = null;
										// Search for the ExendedDataElement in the arrray
										for (int j=0; j < edeArr.size(); j++) {
											if (((ExtendedDataElement)(edeArr.get(j))).getName().equals(edeName)) {
												edeTmp = ((ExtendedDataElement)(edeArr.get(j)));
											}
										}
										
										// If it was not found in the array, create it
										if (edeTmp == null) {
											edeTmp = eventFactory.createExtendedDataElement();
											edeTmp.setName(edeName);
											ede.addChild(edeTmp);
										}
										
										ede = edeTmp;
									}
									
									// Get the next ID for the next Extended Data Element property
									depth += 2;
									property = ((String)path.get(depth));

								}
								else {
									// error - incorrect name
								    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

								    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_ExtendedDataElement_WARN_");
								    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),property});							    

								    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

								    log(event);
									property = null;
								}
							} /* end while */
						}
						else {
							// error - incorrect name
						    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

						    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_CommmonBaseEvent_WARN_");
						    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),((String)path.get(2))});							    

						    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

						    log(event);
						}
					}
					else {
						// error - invalid CBE property name string
					    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

					    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Invalid_Property_WARN_");
					    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),((String)path.get(0))});							    

					    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

					    log(event);
					}
				}
				catch(NumberFormatException e) {
				    
				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();

				    event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Non_Numeric_Property_WARN_");
				    event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),((String)path.get(2))});							    

				    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

				    log(event);
				}
				catch(Exception e) {					
				    
				    CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
				    
				    event.setMsg(Messages.getString("HyadesGA_CBE_Formatter_Exception_WARN_",getUniqueID(),((String)path.get(2)),e.getMessage()));
				    //event.getMsgDataElement().setMsgCatalogId("HyadesGA_CBE_Formatter_Exception_WARN_");
				    //event.getMsgDataElement().setMsgCatalogTokensAsStrings(new String[]{getUniqueID(),((String)path.get(2)),e.getMessage()});							    

				    event.setSeverity(CommonBaseEvent.SEVERITY_WARNING);

				    log(event);
				}
			} /* end while */
			cbes[k] = cbe;
		} /* end for */
		
		return cbes;
	}

	/**)
	 * @see org.eclipse.hyades.logging.adapter.IProcessUnit#testProcessEventItems(java.lang.Object[])
	 */
	public Object[] testProcessEventItems(Object[] msgs)
		throws AdapterInvalidConfig {

		return testProcessDirectedGraphs(msgs);
	}

	/** 
	 * Test processing an array of HashMap objects
	 */
	private CommonBaseEvent[] testProcessDirectedGraphs(Object[] elements) throws AdapterInvalidConfig  {
		if(!(elements[0] instanceof List)) {
			throw new AdapterInvalidConfig("This formatter will only accept arrays of Lists");
		}
		CommonBaseEvent[] eventArray = new CommonBaseEvent[1];
		eventArray[0] = eventFactory.createCommonBaseEvent();
		return eventArray;
	}

	/**
	 * @see org.eclipse.hyades.logging.adapter.IComponent#update()
	 */
	public void update() throws AdapterInvalidConfig {
		// first get the basic configuration set
		super.update();

	}

	/**
	 * Function to determine IP address and fully-qualified host name
	 * of the machine parsing the log file. 
	 */
	private void getLocalHostInfo() {
		try {
			localHostId = java.net.InetAddress.getLocalHost().getHostAddress();
			if (localHostId.indexOf(':') != -1)
				localHostIdFormat = HOST_ID_FORMAT_IPV6;
			else
				localHostIdFormat = HOST_ID_FORMAT_IPV4;
		}
		catch (Exception e) {
			localHostId = UNKNOWN_HOST;
			localHostIdFormat = NONE;
		}

	}

	/**
	 * @return
	 */
	private String getLocalHostId() {
		if (localHostId == null)
			getLocalHostInfo();
		return localHostId;
	}

	/**
	 * @return
	 */
	private String getLocalHostIdFormat() {
		if (localHostIdFormat == null)
			getLocalHostInfo();
		return localHostIdFormat;
	}

}
