/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;

import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IComponentBroker;
import org.eclipse.hyades.logging.adapter.IProcessUnit;
import org.eclipse.hyades.logging.adapter.internal.util.EventQueue;
import org.eclipse.hyades.logging.adapter.IContextListener;

/**
 * Please see description of the IEventBroker interface.
 */
public class ComponentBroker implements IComponentBroker {
	private Object lock;
	private EventConsumer[] consumers = null; 
	private IComponentBroker nextBroker = null; 
	private IContextListener listener;


	public ComponentBroker(String name, IContextListener listener) {
		this.listener = listener;
		lock = new Object();
	}

	public void addEventConsumer(IComponent con) {
		if(consumers == null) {
			consumers = new EventConsumer[1];
			consumers[0] = new EventConsumer(con);
		}
		else {
			synchronized(lock) {
				EventConsumer[] replacement = new EventConsumer[consumers.length + 1];
				for(int i = 0; i < consumers.length; i++) {
					replacement[i] = consumers[i];
				}
				replacement[consumers.length] = new EventConsumer(con);

				consumers = replacement;
			}
		}
	}

	public void setNextBroker(IComponentBroker nextBroker) {
		this.nextBroker = nextBroker;
	}

	/**
	 * Push some events into the queues for processing
	 */
	public void queueEvents(Object[] obj) {
		synchronized(lock) {
			for(int i=0; i<consumers.length; i++) {
				boolean added=false;
				do {
					try {
						consumers[i].queue.addItem(obj);
						added = true;
					} catch (InterruptedException e) {
						/* We could not add the element to the queue as it is full */
					}
				} while(!added);	
			}
		}
	}

	public void start() {
		synchronized(lock) {
			for(int i = 0; i < consumers.length; i++) {
				consumers[i].start();
			}
		}
	}

	public void flushAndTerminate() throws InterruptedException {
		// Flush the consumers so that any unwritten data are pushed out to the next broker
		for(int i = 0; i < consumers.length; i++) {
			consumers[i].flushAndTerminate();
		}

		// Make sure the next broker shuts down before returning
		if(nextBroker != null) {
			nextBroker.flushAndTerminate();
		}
	}

	private class EventConsumer {
		private IComponent entity;
		private EventConsumerThread thread;
		private EventQueue queue;

		private EventConsumer(IComponent e) {
			queue = new EventQueue();
			entity = e;
		}

		public void flushAndTerminate() throws InterruptedException {
			queue.flush();
			thread.terminate();
		}

		public void start() {
			thread = new EventConsumerThread(this);
			thread.setName(entity.getName());
			thread.start();
		}
	}

	private class EventConsumerThread extends Thread {

		private EventConsumer consumer;
		private boolean stopping = false; // whether a stop operation is pending
		private boolean stopped = false; // whether the processing unit has stopped
		private Object stopLock;

		public EventConsumerThread(EventConsumer consumer) {
			this.consumer = consumer;
			stopLock = new Object();
		}

		public void terminate() {
			stopping = true;
			synchronized(stopLock) {
				if(!stopped) {
					try {
						stopLock.wait();
					} catch (InterruptedException e) {
						// TODO: handle this
					}
				}
			}
		}

		public void run() {
outer:
			while(!stopping) { //!isHardStop() && !stopping) {
				Object[] values = null;
				try {
					values = consumer.queue.getItem();
				}
				catch(InterruptedException e) {
					break outer;
				}

				/* If we have a IContextListener tell it we are invoking the component as
				 * well as the data we are giving to teh component.
				 */
				if(listener != null) {
					try {
						listener.preProcessEventItems(consumer.entity, values);
					}
					catch(Throwable e) {
						/* Don't let the listener break us */
					}	
				}

				/* Invoke the component */
				Object[] oa = null;
				if(values != null) {
					oa = ((IProcessUnit) consumer.entity).processEventItems(values);
				}

				if(oa != null) {
					/* If we have a IContextListener tell it we have finished invoking the
					 * component as well as the data we recieved back.
					 */
					if(listener != null) {
						try {
							listener.postProcessEventItems(consumer.entity, oa);
						}
						catch(Throwable e) {
							/* Don't let the listener break us */
						}	
					}

					/* If this is not the last broker in the list then we need to pass the results on to the next broker */
					if(nextBroker != null) {
						nextBroker.queueEvents(oa);
					}
				}
			}
			stopped = true;
			synchronized(stopLock) {
				stopLock.notifyAll();
			}
		}
	}
}
