/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.logging.adapter.impl;

import org.eclipse.hyades.logging.adapter.IComponent;
import org.eclipse.hyades.logging.adapter.IComponentBroker;
import org.eclipse.hyades.logging.adapter.IExtractor;
import org.eclipse.hyades.logging.adapter.IFormatter;
import org.eclipse.hyades.logging.adapter.IOutputter;
import org.eclipse.hyades.logging.adapter.IParser;
import org.eclipse.hyades.logging.adapter.ISensor;
import org.eclipse.hyades.logging.adapter.util.Messages;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.Situation;

public class MultithreadedContext extends BasicContext {
	private IComponentBroker[] brokers = new IComponentBroker[4]; // not needed for Sensors

	public void run() {

		/* Log the fact we have started the context */
		CommonBaseEvent startEvent = getEventFactory().createCommonBaseEvent();

		startEvent.setMsg(Messages.getString("HyadesGAContext_Started_INFO_",
				getName()));
		startEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
		Situation startSituation = getEventFactory().createSituation();
		startSituation.setStartSituation("INTERNAL", "START COMPLETED",
				"SUCCESSFUL");
		startEvent.setSituation(startSituation);
		log(startEvent);

		IComponent contextComponents[] = getComponents();

		/* Create a broker between each component.  All of the outputters should be in the same broker.  This
		 * loop makes the assumption that the outputters are all at the end of the context
		 */
		for (int i = 0; i < contextComponents.length; i++) {
			int brokerIndex = -1;

			if (contextComponents[i] instanceof IExtractor) {
				brokerIndex = 0;
			}
			else if (contextComponents[i] instanceof IParser) {
				brokerIndex = 1;
			}
			else if (contextComponents[i] instanceof IFormatter) {
				brokerIndex = 2;
			}
			else if (contextComponents[i] instanceof IOutputter) {
				brokerIndex = 3;
			}

			if(brokerIndex != -1) {
				if(brokers[brokerIndex] == null) {
					brokers[brokerIndex] = new ComponentBroker("Broker " + brokerIndex, listener);
				}
				brokers[brokerIndex].addEventConsumer(contextComponents[i]);
			}
		}

		// Wire up the brokers
		for(int i = brokers.length - 1; i > 0; i--) {
			if(brokers[i] != null) {
				for(int j = i - 1; j >= 0; j--) {
					if(brokers[j] != null) {
						brokers[j].setNextBroker(brokers[i]);
					}
				}
			}
		}

		/* Get the context components */
		for (int i = 0; i < contextComponents.length; i++) {
			/* Log the fact we have started the component */
			startEvent = getEventFactory().createCommonBaseEvent();

			startEvent.setMsg(Messages.getString(
					"HyadesGAComponent_Started_INFO_", contextComponents[i]
							.getName()));
			startEvent.setSeverity(CommonBaseEvent.SEVERITY_INFORMATION);
			startSituation = getEventFactory().createSituation();
			startSituation.setStartSituation("INTERNAL", "START COMPLETED",
					"SUCCESSFUL");
			startEvent.setSituation(startSituation);
			log(startEvent);
		}

		/* Start up the brokers */
		for(int i = 0; i < brokers.length; i++) {
			if(brokers[i] != null) {
				brokers[i].start();
			}
		}

		try {
			/* Get our extractor.  It is the first component in the component chain */
			ISensor sensor = (ISensor) getComponents()[0];

			try {
				while (!isHardStop() && !stopping) {

					/* If we have a IContextListener tell it we are invoking the sensor for more
					 * information
					 */
					if (listener != null) {
						try {
							listener.preProcessEventItems(sensor, null);
						} catch (Throwable e) {
							/* Don't let the listener break us */
						}
					}

					/* Get our data from the sensor */
					Object[] msgs = sensor.getNext();

					/* If we have a IContextListener tell it we have just returned from the 
					 * sensor and give it the lines we have retrieved.
					 */
					if (listener != null) {
						try {
							listener.postProcessEventItems(sensor, msgs);
						} catch (Throwable e) {
							/* Don't let the listener break us */
						}
					}

					/* If we have recieved no data from the sensor then
					 * we need to start our shutdown sequence.  This can
					 * be reset if we get data on the next call.
					 */
					if (msgs == null) {
						handleControledStop();
					} else {
						resetControledStop();
						brokers[0].queueEvents(msgs);
					}
				}
			} catch (Exception e) {
				CommonBaseEvent event = getEventFactory()
						.createCommonBaseEvent();
				event.setMsg(e.toString());
				event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

				log(event);

			}
		} catch (Throwable e) {
			CommonBaseEvent event = getEventFactory().createCommonBaseEvent();
			event.setMsg(e.toString());
			event.setSeverity(CommonBaseEvent.SEVERITY_FATAL);

			log(event);
		}
	}

	public void stop() {
		// Shut down the brokers
		try {
			brokers[0].flushAndTerminate();
		} catch (InterruptedException e) {
			//TODO: handle this
		}

		super.stop();

	}

}