/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.FileNotFoundException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.ExecutionComponentStateChangeEvent;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.core.IEclipseExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutableObject;
import org.eclipse.hyades.execution.core.IExecutionComponent;
import org.eclipse.hyades.execution.core.IExecutionComponentFactory;
import org.eclipse.hyades.execution.core.IExecutionComponentStateChangeListener;
import org.eclipse.hyades.execution.core.IExecutionEnvironment;
import org.eclipse.hyades.execution.core.IExecutor;
import org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport;
import org.eclipse.hyades.execution.core.INode;
import org.eclipse.hyades.execution.core.IRemoteHyadesComponent;
import org.eclipse.hyades.execution.core.ISession;
import org.eclipse.hyades.execution.core.UnknownDaemonException;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.execution.harness.util.ExecutionAdapterUtilities;
import org.eclipse.hyades.execution.harness.util.ISystemUtility;
import org.eclipse.hyades.execution.harness.util.StandaloneExecutionUtilities;
import org.eclipse.hyades.execution.harness.util.StandaloneSystemUtility;
import org.eclipse.hyades.execution.local.EclipseExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.ExecutableObjectStub;
import org.eclipse.hyades.execution.local.ExecutionComponentFactoryImpl;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.execution.local.HyadesTestNodeImpl;
import org.eclipse.hyades.execution.local.JavaProcessExecutableObjectStub;
import org.eclipse.hyades.execution.local.SessionContext;
import org.eclipse.hyades.execution.local.SessionContextToSessionMap;
import org.eclipse.hyades.internal.execution.local.control.AgentControllerUnavailableException;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGMachineConstraint;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofileFactory;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionType;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictReason;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofilePackageImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.hierarchy.util.StringUtil;

/**
 * <p>This class provides the mechanism for launching Hyades tests.  The 
 * flow in the launchTest method is depicted in the execution use case
 * realization: <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a></p>
 * 
 * <p>This class may used to launch tests from within the workbench (from code
 * running inside the eclipse shell), or it may be used to launch tests from outside
 * the workbench (referred to as "standalone".)  When running standalone, you must
 * use the constructor that accepts a string argument, and pass in a configuration directory
 * as that argument.  The configuration directory should be structured like, and may be,
 * an eclipse directory.  The directory is used to determine information such as
 * the execution components and their mappings to test types, the location of resource
 * bundles, etc.  Also remember that when running standalone, you are responsible for 
 * properly setting the classpath of your JVM so that all necessary classes can be loaded.</p>
 * 
 * An example of how to launch a test standalone is shown below.  Be sure to substitute the
 * correct values for standaloneConfigurationFiles and test suite path.</p><br>
 * 
 * <code><pre>
 *	public static void main( String args[] )
 *	{
 *		TestExecutionHarness harness = new TestExecutionHarness("C:\\Program Files\\eclipse 2.1\\eclipse");
 *		
 *		harness.launchTest(
 *			"C:\\Program Files\\eclipse 2.1\\eclipse\\runtime-workspace\\test\\src\\first.testsuite",
 *			null,
 *			"localhost",
 *			"10002",
 *			"c:\\temp",
 *			"executionresult.execution",
 *			true,
 *			true
 *		);				
 *	}
 * </code></pre>
 * 
 */
public class TestExecutionHarness {
	
	private final Object agentLock = new Object();
	private boolean bAgentReady = false;
	
	private final HashMap testTypeMap = new HashMap();
	
	// TODO: JPT - work with RD to move this registration mechanism into the
	// execution component factory
	private boolean bFactoryInitialized = false;

	private IExecutionComponentFactory factory;
	private ISession session;
	
	private IExecutionEnvironment exeEnvironment;
	private IExecutor executor;
	private String[] standaloneConfigurationFiles;
	private ISystemUtility systemUtility;
	private ExecutionHarnessDataProcessorFactory dpFactory;
	
	public static void main( String args[] )	
	{
		if ( args.length != 8 )
		{
			System.err.println("Usage: TestExecutionHarness <eclipseHomePath> " + //$NON-NLS-1$
					"<versionString> <testSuitePath> <deploymentPath> <hostname> " + //$NON-NLS-1$
					"<resultLocation> <resultName> <overwriteExistingResult>"); //$NON-NLS-1$
			System.exit(-1);
		}
		
		String eclipseHome=args[0];
		String versionString = args[1];
		String testSuitePath=args[2];
		String deploymentPath = args[3];
		String hostName=args[4];
		String resultsLocation=args[5];
		String resultName = args[6];
		boolean overwriteExistingResult = Boolean.valueOf(args[7]).booleanValue();
		
		StringBuffer bufError = new StringBuffer();
		
		
		TestExecutionHarness harness = new TestExecutionHarness(eclipseHome, versionString);
		
		IExecutor executor = harness.launchTest(
			testSuitePath,
			deploymentPath,
			null,
			hostName,
			"10002", //$NON-NLS-1$
			resultsLocation,
			resultName,
			overwriteExistingResult,
			true,
			true,
			bufError);
		
		if ( executor != null )
		{
			// Since we are not running as part of the eclipse shell, we need to
			// keep this thread alive until the model loaders have completed (or else
			// the JVM may exit before the model load is complete and the model saved.
			waitForDataProcessorExit(executor);
		}
		else {
		    System.out.println(bufError.toString());
		    throw new RuntimeException(bufError.toString());
		}
	}

	private static IRemoteHyadesComponent findRemoteHyadesComponent(IExecutor executor)
	{
		IRemoteHyadesComponent comp = null;
		IExecutionComponent[] children= executor.getChildren();
		for (int i=0; i<children.length;i++) {
			if(children[i] instanceof IRemoteHyadesComponent) {
				comp = (IRemoteHyadesComponent) children[i];
			}
		}
		return comp;
	}


	public TestExecutionHarness() {
		super();
		
		try {
			// try to load an eclipse class
			Class.forName("org.eclipse.core.boot.BootLoader"); //$NON-NLS-1$
			
			// if we didn't just throw an exception, then we know that we're
			// running in Eclipse's JVM.
			this.systemUtility = ExecutionHarnessPlugin.getDefault();
		}
		catch ( ClassNotFoundException exc)
		{
			// We couldn't load a class that must be present if we're running from
			// eclipse, so we must be running standalone.  But the caller didn't 
			// specify a config file dir in the constructor, and without that 
			// argument, we can't initialize for standalone execution.
			// We also can't localize the error message, since, without eclipse
			// and without a config file directory, we can't locate the resource
			// bundles.
		    // TODO: Need to externalize this string into a resource bundle
			throw new IllegalArgumentException("Standalone execution requires that " + 
			        "a configuration file directory is passed to the TestExecutionHarness constructor!");
		}
	}
	
	public TestExecutionHarness(String configurationFileDir, String versionString)
	{
		super();
		
		try {
			// try to load an eclipse class
			Class.forName("org.eclipse.core.boot.BootLoader"); //$NON-NLS-1$
			
			// if we didn't just throw an exception, then we know that we're
			// running in Eclipse's JVM.  Ignore the specified configurationFileDir
			this.systemUtility = ExecutionHarnessPlugin.getDefault();
		}
		catch ( ClassNotFoundException exc)
		{
			// We couldn't load a class that must be present if we're running from
			// eclipse, so we must be running standalone.
			String[] plugins = StandaloneExecutionUtilities.getConfigFilesFromConfigDir(configurationFileDir);
			setStandaloneConfigurationFiles(plugins);
			
			// Configure our utility class for standalone use			
			String harnessDir = StandaloneExecutionUtilities.getHarnessDir(configurationFileDir, versionString);
			this.systemUtility = new StandaloneSystemUtility(harnessDir);
			
			// Configure the model.hierarchy code (containing the model loaders) for
			// for standalone execution.
			RegistryReader.standaloneConfiguration = new RegistryReader.StandaloneConfiguration();
			RegistryReader.standaloneConfiguration.setEclipseFolder(configurationFileDir);
			init();

		}
	}

	private void init()
	{
		// Initialize our EMF models.
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.TEST_SUITE_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DEPLOYMENT_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.EXECUTION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.ARTIFACT_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.LOCATION_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.DATAPOOL_FILE_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.SUT_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put(ICommonConstants.TEST_COMPONENT_EXTENSION, new FacadeResourceFactoryImpl());
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("xmi", new FacadeResourceFactoryImpl()); //$NON-NLS-1$
		Common_TestprofilePackageImpl.init();
	}

	protected void releaseAgentLock() {
		synchronized(this.agentLock) {
			this.bAgentReady=true;
			this.agentLock.notify();
		}
	}

	/**
	 * This method initializes the ExecutionComponentFactory with all registered execution component
	 * types.  It also populates a testTypeMap to allow determination of which execution components
	 * should be used by which test types.
	 * 
	 * @param factory the factory to be initialized
	 * @param bStandalone whether we are running outside the workbench.  If we are, this method delegates
	 * to a method in StandaloneExecutionUtilities
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 */
	protected synchronized void initializeRegisteredExecutionComponents(
		IExecutionComponentFactory factory, boolean bStandalone, ExecutionHarnessDataProcessorFactory dpFactory) throws ClassNotFoundException, FileNotFoundException
	{

		if ( !bStandalone )
		{			
			IEclipseExecutionComponentFactory eclipseFactory = (IEclipseExecutionComponentFactory) factory;
			IExtensionRegistry reg = Platform.getExtensionRegistry();
	
			IConfigurationElement[] execComps =
				reg.getConfigurationElementsFor("org.eclipse.hyades.execution.harness.RegisteredExecutionComponentImpl"); //$NON-NLS-1$
				
			for ( int i=0; i<execComps.length; i++)
			{
				String type = execComps[i].getAttribute("type"); //$NON-NLS-1$
				
				String executionComponentName = execComps[i].getAttribute("name"); //$NON-NLS-1$
				eclipseFactory.addExecutionComponent(execComps[i]);
				
				Object map = this.testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					this.testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execComps[i].getChildren("SupportedTestType"); //$NON-NLS-1$
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), executionComponentName); //$NON-NLS-1$
				}
			}
			
			mapAdapterClasses("org.eclipse.hyades.execution.harness.ExecutionEnvironmentAdapter", "EXECUTION_ENVIRONMENT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$
			mapAdapterClasses("org.eclipse.hyades.execution.harness.ExecutableObjectAdapter", "EXECUTABLE_OBJECT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$
			mapAdapterClasses("org.eclipse.hyades.execution.harness.ExecutionDeploymentAdapter", "EXECUTION_DEPLOYMENT_ADAPTER", bStandalone); //$NON-NLS-1$ //$NON-NLS-2$
			
		}
		else
		{
			StandaloneExecutionUtilities.initializeRegisteredExecutionComponents(this.standaloneConfigurationFiles, this.testTypeMap, factory, dpFactory);
		}
	}
	
	private void mapAdapterClasses(String configElements, String type, boolean bStandalone) {
		
		if ( !bStandalone )
		{
			IConfigurationElement[] execEnvAdapters = 
				Platform.getExtensionRegistry().getConfigurationElementsFor(configElements);
				
			for ( int i=0; i<execEnvAdapters.length; i++)
			{
				Object map = this.testTypeMap.get(type);
				if ( map == null )
				{
					map = new HashMap();
					this.testTypeMap.put(type, map);
				}
				HashMap execCompMap = (HashMap)map;
				
				IConfigurationElement[] supportedTestTypes = execEnvAdapters[i].getChildren("SupportedTestType"); //$NON-NLS-1$
				for ( int j=0; j<supportedTestTypes.length; j++ )
				{
					execCompMap.put(supportedTestTypes[j].getAttribute("name"), execEnvAdapters[i]); //$NON-NLS-1$
				}
			}
		}
	}

	/**
	 * This method queries the testTypeMap and determines which execution component should be
	 * used for a given test type and execution component category.
	 * @param executionComponentType the category of execution component being queried
	 * @param testType the test type being queried
	 * @param bStandalone are we running outside the workbench
	 * @return the type of execution component that should be used (which can be passed to the factory)
	 * @throws ClassNotFoundException
	 */
	protected synchronized String getExecutionComponentForTestType( String executionComponentType, String testType, boolean bStandalone )
		throws ClassNotFoundException
	{
		Object temp = this.testTypeMap.get(executionComponentType);
		if ( temp != null && temp instanceof HashMap )
		{
			HashMap execCompMap = (HashMap) temp;
			temp = execCompMap.get(testType);
			if ( temp != null && temp instanceof String )
			{
				return (String) temp;
			}
		}
		
		String msg = this.systemUtility.getString("EXEC_NOT_FOUND_ERR_"); //$NON-NLS-1$
		msg = StringUtil.change(msg, "%1", executionComponentType); //$NON-NLS-1$
		msg = StringUtil.change(msg, "%2", testType); //$NON-NLS-1$
		
		// We couldn't find the right execution component
		throw new ClassNotFoundException(msg);
		  		
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError )
	{
		return this.launchTest(suite, theTest, machineName, port, executionResultLocation, executionResultName, bMonitor, bStandalone, bufError, this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError, IProgressMonitor progressMonitor) {
		ArrayList dpIDs = null;
		if(bMonitor)
		{
			dpIDs =  new ArrayList();
			dpIDs.add(XMLExecutionDataProcessor.IID);
		}
		
		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				IImplementor theImplementor = this.getImplementor(suite, theTest);
				if (!progressMonitor.isCanceled()) {
					executor = launchTest(theImplementor, null, machineName, null, port, executionResultLocation, executionResultName, bStandalone, dpIDs, bufError, progressMonitor);
				}
			} 
			return executor;
		} finally {
			progressMonitor.done();
		}

	}

	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use #launchTest(ITestSuite, ITest, TPFDeployment, String, String, String, boolean, boolean, boolean, StringBuffer) instead.
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , TPFDeployment deployment, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError )
	{
		return this.launchTest(suite, theTest , deployment, port, executionResultLocation, executionResultName, bMonitor, bStandalone, bufError, this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use #launchTest(ITestSuite, ITest, TPFDeployment, String, String, String, boolean, boolean, boolean, StringBuffer) instead.
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , TPFDeployment deployment, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError, IProgressMonitor progressMonitor)
	{
		return this.launchTest(suite, theTest , deployment, port, executionResultLocation, executionResultName, false, bMonitor, bStandalone, bufError, progressMonitor);
	}
	
	/**
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param overrideExisting whether override existing if there is such an execution result already exist
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , TPFDeployment deployment, String port, String executionResultLocation, String executionResultName, boolean overrideExisting, boolean bMonitor, boolean bStandalone, StringBuffer bufError )
	{		
		return this.launchTest(suite, theTest, deployment, port, executionResultLocation, executionResultName, overrideExisting, bMonitor, bStandalone, bufError, this.createNullProgressMonitor());
	}
	
	/**
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param overrideExisting whether override existing if there is such an execution result already exist
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized IExecutor launchTest( ITestSuite suite, ITest theTest , TPFDeployment deployment, String port, String executionResultLocation, String executionResultName, boolean overrideExisting, boolean bMonitor, boolean bStandalone, StringBuffer bufError, IProgressMonitor progressMonitor) {		
		ArrayList dpIDs = null;
		if(bMonitor)
		{
			dpIDs =  new ArrayList();
			dpIDs.add(XMLExecutionDataProcessor.IID);
		}

		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				IImplementor theImplementor = getImplementor(suite, theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, null, deployment, port, executionResultLocation, executionResultName, overrideExisting, bStandalone, dpIDs, bufError, progressMonitor);
				}
			} 
			return executor;
		} finally {
			progressMonitor.done();
		}
	
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param deployment the deployment with which to launch the test 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file	 
	 * @param bStandalone are we not launching from the workbench
	 * @param activeDataProcessorIDs a String[] of the dataprocessors that should be activated for this launch.  This ID should match the ID provided in the extension.
	 * @return a string containing an error message, or null if no error occurred
	 */
	public IExecutor launchTest( ITestSuite suite, 
										   ITest theTest, 
										   TPFDeployment deployment, 
										   String port, 
										   String executionResultLocation, 
										   String executionResultName, 
										   boolean bStandalone, 
										   ArrayList activeDataProcessorIDs,
										   StringBuffer bufError) {
		return this.launchTest(
			suite,
			theTest,
			deployment,
			port,
			executionResultLocation,
			executionResultName,
			bStandalone,
			activeDataProcessorIDs,
			bufError,
			this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param deployment the deployment with which to launch the test 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file	 
	 * @param bStandalone are we not launching from the workbench
	 * @param activeDataProcessorIDs a String[] of the dataprocessors that should be activated for this launch.  This ID should match the ID provided in the extension.
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 */
	public IExecutor launchTest( ITestSuite suite, 
			   ITest theTest, 
			   TPFDeployment deployment, 
			   String port, 
			   String executionResultLocation, 
			   String executionResultName, 
			   boolean bStandalone, 
			   ArrayList activeDataProcessorIDs,
			   StringBuffer bufError,
			   IProgressMonitor progressMonitor) {
		
		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				IImplementor theImplementor = this.getImplementor(suite, theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, null, deployment, port, executionResultLocation, executionResultName, bStandalone, activeDataProcessorIDs, bufError, progressMonitor);
				}
			} 
			return executor;
		} finally {
			progressMonitor.done();
		}
		
	}

	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param overrideTestType if specified, the test type to be used by the execution
	 *        harness (regardless of the test type specified by the test)
	 * @param deployment the deployment with which to launch the test 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file	 
	 * @param bStandalone are we not launching from the workbench
	 * @param activeDataProcessorIDs a String[] of the dataprocessors that should be activated for this launch.  This ID should match the ID provided in the extension.
	 * @return a string containing an error message, or null if no error occurred
	 */
	public IExecutor launchTest( ITestSuite suite, 
										   ITest theTest,
										   String overrideTestType,
										   TPFDeployment deployment, 
										   String port, 
										   String executionResultLocation, 
										   String executionResultName, 
										   boolean bStandalone, 
										   ArrayList activeDataProcessorIDs,
										   StringBuffer bufError) {
		return this.launchTest(
			suite,
			theTest,
			overrideTestType,
			deployment,
			port,
			executionResultLocation,
			executionResultName,
			bStandalone,
			activeDataProcessorIDs,
			bufError, 
			this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param suite the test suite to be launched
	 * @param theTest the test to be launched (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param overrideTestType if specified, the test type to be used by the execution
	 *        harness (regardless of the test type specified by the test)
	 * @param deployment the deployment with which to launch the test 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file	 
	 * @param bStandalone are we not launching from the workbench
	 * @param activeDataProcessorIDs a String[] of the dataprocessors that should be activated for this launch.  This ID should match the ID provided in the extension.
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 */
	public IExecutor launchTest( ITestSuite suite, 
			   ITest theTest,
			   String overrideTestType,
			   TPFDeployment deployment, 
			   String port, 
			   String executionResultLocation, 
			   String executionResultName, 
			   boolean bStandalone, 
			   ArrayList activeDataProcessorIDs,
			   StringBuffer bufError,
			   IProgressMonitor progressMonitor) {
		
		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				IImplementor theImplementor = this.getImplementor(suite, theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, overrideTestType, null, deployment, port, executionResultLocation, executionResultName, bStandalone, activeDataProcessorIDs, bufError, progressMonitor);
				}
			} 
			return executor;
		} finally {
			progressMonitor.done();
		}
		
	}
	
	public IExecutor launchTest( ITestSuite suite, 
											ITest theTest, 
											String hostName, 
											String port, 
											String executionResultLocation, 
											String executionResultName, 
											boolean bStandalone, 
											ArrayList activeDataProcessorIDs,
											StringBuffer bufError) {
		return this.launchTest(
			suite,
			theTest,
			hostName,
			port,
			executionResultLocation,
			executionResultName,
			bStandalone,
			activeDataProcessorIDs,
			bufError,
			this.createNullProgressMonitor());
	}

	public IExecutor launchTest( ITestSuite suite, 
			ITest theTest, 
			String hostName, 
			String port, 
			String executionResultLocation, 
			String executionResultName, 
			boolean bStandalone, 
			ArrayList activeDataProcessorIDs,
			StringBuffer bufError,
			IProgressMonitor progressMonitor) {
		
		try {
			IExecutor executor = null;
			if (!progressMonitor.isCanceled()) {
				IImplementor theImplementor = this.getImplementor(suite, theTest);
				if (!progressMonitor.isCanceled()) {
					executor = this.launchTest(theImplementor, null, hostName, null, port, executionResultLocation, executionResultName, bStandalone, activeDataProcessorIDs, bufError, progressMonitor );
				}
			} 
			return executor;
		} finally {
			progressMonitor.done();
		}
			
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use {@link #launchTest( String rootSuitePath, String deploymentPath, String testID, String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError)} instead.
	 */
	public IExecutor launchTest( String rootSuitePath, 
							  String testID, 
							  String machineName, 
							  String port, 
							  String executionResultLocation, 
							  String executionResultName, 
							  boolean bMonitor, 
							  boolean bStandalone,
							  StringBuffer bufError) {
		return this.launchTest(
			rootSuitePath,
			testID,
			machineName,
			port,
			executionResultLocation,
			executionResultName,
			bMonitor,
			bStandalone,
			bufError,
			this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use {@link #launchTest( String rootSuitePath, String deploymentPath, String testID, String machineName, String port, String executionResultLocation, String executionResultName, boolean bMonitor, boolean bStandalone, StringBuffer bufError)} instead.
	 */
	public IExecutor launchTest( String rootSuitePath, 
			  String testID, 
			  String machineName, 
			  String port, 
			  String executionResultLocation, 
			  String executionResultName, 
			  boolean bMonitor, 
			  boolean bStandalone,
			  StringBuffer bufError,
			  IProgressMonitor progressMonitor) {
		
		ITest theTest = null;
		ITestSuite suite = null;
		TPFDeployment deployment = null;
		bufError.setLength(0);
		
		// 	Load the test suite.
		try {
			suite = HyadesFactory.INSTANCE.loadTestSuite(rootSuitePath);
		}
		catch ( Exception exc )
		{
			this.systemUtility.logError(exc);
			bufError.append(StringUtil.change(
				this.systemUtility.getString("LOAD_SUITE_ERR_"), //$NON-NLS-1$
				"%1", //$NON-NLS-1$
				rootSuitePath));
			return null;
		}
		
		// 	Create a Deployment with the test suite and machine name
		if(suite instanceof CFGClass)
			deployment = ConfigurationUtil.createDeploymentWithPair(machineName, null, (CFGClass)suite, machineName);
		
		// 	If we're executing a test case and not a suite, get the test case.
		if ( testID != null && testID.length() != 0 )
		{
			theTest = (ITest) HyadesFactory.INSTANCE.getObjectByID(suite, testID);
		}
		else
		{
			theTest = suite;
		}

		return launchTest(suite, theTest, deployment, port, executionResultLocation, executionResultName, bMonitor, bStandalone, bufError, progressMonitor);
	}
	
	/**
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param deploymentPath fully qualified path to the deployment to be used with execution.
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use #launchTest( String rootSuitePath, 
								 String deploymentPath,
								 String testID, 
								  String machineName, 
								  String port, 
								  String executionResultLocation, 
								  String executionResultName, 
								  boolean overwriteExistingResult,
								  boolean bMonitor, 
								  boolean bStandalone,
								  StringBuffer bufError) instead.
	 */
	public IExecutor launchTest( String rootSuitePath, 
								 String deploymentPath,
								 String testID, 
								  String machineName, 
								  String port, 
								  String executionResultLocation, 
								  String executionResultName, 
								  boolean bMonitor, 
								  boolean bStandalone,
								  StringBuffer bufError)
	{
		return this.launchTest(
			rootSuitePath,
			deploymentPath,
			testID,
			machineName,
			port,
			executionResultLocation,
			executionResultName,
			bMonitor,
			bStandalone,
			bufError,
			this.createNullProgressMonitor());
	}
	
	/**
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param deploymentPath fully qualified path to the deployment to be used with execution.
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use #launchTest( String rootSuitePath, 
								 String deploymentPath,
								 String testID, 
								  String machineName, 
								  String port, 
								  String executionResultLocation, 
								  String executionResultName, 
								  boolean overwriteExistingResult,
								  boolean bMonitor, 
								  boolean bStandalone,
								  StringBuffer bufError,
								  IProgressMonitor progressMonitor) instead.
	 */
	public IExecutor launchTest( String rootSuitePath, 
			 String deploymentPath,
			 String testID, 
			  String machineName, 
			  String port, 
			  String executionResultLocation, 
			  String executionResultName, 
			  boolean bMonitor, 
			  boolean bStandalone,
			  StringBuffer bufError,
			  IProgressMonitor progressMonitor) {
		return this.launchTest(rootSuitePath, 
			deploymentPath,
			testID, 
			machineName, 
			port, 
			executionResultLocation, 
			executionResultName, 
			false,
			bMonitor, 
			bStandalone,
			bufError,
			progressMonitor);
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param deploymentPath fully qualified path to the deployment to be used with execution.
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @return a string containing an error message, or null if no error occurred
	 */
	public IExecutor launchTest( String rootSuitePath, 
								 String deploymentPath,
								 String testID, 
								  String machineName, 
								  String port, 
								  String executionResultLocation, 
								  String executionResultName, 
								  boolean overwriteExistingResult,
								  boolean bMonitor, 
								  boolean bStandalone,
								  StringBuffer bufError) {
		return this.launchTest(
			rootSuitePath,
			deploymentPath,
			testID,
			machineName,
			port,
			executionResultLocation,
			executionResultName,
			overwriteExistingResult,
			bMonitor,
			bStandalone,
			bufError,
			this.createNullProgressMonitor());
	}
	
	/**
	 * 
	 * Launches the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>

	 * @param rootSuitePath the fully qualified path to the test suite to be launched
	 * @param deploymentPath fully qualified path to the deployment to be used with execution.
	 * @param testID the ID of the test to be launched (may be null, or may 
	 *        be the ID of a test case contained within suite.
	 * @param machineName the machine name on which the test will be launched 
	 *        (IP address or host name) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param executionResultLocation the directory in which to write the execution history file
	 * @param executionResultName the filename for the execution history file
	 * @param bMonitor should we monitor the test
	 * @param bStandalone are we not launching from the workbench
	 * @param progressMonitor progress monitor to use
	 * @return a string containing an error message, or null if no error occurred
	 */	
	public IExecutor launchTest( String rootSuitePath, 
			 String deploymentPath,
			 String testID, 
			  String machineName, 
			  String port, 
			  String executionResultLocation, 
			  String executionResultName, 
			  boolean overwriteExistingResult,
			  boolean bMonitor, 
			  boolean bStandalone,
			  StringBuffer bufError,
			  IProgressMonitor progressMonitor) {
		
		ITest theTest = null;
		ITestSuite suite = null;
		TPFDeployment deployment = null;
		bufError.setLength(0);
		
		// 	Load the test suite.
		try {
			suite = HyadesFactory.INSTANCE.loadTestSuite(rootSuitePath);
		}
		catch ( Exception exc )
		{
			this.systemUtility.logError(exc);
			bufError.append(StringUtil.change(
				this.systemUtility.getString("LOAD_SUITE_ERR_"), //$NON-NLS-1$
				"%1", //$NON-NLS-1$
				rootSuitePath));
			return null;
		}
		
		// 	Load the deployment if it is specified.
		try {
			if(deploymentPath != null && deploymentPath.length() > 0)
				deployment = HyadesFactory.INSTANCE.loadDeployment(deploymentPath);
		}
		catch ( Exception exc ){}
		// Create a Deployment with the test suite and machine name
		if(deployment == null && suite instanceof CFGClass)
			deployment = ConfigurationUtil.createDeploymentWithPair(machineName, null, (CFGClass)suite, machineName);
		
		// 	If we're executing a test case and not a suite, get the test case.
		if ( testID != null && testID.length() != 0 )
		{
			theTest = (ITest) HyadesFactory.INSTANCE.getObjectByID(suite, testID);
		}
		else
		{
			theTest = suite;
		}
		
		return this.launchTest(suite, theTest, deployment, port, executionResultLocation, executionResultName, overwriteExistingResult, bMonitor, bStandalone, bufError, progressMonitor);
	}
	
	private synchronized IExecutor launchTest( IImplementor theImplementor, 
			String overrideTestType, 
			String machineName, 
			TPFDeployment deployment, 
			String port, 
			String executionResultLocation, 
			String executionResultName, 
			boolean bStandalone, 
			ArrayList activeDataProcessorIDs, 
			StringBuffer bufError,
			IProgressMonitor progressMonitor) {
		return this.launchTest( theImplementor, 
			overrideTestType, 
			machineName, 
			deployment, 
			port, 
			executionResultLocation, 
			executionResultName, 
			false, 
			bStandalone, 
			activeDataProcessorIDs, 
			bufError,
			progressMonitor);
	}
	
	/**
	 * 
	 * @param theImplementor
	 * @param overrideTestType
	 * @param machineName
	 * @param deployment
	 * @param port
	 * @param executionResultLocation
	 * @param executionResultName
	 * @param overrideExistingResult
	 * @param bStandalone
	 * @param activeDataProcessorIDs
	 * @param bufError
	 * @param progressMonitor
	 * @return
	 */
	private synchronized IExecutor launchTest( IImplementor theImplementor, 
			String overrideTestType, 
			String machineName, 
			TPFDeployment deployment, 
			String port, 
			String executionResultLocation, 
			String executionResultName,
			boolean overrideExistingResult,
			boolean bStandalone, 
			ArrayList activeDataProcessorIDs, 
			StringBuffer bufError,
			IProgressMonitor progressMonitor)
	{
			
		// Begin task with work units allocated
		progressMonitor.beginTask("", 100); //$NON-NLS-1$
		
		// Reinitialize all fields
		this.factory = null;
		this.session = null;
		this.exeEnvironment = null;
		this.executor = null;		
		
		// Empty the error buffer
		bufError.setLength(0);
		
		// Determine what type of implementor we're dealing with
		CFGClass rootResource = null;

		ITest owningTest = theImplementor.getOwner();

		if ( owningTest != null )
		{
			if ( owningTest instanceof ITestSuite )
				rootResource = (CFGClass) owningTest;
			else 
			{
				rootResource = (CFGClass)((ITestCase)owningTest).getOwner();
			}
		}
		else
		{
			// If the implementor is not owned by a test, then it must
			// be owned by a test component
			TPFTestComponent component = ((TPFBehavior)theImplementor).getTestComponent();
			
			if ( component.eContainer() == null )
			{
				// This is a standalone test component.
				rootResource = component;
			}
			else
			{
				// This test component is contained by a test suite
				rootResource = (CFGClass) component.eContainer();
			}			
		}
	
		try {
			// Allow the test type to be overridden if the caller has specified.
			String testType;
			if ( overrideTestType != null && overrideTestType.length() > 0 )
				testType = overrideTestType;
			else
				testType = owningTest.getType();
			
			String machine = machineName;
			
			// determine deployment and hostname
			if (deployment == null)
			{
				deployment = ConfigurationUtil.createDeploymentWithPair(null, null, rootResource, machineName);
			}
			else
			{
				machine = ConfigurationUtil.getHostNameForTestAsset(rootResource, deployment);			
				if(machine == null) {
					machine = machineName;
				}
			}
			
			if(machine == null) {
			    bufError.append(this.systemUtility.getString("NULL_HOST_NAME_ERR")); 
				return null;
			}
			
			// Get a session for this Node; allocate work units to session connect sub progress monitor
			this.session = this.sessionConnect(port, this.session, machine, bufError, new SubProgressMonitor(progressMonitor, 25));
			
			// poll for cancel status
			if (progressMonitor.isCanceled()) {
				return this.executor;
			}
			
			if ( this.session == null )
			{
				return null;
			}
			
			if ( bStandalone )
			{		
				this.factory = ExecutionComponentFactoryImpl.getInstance(this.session);
			}
			else
			{
				this.factory = EclipseExecutionComponentFactoryImpl.getInstance(this.session);			
			}
			
			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return this.executor;
			} 
			progressMonitor.worked(10);
			
			if ( !this.bFactoryInitialized )
			{
				//initialize the dataPRocessor factory
				this.dpFactory = new ExecutionHarnessDataProcessorFactory(bStandalone);
				
				initializeRegisteredExecutionComponents(this.factory, bStandalone, this.dpFactory);
				// TODO: investigate this -- is the session not being reused??
				this.bFactoryInitialized = true;
			}
			
			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return this.executor;
			} 
			progressMonitor.worked(10);
						
			// Create the appropriate type of execution environment
			String envType = getExecutionComponentForTestType("ENVIRONMENT", testType, bStandalone); //$NON-NLS-1$
			this.exeEnvironment = (IExecutionEnvironment) this.factory.createExecutionComponentByType(envType);
			this.session.addChild(this.exeEnvironment);
			
			if ( !ExecutionAdapterUtilities.adaptExecutionDeployment(this.session.getNode(), deployment, bStandalone, testType, bufError, this.systemUtility, this.testTypeMap))
				return null;
	
			// Call out into an adapter class to configure this execution 
			// environment for this test
			if ( !ExecutionAdapterUtilities.adaptExecutionEnvironment(
					this.exeEnvironment, rootResource, theImplementor, bStandalone, testType, 
					deployment, bufError, this.systemUtility, this.testTypeMap))
				return null;
			
			// Create the appropriate type of executor
			String executorType = getExecutionComponentForTestType("EXECUTOR", testType, bStandalone); //$NON-NLS-1$
			this.executor = (IExecutor) this.factory.createExecutionComponentByType(executorType);
			this.exeEnvironment.addChild(this.executor);
						
			String execObjType = getExecutionComponentForTestType("EXECUTABLEOBJECT", testType, bStandalone); //$NON-NLS-1$
			IExecutableObject executableObject = this.executor.getCompatibleExecutableObject(execObjType);
			
			//Call out into an adapter class to configure this execution 
		  	// environment for this test
		    if ( !ExecutionAdapterUtilities.adaptExecutableObject(
		    		executableObject, rootResource, theImplementor, bStandalone, testType, deployment, bufError, this.systemUtility, this.testTypeMap))
		    {
			    return null;
		    }		
			
			if(executableObject!=null) 
			{
				 // Set our executable object -- this will move it to the remote side	
				 this.executor.setExecutableObject(executableObject);
	
				 if (this.executor instanceof ExecutorStub)
				 {
					((ExecutorStub) this.executor).setExecutionResultLocation(executionResultLocation);
					((ExecutorStub) this.executor).setExecutionResultName(executionResultName);
					((ExecutorStub) this.executor).setCommunicationPort(port);
				 }
			}					
							 
			//get the agentType that we need to create
			String agentType = getExecutionComponentForTestType("AGENT", testType, bStandalone); //$NON-NLS-1$
			
			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return this.executor;
			} 
			progressMonitor.worked(10);
			
			if(activeDataProcessorIDs!=null && activeDataProcessorIDs.size() > 0 && this.executor instanceof ExecutorStub)
			{			   	 
				 ArrayList activeDataProcessors = initValidDataProcessors(owningTest, machine, agentType, executionResultName, executionResultLocation, overrideExistingResult, port, activeDataProcessorIDs);
				 				 
				 //now we need to set the agent data which will be intercepted by the exec history dp 
				 //this data will enable it to bootstrap the others.
				 IExecutionHarnessDataProcessor[] dpArray = convertListToArray(activeDataProcessors);
				 setAgentData(rootResource, executableObject,dpArray);
				 ((ExecutorStub) this.executor).setDataProcessors(dpArray);
		 	}
		 	else
		 	{
		 		initializeRemoteHyadesComponent(agentType,null);	 		
		 	}
			
			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return this.executor;
			} 
			progressMonitor.worked(10);
			
			// if this executor supports progress monitor support, pass some work units in to the launch
			if (this.executor instanceof IExecutorWithProgressMonitorSupport) {
				((IExecutorWithProgressMonitorSupport) this.executor).launch(new SubProgressMonitor(progressMonitor, 25));
			} else {
				this.executor.launch();
			}
			
			// poll for cancel status and track progress
			if (progressMonitor.isCanceled()) {
				return this.executor;
			}
			progressMonitor.worked(10);
			
		}			
		catch(ClassNotFoundException e) {
			
			this.systemUtility.logError(e);	
			String msg = e.getMessage();
			if ( msg == null )
				msg = e.toString();
			
			bufError.append( StringUtil.change(
								this.systemUtility.getString("CLASS_NOT_FOUND_ERR_"), //$NON-NLS-1$
								"%1", //$NON-NLS-1$
								msg));
		}
		catch(Throwable t) {
			this.systemUtility.logError(t);			
			String msg = t.getMessage();
			if ( msg != null )
				bufError.append(t.getMessage());
			else
				bufError.append(t.toString());
		}
		finally {
			progressMonitor.done();
		}
		
		return this.executor;
	}



	private IExecutionHarnessDataProcessor[] convertListToArray(ArrayList registeredDataProcessorsForTestType)
	{
		IExecutionHarnessDataProcessor dp[] = new IExecutionHarnessDataProcessor[registeredDataProcessorsForTestType.size()];
		for(int i=0; i<registeredDataProcessorsForTestType.size(); i++)
		{
			dp[i]=(IExecutionHarnessDataProcessor)registeredDataProcessorsForTestType.get(i);
		}
		return dp;
	}

	private void setAgentData(
		CFGClass rootResource,
		IExecutableObject executableObject,
		IDataProcessor[] dataProcessors)
	{
		if(executableObject instanceof JavaProcessExecutableObjectStub && dataProcessors!=null && dataProcessors.length>1)
		{
		   //since the xmlexecutionhistorydataprocessor is initialized from main(), it does not need to be passed in the 
		   //agent data String.  Therefore, will skip it in the collection as we build the list
		   //it is always first in the dataProcessors collection regardless of when it was sent in the spec ArrayList.
		   String dpString = "AgentIDs:"; //$NON-NLS-1$
		   for(int i=1; i<dataProcessors.length; i++)
		   {
		  	  dpString+=((IExecutionHarnessDataProcessor)dataProcessors[i]).getID();
		 	  if(i<dataProcessors.length-1)
				  dpString+=","; //$NON-NLS-1$
		
		   }
		   ((JavaProcessExecutableObjectStub)executableObject).setAgentData(dpString);
		
		}
		
	}


	private void initializeRemoteHyadesComponent(String agentType,	IExecutionHarnessDataProcessor dp)
	{
		IRemoteHyadesComponent remoteComponent = (IRemoteHyadesComponent) this.factory.createExecutionComponentByType(agentType);
			
		this.executor.addChild(remoteComponent);					
				
		// Create a state change listener on the Agent.
		remoteComponent.addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() 
		{
			public void stateChanged(ExecutionComponentStateChangeEvent event) {
				// When the agent is ready, release the agent lock
				if(event.getState()==IExecutionComponent.READY) {
					releaseAgentLock();
				}	
			}
		});	
		
														
		if(dp!=null)
			remoteComponent.startMonitoring(dp);
				
		//	Wait for the agent to become active
		waitForActiveAgent(remoteComponent);
	}
	
	/**
	 * initializes dataprocessors that are valid for this run.
	 * @param theTest
	 * @param hostName
	 * @param agentType
	 * @param executionResultName
	 * @param executionResultLocation
	 * @param port
	 * @param validDPIds
	 * @return
	 * @throws CoreException
	 */
	private ArrayList initValidDataProcessors(ITest theTest, String hostName, String agentType, String executionResultName, String executionResultLocation, boolean overrideExisting, String port, ArrayList validDPIds) 
	{
				
		ArrayList retList = new ArrayList();
		ArrayList dpList = this.dpFactory.getDataProcessors();
		for ( int i=0; i<dpList.size(); i++)
		{
		   IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor)dpList.get(i);
		   if(validDPIds.contains(dataProcessor.getID()))
		   {	   
			   if(dataProcessor instanceof IExecutionHarnessDataProcessor2) {
			   		((IExecutionHarnessDataProcessor2)dataProcessor).setInitData(theTest,hostName,executionResultName,executionResultLocation,overrideExisting,port);
			   } else {
			   		dataProcessor.setInitData(theTest,hostName,executionResultName,executionResultLocation,port);
			   }
			   
			   //this should ensure that the executionhistory dataprocessor is first in the list
			   if(dataProcessor.getID().equalsIgnoreCase(XMLExecutionDataProcessor.IID)) {
			   		retList.add(0,dataProcessor);
			   } else {
			   		retList.add(dataProcessor);
			   }
		   }				   
		} 
		
		//now lets go through the collection again and init them all starting with the execution history dp if it's present
		//if it is not present, we will send a dummy control agent in it's place.
		for (int i=0; i<retList.size(); i++)
		{
			IExecutionHarnessDataProcessor dp =(IExecutionHarnessDataProcessor)retList.get(i); 
			if(i==0 && dp.getID().compareToIgnoreCase(XMLExecutionDataProcessor.IID)!=0) {
				initializeRemoteHyadesComponent(agentType,null);
			} else {
				initializeRemoteHyadesComponent(agentType,dp);
			}
		}
		
		return retList;
	}



	
	
	private void waitForActiveAgent(IRemoteHyadesComponent agent) 
	{				
		if ( agent.getState() == IExecutionComponent.SUSPENDED)
		{
			// Block and wait for the agent to come up.
			while(!this.bAgentReady) {
				synchronized(this.agentLock) {
					try {
						this.agentLock.wait();
					}
					catch(InterruptedException e) {
						/* We can ignore this */
					}
				}
			}
		}
	}
	
	/**
	 * Given a port, session, machine, error buffer and progress monitor, connect to the session.
	 * 
	 * @param port the port to connect to
	 * @param machine the machine to connect to
	 * @param bufError a buffer for errors to accumulate
	 * @param progressMonitor monitors progress and is polled to check cancel status
	 * @return the 
	 */
	private ISession sessionConnect(final String port, ISession session, final String machine, final StringBuffer bufError, final IProgressMonitor progressMonitor) {
	    
		// Begin the task of session connect
		progressMonitor.beginTask("", 20); //$NON-NLS-1$
		
		// Initialize error buffer
		bufError.setLength(0);
		
		// Use a mutable object to wrap session so it can be set within connect thread
		final MutableObject mutableSession = new MutableObject(session);
		
		try {
			
			// Use another thread to increase responsiveness of cancel when connecting
			Runnable connectRunnable = new Runnable() {
				public void run() {
					try {
						INode node = new HyadesTestNodeImpl(machine);
						progressMonitor.worked(5);
						mutableSession.setValue(node.connect(port, null));
						progressMonitor.worked(5);
					} catch (UnknownHostException e) {
						bufError.append(StringUtil.change(TestExecutionHarness.this.systemUtility.getString("TIMEOUT_NODE_ERROR_"), "%1", machine)); //$NON-NLS-1$ //$NON-NLS-2$
					} catch (AgentControllerUnavailableException e) {
						bufError.append(StringUtil.change(TestExecutionHarness.this.systemUtility.getString("AGENT_UNAV_ERROR_"), "%1", machine)); //$NON-NLS-1$ //$NON-NLS-2$
					} catch (UnknownDaemonException e) {
						bufError.append(StringUtil.change(TestExecutionHarness.this.systemUtility.getString("AGENT_UNAV_ERROR_"), "%1", machine)); //$NON-NLS-1$ //$NON-NLS-2$
					} catch(Exception e) {
					    TestExecutionHarness.this.systemUtility.logError(e);			
						bufError.append(StringUtil.change(TestExecutionHarness.this.systemUtility.getString("RUN_FAILED_ERR_"), "%1", machine)); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			};
			
			/* 
			 * Create a new thread hooked up with the connect runnable and start the thread;
			 * creates as a daemon thread in case interrupt isn't effective and a cancel causes
			 * execution to continue without the connect thread ending.  Basically exiting the
			 * VM is okay even if the connect thread is still alive.
			 */
			Thread connectThread = new Thread(connectRunnable, "Hyades Session Connect");  //$NON-NLS-1$
			connectThread.setDaemon(true);
			connectThread.start();
			
			// Primary thread blocks at intervals and then polls the cancel status of the progress monitor
			try {
			    for (boolean isCanceled = false; connectThread.isAlive() && !isCanceled; ) {
			        progressMonitor.worked(1);
		            if (progressMonitor.isCanceled()) {
		                isCanceled = true;
		                connectThread.interrupt();
		            } else {
		                connectThread.join(1000);
		            }
			    }
			} catch (InterruptedException e) {
		    }
			
		}
		finally {
			progressMonitor.done();
		}
		
		return (ISession) mutableSession.getValue();
	}

	private ISession getSession(String port, ISession session, String machine, StringBuffer bufError) {
		bufError.setLength(0);
		// TODO process Error message
		SessionContext ctx = new SessionContext(machine, port, null);
		return SessionContextToSessionMap.getInstance().getSessionEntry(ctx);
	}

	/**
	 * Sets the standaloneConfigurationFiles member to an array of strings containing
	 * all registered execution components and adapters.
	 * @param standaloneConfigurationFiles
	 */
	
	
	private void setStandaloneConfigurationFiles(String[] standaloneConfigurationFiles) {
		this.standaloneConfigurationFiles = standaloneConfigurationFiles;
	}

	/**
	 * Stops the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>
	 * @param suite the test suite to be stopped
	 * @param theTest the test to be stopped (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param location on which the test will be stopped 
	 *        (CGFLocation or CFGMachine) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param execResult the execution result
	 * @return a string containing an error message, or null if no error occurred
	 * @deprecated use #stopTest( IExecutor, TPFExecutionResult) instead
	 */
	public synchronized String stopTest(
		ITestSuite suite,
		ITest theTest,
		CFGLocation location,
		String port,
		TPFExecutionResult execResult) {
	
		String machine = ((CFGMachineConstraint) location).getHostname();
		StringBuffer bufError = new StringBuffer();
	
		// Get a session for this Node.  This is returning an session that was created during launch test method		
		this.session = getSession(port, this.session, machine, bufError);
		if (this.session == null) {
			return StringUtil.change(this.systemUtility.getString("NO_SESSION_FOUND_ERROR"), "%1", machine); //$NON-NLS-1$ //$NON-NLS-2$			
		}
	
		try {
			boolean testStopped = false;
			boolean testFound = false;
			ArrayList componentstoremove = new ArrayList();

			IExecutionComponent[] exeEnvironments = this.session.getChildren();
			// Navigate to the IExecutor and invoke kill method to stop the running test.
			for (int i = 0; i < exeEnvironments.length; i++) {
				if (exeEnvironments[i] instanceof IExecutionEnvironment) {
					IExecutionComponent[] executors = exeEnvironments[i].getChildren();
					for (int j = 0; j < executors.length; j++) {
						if (executors[j] instanceof ExecutorStub) {
							ExecutableObjectStub exeobj = (ExecutableObjectStub)((IExecutor) executors[j]).getExecutableObject();
							if (exeobj.getTestID().equals(theTest.getId())
								&& exeobj.getSuiteID().equals(suite.getId())
								&& ((ExecutorStub) executors[j]).getExecutionResultName().equals(execResult.getName())) {

								testFound = true;
								IExecutionComponent[] agents = executors[j].getChildren();
								// Stop monitoring call to the agent is necessary for clean-up including saving of the 
								// execution results to occur.
								for (int k = 0; k < agents.length; k++){
									if (agents[k] instanceof IRemoteHyadesComponent){
										stopMonitoringAndWait((IRemoteHyadesComponent) agents[k]);
										componentstoremove.add(agents[k]);
									}
								}
								((IExecutor) executors[j]).kill();
								removeChildComponents(componentstoremove, executors[j]);
								componentstoremove.clear();
								testStopped = true;
								if (executors[j].getChildren().length == 0){
									componentstoremove.add(executors[j]);
								}
							}
						}
					}
					removeChildComponents(componentstoremove, exeEnvironments[i]);
					componentstoremove.clear();
					if (exeEnvironments[i].getChildren().length == 0){
						componentstoremove.add(exeEnvironments[i]);
					}
				}
			}
			removeChildComponents(componentstoremove, this.session);

			if (!testStopped) {
				if (!testFound){
					return this.systemUtility.getString("NO_TEST_TO_STOP_ERROR_"); //$NON-NLS-1$		
				}
				return this.systemUtility.getString("TEST_STOP_ERROR_"); //$NON-NLS-1$
			}
			logStopEvents(execResult);
			execResult.eResource().save(Collections.EMPTY_MAP);
			
		} catch (Throwable t) {
			this.systemUtility.logError(t);
			return t.getMessage();
		}
		return null;
	}
	
	/**
	 * Stops the specified test on the specified machine.  The flow in the 
	 * launchTest method is depicted in the execution use case realization: 
	 * <a href="http://dev.eclipse.org/viewcvs/indextools.cgi/~checkout~/hyades-home/docs/components/execution_environment/Launch_A_Test_On_A_Specified_Node_And_Send_Control_Events.html"> UC1.1 </a>
	 * 
	 * @param executor the executor being used to execute the test.
	 * @param execResult the execution result
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized String stopTest( IExecutor executor, TPFExecutionResult execResult )
	{
		try {
			if (!stopTest(executor)) {
				return this.systemUtility.getString("TEST_STOP_ERROR_"); //$NON-NLS-1$
			}
			logStopEvents(execResult);
			execResult.eResource().save(Collections.EMPTY_MAP);
		} catch (Throwable t) {
			this.systemUtility.logError(t);
			return t.getMessage();
		}
		
		return null;		
	}

	private boolean stopTest( IExecutor executor )
	{
		IExecutionComponent[] agents = executor.getChildren();
		// Stop monitoring call to the agent is necessary for clean-up including saving of the 
		// execution results to occur.
		ArrayList componentsToRemove = new ArrayList();
		for (int i = 0; i < agents.length; i++){
			if (agents[i] instanceof IRemoteHyadesComponent){
				try {
					stopMonitoringAndWait((IRemoteHyadesComponent) agents[i]);
					componentsToRemove.add(agents[i]);
				} catch (InterruptedException e) {
					this.systemUtility.logError(e);
				}				
			}
		}
		// TODO check with Joe only remove children from executor?
		// remove from exeEnvironment, session?
		removeChildComponents(componentsToRemove, executor);
		componentsToRemove.clear();
		executor.kill();		
		return true;
	}

	private void logStopEvents(TPFExecutionResult result)
	{
		String msg = this.systemUtility.getString("TEST_ABORTED_MESSAGE_"); //$NON-NLS-1$
		// verdict
		TPFVerdictEvent verdict = Common_TestprofileFactory.eINSTANCE.createTPFVerdictEvent();
		verdict.setVerdict(TPFVerdict.INCONCLUSIVE_LITERAL);
		verdict.setReason(TPFVerdictReason.ABORT_LITERAL);
		verdict.setText(msg);
		verdict.setExecutionHistory(result.getExecutionHistory());
		
		// stop
		TPFTypedEvent stopEvent = Common_TestprofileFactory.eINSTANCE.createTPFTypedEvent();
		stopEvent.setType(TPFExecutionType.STOP_LITERAL);
		stopEvent.setText(msg);
		stopEvent.setTimestamp(System.currentTimeMillis());
		stopEvent.setExecutionHistory(result.getExecutionHistory());
		
		// promote verdict to result
		result.setVerdict(TPFVerdict.INCONCLUSIVE_LITERAL);
		
	}
	
	private void stopMonitoringAndWait(IRemoteHyadesComponent agent) throws InterruptedException{
		// State does not change when an agent goes from Monitoring to 
		// Not monitoring state.  This bug need to get fixed.
		// Until then just sleeping the thread for 3 secs. 
/*		((IRemoteHyadesComponent) agent).addExecutionComponentStateChangeListener(new IExecutionComponentStateChangeListener() {
			public void stateChanged(ExecutionComponentStateChangeEvent event) {
				// When the agent is dead, release the agent lock
				if(event.getState()==IExecutionComponent.DEAD) {
					releaseAgentLock();
				}	
			}
		});*/	
		agent.stopMonitoring();
		Thread.sleep(3000);
//		waitForInactiveAgent((IRemoteHyadesComponent) agent);
	}
	private void removeChildComponents(ArrayList children, IExecutionComponent parent){
		for (int i = 0; i < children.size(); i++){
			parent.removeChild((IExecutionComponent) children.get(i));
		}
	}

	/**
	 * Perfromed the control event on the specified test on the specified machine.  
	 * @param suite the test suite to be controlled
	 * @param theTest the test to be suspended (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param location on which the test will be controlled 
	 *        (CGFLocation or CFGMachine) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param execResult execution Result
	 * @param controlEvent the control event
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized String performControlEvent(
		ITestSuite suite,
		ITest theTest,
		CFGLocation location,
		String port,
		TPFExecutionResult execResult,
		String controlEvent,
		String[] params) {
	
		String machine = ((CFGMachineConstraint) location).getHostname();
		StringBuffer bufError = new StringBuffer();
	
		if (canSupportControlEvent(suite, theTest, location, port, execResult, controlEvent) == false){
			return StringUtil.change(this.systemUtility.getString("CANNOT_SUPPORT_MSG_"), "%1", controlEvent); //$NON-NLS-1$ //$NON-NLS-2$
		}
		
		// Get a session for this Node.  This is returning an session that was created during launch test method		
		this.session = this.getSession(port, this.session, machine, bufError);
	
		if (this.session == null) {
			return StringUtil.change(this.systemUtility.getString("NO_SESSION_FOUND_ERROR"), "%1", machine); //$NON-NLS-1$ //$NON-NLS-2$		
		}
	
		String error = null;
		try {
			boolean controlEventPerformed = false;
	
			IExecutionComponent[] exeEnvironments = this.session.getChildren();
			// Navigate to the IExecutor and invoke kill method to stop the running test.
			for (int i = 0; i < exeEnvironments.length; i++) {
				if (exeEnvironments[i] instanceof IExecutionEnvironment) {
					IExecutionComponent[] executors = exeEnvironments[i].getChildren();
					for (int j = 0; j < executors.length; j++) {
						if (executors[j] instanceof ExecutorStub) {
							ExecutableObjectStub exeobj = (ExecutableObjectStub)((IExecutor) executors[j]).getExecutableObject();
							if (exeobj.getTestID().equals(theTest.getId())
								&& exeobj.getSuiteID().equals(suite.getId())
								&& ((ExecutorStub) executors[j]).getExecutionResultName().equals(execResult.getName())) {
	
								error = ((IExecutor) executors[j]).performControlEvent(IExecutor.CONTROL_EVENT_SUSPEND, params);
								if (error == null){
									controlEventPerformed = true;
								}
							}
						}
					}
				}
			}
			if (!controlEventPerformed) {
				if (error == null){
					return StringUtil.change(this.systemUtility.getString("CANNOT_PERFORM_MSG_"), "%1", controlEvent); //$NON-NLS-1$ //$NON-NLS-2$
				}
				return error;
				
			}
			
		} catch (Throwable t) {
			this.systemUtility.logError(t);
			return t.getMessage();
		}
		return null;
	}

	/**
	 * Returns true if the particular control event can be performed on test specified.  
	 * @param suite the test suite to be controlled.
	 * @param theTest the test to be controlled (may be the same as suite, or may 
	 *        be a test case contained within suite.
	 * @param location on which the test will be controlled 
	 *        (CGFLocation or CFGMachine) 
	 * @param port the port on which the RAC is running (usually 10002)
	 * @param execResult the execution Result for the test run
	 * @param controlEvent the control event
	 * @return a string containing an error message, or null if no error occurred
	 */
	public synchronized boolean canSupportControlEvent(
		ITestSuite suite,
		ITest theTest,
		CFGLocation location,
		String port,
		TPFExecutionResult execResult,
		String controlEvent) {
	
		String machine = ((CFGMachineConstraint) location).getHostname();
		StringBuffer bufError = new StringBuffer();
	
		// Get a session for this Node.  This is returning an session that was created during launch test method		
		this.session = getSession(port, this.session, machine, bufError);
		if (this.session == null) {
			return false;
		}

		boolean canSupport = false;
		IExecutionComponent[] exeEnvironments = this.session.getChildren();
		// Navigate to the IExecutor find out if this can be stopped.
		for (int i = 0; i < exeEnvironments.length; i++) {
			if (exeEnvironments[i] instanceof IExecutionEnvironment) {
				IExecutionComponent[] executors = exeEnvironments[i].getChildren();
				for (int j = 0; j < executors.length; j++) {
					if (executors[j] instanceof ExecutorStub) {
						ExecutableObjectStub exeobj = (ExecutableObjectStub)((IExecutor) executors[j]).getExecutableObject();
						if (exeobj.getTestID().equals(theTest.getId())
							&& exeobj.getSuiteID().equals(suite.getId())
							&& ((ExecutorStub) executors[j]).getExecutionResultName().equals(execResult.getName())){
							if (((IExecutor) executors[j]).supportsControlEvent(controlEvent)){
								canSupport = true;
							}else{
								return false;
							}
							
						}
					}				
				}
			}
		}
		return canSupport;
	}

	/**
	 * @param suite
	 * @param theTest
	 * @return
	 */
	private IImplementor getImplementor(ITestSuite suite, ITest theTest) {
		IImplementor theImplementor = null;
		
		if ( theTest != null )
		{
			theImplementor = theTest.getImplementor();
		}
		else
		{
			theImplementor = suite.getImplementor();
		}
		
		return theImplementor;
	}
	
	public static void waitForDataProcessorExit(IExecutor executor)
	{
		IRemoteHyadesComponent remoteComponent = TestExecutionHarness.findRemoteHyadesComponent(executor);
		if ( remoteComponent != null )
		{
			while ( remoteComponent.isActive() )
			{
				try 
				{
					Thread.sleep(1000);
				}
				catch (InterruptedException ex)
				{}
			}
		}
	}
	
	/**
	 * Factory method to create a null progress monitor, this method could return any object
	 * that implements the progress monitor interface for debugging purposes in tracking calls
	 * into the concrete progress monitor instance.  The proper object to return for release
	 * purposes is a null progress monitor instance.  This method is typically called when
	 * a "real" progress monitor is not available. 
	 * 
	 * @return a null object progress monitor (instead of performing continuous null checks) 
	 */
	private IProgressMonitor createNullProgressMonitor() {
		return new NullProgressMonitor();
	}

}
