/*******************************************************************************
 * Copyright (c) 2003, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.harness;

import java.io.UnsupportedEncodingException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.execution.core.ExecutionComponentStateException;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.execution.core.IDataProcessor;
import org.eclipse.hyades.execution.local.ExecutorStub;
import org.eclipse.hyades.execution.local.SessionStub;
import org.eclipse.hyades.internal.execution.local.common.BinaryCustomCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;

/**
 * The test execution harness executor stub is used by the test execution
 * harness to launch tests.
 * 
 * @author Ernest Jessee
 * @author Scott E. Schneider
 */
public class TestExecutionHarnessExecutorStub extends ExecutorStub {

    /**
     * Agent type used for executing tests.
     */
    private static final String AGENT_TYPE = "tester";

    /**
     * Creates a custom command which is sent to the skeleton to start it.
     * 
     * @return a resume custom command
     */
    private CustomCommand createResumeCommand() {
        BinaryCustomCommand resumeCommand = new BinaryCustomCommand();
        resumeCommand.setData(IControlMessage.START);
        return resumeCommand;
    }

    /**
     * Retrieves the agent listener to be used to listen to the agent
     * controller's control channel.
     * 
     * @return the agent listener
     */
    protected AgentListener getAgentListener() {
        return new AgentListener() {
            public void agentActive(Agent agent) {
            }

            public void agentInactive(Agent agent) {
            }

            public void error(Agent agent, String errorId, String errorMessage) {
            }

            public void handleCommand(Agent agent, CommandElement command) {
            }
        };

    }

    /**
     * Returns the process associated with the session's agent.
     * 
     * @return the agent's process
     */
    private Process getAgentProcess() {
        return ((SessionStub) this.getSessionContext()).getAgent().getProcess();
    }

    /**
     * Ensures consistent handling of caught exception; exceptions are logged.
     * 
     * @param exception
     *            exception to handle
     */
    private void handleException(Exception exception) {
        ExecutionHarnessPlugin.getDefault().logError(exception);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.IExecutor#launch()
     */
    public void launch() throws ExecutionComponentStateException {
        this.launch(new NullProgressMonitor());
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.IExecutorWithProgressMonitorSupport
     *      #launch(org.eclipse.core.runtime.IProgressMonitor)
     */
    public void launch(IProgressMonitor progressMonitor)
            throws ExecutionComponentStateException {

        // Launches and then suspends execution component
        super.launch(progressMonitor);

        // Define a timeout so launch will not hang indefinitely
        int timeout = 60000;
        
        // Begin the launch task, allocating work units
        progressMonitor.beginTask("", (timeout * 10)
                + ((this.getDataProcessors().length) * 12)); //$NON-NLS-1$

        try {

            // Resets agent counter, it is important to count initialized agents
            this.resetAgentInitCount();

            // Retrieve the node from the agent's process
            Node node = this.getAgentProcess().getNode();

            // Retrieve the active data processors, set by the execution harness
            IDataProcessor[] dataProcessors = this.getDataProcessors();

            /*
             * Retrieves the specified process by identity on the given node;
             * the wait for process method blocks until the process has been
             * found, the timeout has been reached or the cancel is detected.
             */
            Process process = this.waitForProcess(this.getPid(), node, timeout,
                    progressMonitor);

            // If the process found is null then exit from launch
            if (process != null) {

                // Launch behavior determined by existence of data processors
                if (dataProcessors != null) {
                    this.launch(process, progressMonitor, dataProcessors);
                } else {
                    this.launch(process);
                }

            } else {
                
                // Handle the case of process retrieved as null
                System.out.println("Process could not be retrieved!");
                
            }

        } catch (InactiveAgentException e) {

            // Thrown from launch
            this.handleException(e);

        } catch (InactiveProcessException e) {

            // Thrown from find process
            this.handleException(e);

        } catch (NotConnectedException e) {

            // Thrown from find process
            this.handleException(e);

        } catch (UnsupportedEncodingException e) {

            // Thrown from launch
            this.handleException(e);

        } finally {

            // Before this method returns, we must mark the task as completed
            progressMonitor.done();

        }

    }

    /**
     * Launch without data processors
     *  
     */
    private void launch(Process process) throws InactiveAgentException,
            InactiveProcessException, UnsupportedEncodingException {

        Agent agent = findAgent(process,
                TestExecutionHarnessExecutorStub.AGENT_TYPE,
                XMLExecutionDataProcessor.IID);

        this.setupControlListener(this.getAgentListener(), agent);
        this.resume(agent);

    }

    /**
     * Launch with data processors
     *  
     */
    private void launch(Process process, IProgressMonitor progressMonitor,
            IDataProcessor[] dataProcessors) throws InactiveAgentException,
            InactiveProcessException, UnsupportedEncodingException {

        synchronized (process) {
            for (int i = 0, n = dataProcessors.length; i < n; i++) {

                if (progressMonitor.isCanceled()) {
                    return;
                }
                progressMonitor.worked(4);

                IExecutionHarnessDataProcessor dataProcessor = (IExecutionHarnessDataProcessor) dataProcessors[i];
                if (dataProcessor.getControlAgent() == null) {
                    Agent agent = findAgent(process,
                            TestExecutionHarnessExecutorStub.AGENT_TYPE,
                            dataProcessor.getName());

                    if (progressMonitor.isCanceled()) {
                        return;
                    }
                    progressMonitor.worked(4);

                    this.setupDataProcessor(agent, process, dataProcessor);

                    if (progressMonitor.isCanceled()) {
                        return;
                    }
                    progressMonitor.worked(4);

                    this.setupControlListener(this.getAgentListener(), agent);
                }
            }
        }

        IExecutionHarnessDataProcessor controller = (IExecutionHarnessDataProcessor) dataProcessors[0];
        if (controller != null) {
            this.resume(controller.getControlAgent());
        }

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.IExecutor
     *      #performControlEvent(java.lang.String, java.lang.String[])
     */
    public String performControlEvent(String controlEvent, String[] params) {
        return "";
    }

    /**
     * Resumes execution component with the specified agent, if the agent is
     * null then this method does nothing.
     * 
     * @param agent
     *            the agent to send command to
     */
    private void resume(Agent agent) throws InactiveAgentException {
        if (agent != null) {
            agent.invokeCustomCommand(this.createResumeCommand());
        }
    }

    /**
     * Associates specified data processor with the given agent and process and
     * then intializes the data processor.
     * 
     * @param agent
     *            the agent to use
     * @param process
     *            the process to associate with this data processor
     * @param dataProcessor
     *            the data processor to associate with the agent and process
     */
    protected void setupDataProcessor(Agent agent, Process process,
            IExecutionHarnessDataProcessor dataProcessor) {
        dataProcessor.setControlAgent(agent);
        dataProcessor.setProcess(process);
        dataProcessor.init();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.hyades.execution.core.IExecutor#supportsControlEvent(java.lang.String)
     */
    public boolean supportsControlEvent(String controlEvent) {
        return true;
    }

}