/*******************************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui;

import java.net.URL;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.HashSet;
import java.util.Set;
import java.text.MessageFormat;

import org.eclipse.core.runtime.*;
import org.eclipse.hyades.ui.adapter.HyadesAdapterFactory;
import org.eclipse.hyades.ui.internal.extension.NavigatorFilterSet;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.ResourceBundleManager;
import org.eclipse.hyades.ui.util.ILogger;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IPluginContribution;
import org.eclipse.ui.activities.IWorkbenchActivitySupport;
import org.eclipse.ui.activities.IActivityManager;
import org.eclipse.ui.activities.IIdentifier;
import org.eclipse.ui.activities.WorkbenchActivityHelper;

/**
 * Hyades UI Plugin class.  This class should not be used outside the context of
 * Eclipse.
 * 
 * @author marcelop
 * @author popescu
 * @since 0.0.1
 */
public class HyadesUIPlugin 
extends AbstractUIPlugin
{
	/**
	 * Extension point id used to register the editor extensions.
	 */
	public static final String EP_EDITOR_EXTENSIONS = "editorExtensions";

	/**
	 * Extension point id used to register the analyzer extensions.
	 */
	public static final String EP_ANALYZER_EXTENSIONS = "analyzerExtensions";
	
	/**
	 * Extension point id used to register the report extensions.
	 */
	public static final String EP_REPORT_EXTENSIONS = "reportExtensions";
	
	/**
	 * Extension point id used to register the type descriptions.
	 */
	public static final String EP_TYPE_DESCRIPTIONS = "typeDescriptions";
	
	/**
	 * Extension point id used to register the type validators.
	 */
	public static final String EP_TYPE_VALIDATORS = "typeValidators";
	
	/**
	 * Extension point id used to register the sample projects.
	 */
	public static final String EP_SAMPLE_WIZARD = "sampleWizards";

	/**
	 * Extension point id used to register navigator extensions.
	 */
	public static final String EP_NAVIGATOR_EXTENSIONS = "navigatorExtensions";
	
	private static HyadesUIPlugin instance;
	private ILogger logger;
	private ResourceBundleManager resourceBundleManager;
	
	/**
	 * Constructor for HyadesUIPlugin
	 * @param descriptor
	 */
	public HyadesUIPlugin(IPluginDescriptor descriptor)
	{
		super(descriptor);
		instance = this;
	}
	
	/**
	 * Returns the instance of this class created by the eclipse framework.
	 * @return HyadesUIPlugin
	 */
	public static HyadesUIPlugin getInstance()
	{
		return instance;
	}
	
	/**
	 * @see org.eclipse.core.runtime.Plugin#startup()
	 */
	public void startup()
	throws CoreException
	{
		resourceBundleManager = new ResourceBundleManager(); 
		resourceBundleManager.add(getDescriptor().getResourceBundle());

		Platform.getAdapterManager().registerAdapters(HyadesAdapterFactory.INSTANCE, LogicalFolder.class);

		super.startup();		

		Runnable operation = new Runnable() {
			public void run() {
				try	{
					HyadesUIImages.INSTANCE.initialize(new URL(getDescriptor().getInstallURL(), "icons/full/"), getImageRegistry());
				}
				catch(Exception e) {
					logError(e);
				}		
			}
		};
		
		/*
		 * #63850: Platform discourages use of thread syncronization in
		 * startup(). Using asyncExec() in the case where this is not
		 * already in the UI thread (not the common case).
		 */
		if (Display.getCurrent() == Display.getDefault()) {
			operation.run();
		}
		else {
			Display.getDefault().asyncExec(operation);
		}
	}

	/**
	 * @see org.eclipse.core.runtime.Plugin#shutdown()
	 */
	public void shutdown()
	throws CoreException
	{
		NavigatorExtensionUtil.disposeAll();
		resourceBundleManager.dispose();
		instance = null;
		savePluginPreferences();
		logger = null;
		Platform.getAdapterManager().unregisterAdapters(HyadesAdapterFactory.INSTANCE);
		HyadesAdapterFactory.INSTANCE.dispose();
				
		super.shutdown();
	}
	
	/**
	 * Returns the an instance of {@link ILogger} that used the
	 * <code>log</code> methods of this class.
	 * @return ILogger
	 */
	public static ILogger getLogger()
	{
		if(instance.logger == null)
		{
			instance.logger = new ILogger()
			{
				public void logError(Throwable throwable)
				{
					logError(throwable);
				}

				public void logError(String text)
				{
					logError(text);
				}

				public void logInfo(String text)
				{
					logInfo(text);
				}
			};
		}
		return instance.logger;
	}

	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return instance.getDescriptor().getUniqueIdentifier();
	}
	
	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 */
	public static void logError(Throwable throwable)
	{
		Status status = new Status(1, getID(), 0, throwable.toString(), throwable);
		getInstance().getLog().log(status);
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logError(String text)
	{
		logError(new Throwable(text));
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 */
	public static void logInfo(String text)
	{
		Status status = new Status(3, getID(), 0, text, new Throwable(text));
		getInstance().getLog().log(status);
	}
	
	/**
	 * Returns the resource bundle used by this plugin.
	 * 
	 * <p>IMPORTANT: Don't use this method to retrieve values from the
	 * resource bundle.  For this purpose use the static <code>getString()</code> 
	 * defined in this class.
	 * 
	 * <p>This method is provided so this resource bundle can
	 * be used as the parent of another resource bundle.
	 * 
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		return getDescriptor().getResourceBundle();
	}

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key);
	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The <code>arg</code> string defined replaces the %1 
	 * variable defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and arg is <code>"v1"</code>,
	 * the return of this method is <code>"%0 v1  %2 %3 %4"</code>.
	 * 
	 * @param key
	 * @param arg
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String arg)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, arg);
	}	

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String[]args)
	throws NullPointerException, MissingResourceException
	{
		return instance.resourceBundleManager.getString(key, args);
	}
	
	/**
	 * Sets default preference values. These values will be used
	 * until other preferences are actually set.
	 * 
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#initializeDefaultPreferences(org.eclipse.jface.preference.IPreferenceStore)
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store) {
		super.initializeDefaultPreferences(store);

		NavigatorFilterSet[] filters = NavigatorExtensionUtil.getFilterSetElements(null);
		for (int i=0;i<filters.length;++i)
			store.setDefault(filters[i].getPreferenceKey(), filters[i].isDefaultEnabled());
	}
	
	/**
	 * Asks the user whether they want to switch to the given
	 * perspective. Returns the user's response. If we are already
	 * in the given perspective, returns false. Must be called from
	 * the UI thread.
	 */
	public boolean openPerspectiveSwitchDialog(Shell shell, String perspectiveId, IPreferenceStore preferenceStore, String preferenceKey, String message) {
		if (isCurrentPerspective(PlatformUI.getWorkbench().getActiveWorkbenchWindow(), perspectiveId)) {
			return false;
		}
		String perspectiveName= getPerspectiveLabel(perspectiveId);
		if (perspectiveName == null) {
			return false;
		}
		String preferenceValue = preferenceStore.getString(preferenceKey);
		if (preferenceValue == null || preferenceValue.length() == 0) {
			preferenceValue = MessageDialogWithToggle.PROMPT;
			preferenceStore.setValue(preferenceKey, preferenceValue);
		}
		boolean saveAndChange = false;
		if (MessageDialogWithToggle.PROMPT.equals(preferenceValue)) {
			MessageDialogWithToggle dialog = MessageDialogWithToggle.openYesNoQuestion(shell, 
									getResourceBundle().getString("CONFIRM_PERSP_SWITCH"), 
									MessageFormat.format(message, new String[] { perspectiveName }),
									null, false, preferenceStore, preferenceKey); //$NON-NLS-1$
			saveAndChange = (dialog.getReturnCode()== IDialogConstants.YES_ID);
		}

		if (!getContributions(perspectiveId)){
			return false;
		}
		preferenceValue = preferenceStore.getString(preferenceKey);
		if (MessageDialogWithToggle.ALWAYS.equals(preferenceValue)) {
			return true;
		}
		else if (!saveAndChange || MessageDialogWithToggle.NEVER.equals(preferenceValue)){
			return false;
		}
		return true;
	}
	
	/**
	 * Returns whether the given perspective identifier matches the
	 * identifier of the current perspective.
	 * 
	 * @param perspectiveId the identifier
	 * @return whether the given perspective identifier matches the
	 *  identifier of the current perspective
	 */
	private boolean isCurrentPerspective(IWorkbenchWindow window, String perspectiveId) {
		boolean isCurrent= false;
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				IPerspectiveDescriptor perspectiveDescriptor = page.getPerspective();
				if (perspectiveDescriptor != null) {
					isCurrent= perspectiveId.equals(perspectiveDescriptor.getId());
				}
			}
		}
		return isCurrent;
	}	

	/**
	 * Returns the label of the perspective with the given identifier or
	 * <code>null</code> if no such perspective exists.
	 * 
	 * @param perspectiveId the identifier
	 * @return the label of the perspective with the given identifier or
	 *  <code>null</code> if no such perspective exists 
	 */
	private String getPerspectiveLabel(String perspectiveId) {
		IPerspectiveDescriptor newPerspective = PlatformUI.getWorkbench().getPerspectiveRegistry().findPerspectiveWithId(perspectiveId);
		if (newPerspective == null) {
			return null;
		}
		return newPerspective.getLabel();
	}
	
	/**
	 * Check if required activities are enabled.
	 */
	private boolean getContributions(String perspectiveId){
		// Map perspective id to descriptor.
        IPerspectiveRegistry reg = PlatformUI.getWorkbench().getPerspectiveRegistry();

        // leave this code in - the perspective of a given project may map to
        // activities other than those that the wizard itself maps to.
        IPerspectiveDescriptor finalPersp = reg.findPerspectiveWithId(perspectiveId);
        if (finalPersp != null && finalPersp instanceof IPluginContribution) {
            IPluginContribution contribution = (IPluginContribution) finalPersp;
            
            if (contribution.getPluginId() != null) {
                IWorkbenchActivitySupport workbenchActivitySupport = PlatformUI.getWorkbench().getActivitySupport();
                IActivityManager activityManager = workbenchActivitySupport.getActivityManager();
                IIdentifier identifier = activityManager.getIdentifier(WorkbenchActivityHelper.createUnifiedId(contribution));
                Set idActivities = identifier.getActivityIds();

                if (!idActivities.isEmpty()) {
                    Set enabledIds = new HashSet(activityManager.getEnabledActivityIds());

                    if (enabledIds.addAll(idActivities))
                        workbenchActivitySupport.setEnabledActivityIds(enabledIds);
                }
            }
        } else {
            logError("Unable to find perspective " + perspectiveId);//$NON-NLS-1$
            return false;
        }
        return true;
	}
}
