/*******************************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;

/**
 * Internal filtering mechanism for Hyades navigators. Content
 * providers are wrapped in this content provider, which consults
 * all navigatorFilterSet extensions and provides only unfiltered
 * content.
 * 
 * @since 3.0 
 */
public class FilteredContentProvider implements ITreeContentProvider {
	
	private static final Object[] EMPTY = new Object[0];
	
	private ITreeContentProvider _tree;
	private IStructuredContentProvider _structured;
	private IContentProvider _provider;
	private String _navigatorID;
	private Viewer _viewer;
	
	/**
	 * Constructs a new FilteredContentProvider, wrapping the
	 * given one.
	 *  
	 * @param provider the original content provider.
	 * @param navigatorID the id of the content provider's navigator.
	 */
	public FilteredContentProvider(IContentProvider provider, String navigatorID, Viewer viewer) {
		_provider = provider;
		if (provider instanceof IStructuredContentProvider)
			_structured = (IStructuredContentProvider)provider;
		if (provider instanceof ITreeContentProvider)
			_tree = (ITreeContentProvider)provider;
		_navigatorID = navigatorID;
		_viewer = viewer;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement) {
		if (_tree != null) {
			Object[] e = _tree.getChildren(parentElement);
			List list = new ArrayList();
			for (int i=0;i<e.length;++i) {
				if (NavigatorExtensionUtil.select(_viewer, parentElement, e[i], _navigatorID, false))
					list.addAll(Arrays.asList(getChildren(e[i])));
				else if (!NavigatorExtensionUtil.select(_viewer, parentElement, e[i], _navigatorID, true))
					list.add(e[i]);
			}
			return list.toArray();
		}
		return EMPTY;
	}
	
	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element) {
		if (_tree != null) {
			Object parent = _tree.getParent(element);
			if (parent != null) {
				Object grandParent = _tree.getParent(parent);
				if (NavigatorExtensionUtil.select(_viewer, grandParent, parent, _navigatorID))
					return getParent(parent);
				return parent;
			}
		}
		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element) {
		if (_tree != null) {
			if (!_tree.hasChildren(element))
				return false;
			return (getChildren(element).length > 0);
		}
		return false;
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		if (_structured != null) {
			Object[] e = _structured.getElements(inputElement);
			List list = new ArrayList();
			for (int i=0;i<e.length;++i) {
				if (NavigatorExtensionUtil.select(_viewer, inputElement, e[i], _navigatorID, false))
					list.addAll(Arrays.asList(getChildren(e[i])));
				else if (!NavigatorExtensionUtil.select(_viewer, inputElement, e[i], _navigatorID, true))
					list.add(e[i]);
			}
			return list.toArray();
		}
		return EMPTY;
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose() {
		_provider.dispose();
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		_provider.inputChanged(viewer, oldInput, newInput);
	}
}
