/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui;

import java.net.URL;
import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.hyades.uml2sd.ui.load.BackgroundLoader;
import org.eclipse.hyades.uml2sd.ui.load.IUml2SDLoader;
import org.eclipse.hyades.uml2sd.ui.load.LoadersManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * The main plugin class to be used in the desktop.
 */
public class SDViewerPlugin extends AbstractUIPlugin {

	public static final String LOADER_TAG = "uml2SDLoader"; //$NON-NLS-1$
	public static final String LOADER_PREFIX = LOADER_TAG+"."; //$NON-NLS-1$
	
	//The shared instance.
	private static SDViewerPlugin plugin;
	//Resource bundle.
	private ResourceBundle resourceBundle;
	
	/**
	 * The constructor.
	 */
	public SDViewerPlugin()
   {
		super();
		plugin = this;
	}
	
	public void start(BundleContext context) throws Exception {
		super.start(context);
		try {
			this.resourceBundle = Platform.getResourceBundle(this.getBundle());
		} catch (MissingResourceException x) {
			this.resourceBundle = null;
		}
		createLastLoaderIfAny();
	}
	
	public void stop(BundleContext context) throws Exception {
		BackgroundLoader.shutdown();
		super.stop(context);
	}

	/**
	 * Loads the last loader again (if still here!)
	 */
	public final void createLastLoaderIfAny()
	{
		IExtensionPoint iep	=Platform.getExtensionRegistry().getExtensionPoint(
				getId(),LOADER_TAG);
		IExtension ie[] = iep.getExtensions();
		
		Preferences p = getPluginPreferences();
		
		String[]loaders=p.propertyNames();
		
		for (int h=0;h<loaders.length;h++)
		{
		if (loaders[h].startsWith(LOADER_PREFIX))	
		{
		String viewId=loaders[h].substring(LOADER_PREFIX.length());
		String lastLoader = p.getString(loaders[h]);
		
		if (ie==null || lastLoader == null || lastLoader.length() == 0) return;
		for (int i = 0; i < ie.length; i++)
		{
			IExtension e = ie[i];
			IConfigurationElement c[] = e.getConfigurationElements();
			for (int j = 0; j < c.length; j++) {
				if (c[j].getName().equals(LOADER_TAG)) {
					String attr_class = c[j].getAttribute("class"); //$NON-NLS-1$
					
					if (lastLoader.equals(attr_class)) {
						try {
							Object obj = c[j].createExecutableExtension("class");//$NON-NLS-1$
							IUml2SDLoader l = (IUml2SDLoader)obj;
							if (viewId!=null)
								LoadersManager.getLoadersManager().setCurrentLoader(l,viewId);//$NON-NLS-1$
						} catch (CoreException e4) {
							System.err.println("Error 'uml2SDLoader' Extension point :"+e4);//$NON-NLS-1$
						} catch(Exception e5) {
							e5.printStackTrace();
							System.err.println("Error 'uml2SDLoader' Extension point :"+e5);//$NON-NLS-1$
						}
						//return;
					}
				}
			}
		}
		}
		}
	}
	
	/**
	 * Translate the given item into a localised string.
	 * @param time - the time to translate
	 * @return the localised string
	 */
	public static String timeToString(double time) {
		String ret;
		if (Math.abs(time) > 24*60*60) {
			int days, hours;
			days = (int)(time / (24*60*60));
			hours = Math.abs((int)(time % (24*60*60))/(60*60));
			if ((time<0)&&(days==0))
					days=-days;
			if ((days != 1)&&(days != -1)) {
				ret = MessageFormat.format(SDViewerPlugin.getResourceString("STR_DAYS_HOURS"), //$NON-NLS-1$
										   new Object[] { new Integer(days), new Integer(hours) } );
			} else {
				ret = MessageFormat.format(SDViewerPlugin.getResourceString("STR_DAY_HOURS"), //$NON-NLS-1$
										   new Object[] { new Integer(days), new Integer(hours) } );
			}
		} else if (Math.abs(time) > 60*60) {
			int hours, mins;
			hours = (int)(time / (60*60));
			mins = Math.abs((int)(time % (60*60))/(60));
			if ((time<0)&&(hours==0))
				hours=-hours;
			ret = MessageFormat.format(SDViewerPlugin.getResourceString("STR_HOURS_MINUTES"), //$NON-NLS-1$
									   new Object[] { new Integer(hours), new Integer(mins) } );
		} else if (Math.abs(time) > 60) {
			int mins, s;
			mins = (int)(time / 60);
			s = (int)(time % 60);
			if ((time<0)&&(mins==0))
					mins=-mins;
			ret = MessageFormat.format(SDViewerPlugin.getResourceString("STR_MINUTES_SECONDS"), //$NON-NLS-1$
								 	   new Object[] { new Integer(mins), new Integer(s) } );
		} else {
			double s; double us;
			s = (int)(time);
			us = Math.abs((time-s)*1000000);
			if ((time<0)&&(s==0))
				s=-s;
			ret = MessageFormat.format(SDViewerPlugin.getResourceString("STR_SECONDS_MICROSECONDS"), //$NON-NLS-1$
									   new Object[] { new Double(s), new Double(us) } );
		}
		return ret;
	}
	
	/**
	 * Saves the last loader in order to reload it on next session.
	 */
	public void setLastLoader(String id,String id2) {
		Preferences p = getPluginPreferences();
		p.setValue(LOADER_PREFIX+id2, id);
	}
	
	/**
	 * Returns the shared instance.
	 */
	public static SDViewerPlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle= SDViewerPlugin.getDefault().getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}


	private static URL BASIC_URL=null;
	static
	{
		//URL to get images from plugins
		//perhaps we can found better code to load images ...
		//but at this time I know this: 
		try {
		BASIC_URL = new URL( "platform", "localhost", "plugin");//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		} catch( Exception E ) {
			System.err.println(E);
		}
	}

	public static ImageDescriptor getResourceImage( String _name )
	{
		try {
		 URL url=new URL( BASIC_URL, "plugin/org.eclipse.hyades.uml2sd.ui/icons/"+_name);//$NON-NLS-1$
		 ImageDescriptor img = ImageDescriptor.createFromURL( url );
		return img ;
		} catch( Exception E ) {
					 System.err.println(E);
		}
		return null;
	}
	
	public static String getId()
	{
		return getDefault().getBundle().getSymbolicName(); 
	}
	
	public static boolean debugIndex()
	{
		if (debug())
		{
			if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/updateindex").//$NON-NLS-1$
				equalsIgnoreCase("true"))//$NON-NLS-1$
				return true;
		}
		return false;
	}
	
	public static boolean debugDisplay()
	{
		if (debug())
		{
			if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/display").//$NON-NLS-1$
				equalsIgnoreCase("true"))//$NON-NLS-1$
				return true;
		}
		return false;
	}
	
	public static boolean debugDoubleBuffer()
	{
		if (debug())
		{
			if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/doublebuffer").//$NON-NLS-1$
				equalsIgnoreCase("true"))//$NON-NLS-1$
				return true;
		}
		return false;
	}
	
	public static boolean debugSorting()
	{
		if (debug())
		{
			if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/sorting").//$NON-NLS-1$
				equalsIgnoreCase("true"))//$NON-NLS-1$
				return true;
		}
		return false;
	}
	
	public static boolean debug()
	{
		if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/debug")!=null) //$NON-NLS-1$
		{
			if (Platform.getDebugOption("org.eclipse.hyades.uml2sd.ui/debug").//$NON-NLS-1$
				equalsIgnoreCase("true")) //$NON-NLS-1$
				return true; 
		}
		return false;
	}
	
}
