/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;

import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;

/**
 * A AsyncMessage is a asynchronous message which appear at two different event
 *  occurences on each lifeline ends (sender and receiver).<br>
 * <br>
 * <br>
 * Usage example:
 * <pre>
 *  Frame frame;
 *  Lifeline lifeLine1;
 *  Lifeline lifeLine2;
 * 
 *	AsyncMessage message = new AsyncMessage();
 *	//Create a new event occurrence on each lifeline
 *	lifeline1.getNewOccurrenceIndex();
 *	lifeline2.getNewOccurrenceIndex();
 *  //Set the message sender and receiver
 *	message.setStartLifeline(lifeLine1);
 *	message.setEndLifline(lifeline2);
 *	message.setName("Message label");
 *  //add the message to the frame
 *	frame.addMessage(message);
 * </pre>
 * 
* @see Lifeline Lifeline for more event occurence details 
 * @author sveyrier
 *
 */
public class AsyncMessage extends BaseMessage implements ITimeRange{
	
	
	protected boolean hasTime = false;
	/**
	 * The time when the message begin
	 */
	protected double endTime = 0;
	
	/**
	 * The time when the message end
	 */
	protected double startTime = 0;
	
	/**
	 * The associated message.
	 */
	protected AsyncMessageReturn	messageReturn = null;
	
	/**
	 * This basicaly represents the time when the message has been sent.
	 * The message reception property in define in BaseMesssage.
	 * @see Lifeline Lifeline for more event occurence details
	 */
	protected int	startEventOccurrence = 0;
	
	public AsyncMessage()
	{
		prefId =ISDPreferences.PREF_ASYNC_MESS;
	}
	
	public int getX()
	{
		int x= super.getX(true);
		int activationWidth = Metrics.EXECUTION_OCCURRENCE_WIDTH/2;
		if ((startLifeline!=null)&&(endLifeline!=null)&&(startLifeline.getX()>endLifeline.getX()))
		{
			activationWidth = - activationWidth;
		}

		if (isMessageStartInActivation(startEventOccurrence))
		{
			x = x + activationWidth;
		}
		return x;
	}
	
	public int getY()
	{
		if ((startLifeline!=null)&&(endLifeline!=null))
		{
			return endLifeline.getY() + endLifeline.getHeight() + 
					(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * startEventOccurrence;
		}	
		return super.getY();
	}
	
	public int getWidth()
	{ 
		int width = super.getWidth(true);
		int activationWidth = Metrics.EXECUTION_OCCURRENCE_WIDTH/2;
		if ((startLifeline!=null)&&(endLifeline!=null)&&(startLifeline.getX()>endLifeline.getX()))
		{
		   	activationWidth = - activationWidth;
		}
			 
	  	if (isMessageStartInActivation(startEventOccurrence)) 
		   	width = width - activationWidth;
				
	   	if (isMessageEndInActivation(eventOccurrence))
		   	width = width - activationWidth;
			 
	   	return width;
	}
	
	public int getHeight()
	{
		if ((startLifeline!=null)&&(endLifeline!=null))
		{
			return (endLifeline.getY() + endLifeline.getHeight() + 
					(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * eventOccurrence) -getY();
		}
		return super.getHeight();
	}	
	
	/**
	 * Set the message return associated with this message.
	 * @param message the message return to associate
	 */			
	protected void setMessageReturn(AsyncMessageReturn message)
	{
		messageReturn = message;
	}
	
	/**
	 * Set the event occurrence attached to this message
	 * for its end lifeline
	 * @param occurrence the event occurrence to set
	 */
	public void setEndOccurrence (int occurrence)
	{
		eventOccurrence = occurrence;
		if (getStartLifeline() == null)
			startEventOccurrence = occurrence;
		informFrame(getEndLifeline(),occurrence);
	}
	
	/**
	 * Returns the event occurrence attached to this message
	 * for its start lifeline
	 * @return the start event occurrence attached to the message
	 */
	public int getEndOccurrence()
	{
		return eventOccurrence;
	}
	
	
	protected void informFrame(Lifeline lifeLine,int occurrence)
	{
		if ((lifeLine !=null)&&(lifeLine.getFrame()!= null))
			if (lifeLine.getFrame().getMaxEventOccurrence()<occurrence)
				lifeLine.getFrame().setMaxEventOccurrence(occurrence);
	}
	
	
	/**
	 * Set the event occurrence attached to this message
	 * for its start lifeline
	 * @param occurrence the event occurrence to set
	 */
	public void setStartOccurrence (int occurrence)
	{
		startEventOccurrence = occurrence;
		if (getEndLifeline() == null)
			eventOccurrence = startEventOccurrence;
		informFrame(getStartLifeline(),occurrence);
	}

	/**
	 * Returns the event occurrence attached to this message
	 * for its start lifeline
	 * @return the start event occurrence attached to the message
	 */
	public int getStartOccurrence()
	{
		return startEventOccurrence;
	}
	
	/**
	 * Set the lifeLine which has sent the message.<br>
	 * A new EventOccurence will be create on this lifeLine.<br>
	 * @param lifeline the message sender
	 * @param autoCreateEvent if true, create an eventOccurence lifeline given in parameter
	 */
	public void autoSetStartLifeline(Lifeline lifeline)
	{
		lifeline.getNewEventOccurrence();
		setStartLifeline(lifeline);
	}
	
	/**
	 * Set the lifeLine which has received the message.<br>
	 * A new EventOccurence will be create on this lifeLine.<br>
	 * @param lifeline the message receiver
	 * @param autoCreateEvent if true, create an eventOccurence lifeline given in parameter
	 */
	public void autoSetEndLifeline(Lifeline lifeline)
	{
		lifeline.getNewEventOccurrence();
		setEndLifeline(lifeline);
	}
	
	/**
	 * Set the lifeLine which has sent the message.<br>
	 * @param lifeline the message sender
	 */
	public void setStartLifeline(Lifeline lifeline)
	{
		super.setStartLifeline(lifeline);
		setStartOccurrence(getStartLifeline().getEventOccurrence());
		if (getEndLifeline() == null)
			eventOccurrence = startEventOccurrence;
	}
	
	/**
	 * Set the lifeLine which has received the message.<br>
	 * @param lifeline the message receiver
	 */
	public void setEndLifeline(Lifeline lifeline)
	{
		super.setEndLifeline(lifeline);
		setEventOccurrence(getEndLifeline().getEventOccurrence());
	}
	
	/**
	 * Returns true if the point C is on the segment defined with the point A and B
	 * 
	 * @param xA	point A x coordinate
	 * @param yA	point A y coordinate
	 * @param xB	point B x coordinate
	 * @param yB	point B y coordinate
	 * @param xC	point C x coordinate
	 * @param yC	point C y coordinate
	 * @return Return true if the point C is on the segment defined with the point A and B, else otherwise
	 */
	protected boolean isNearSegment(int xA, int yA, int xB, int yB,int xC, int yC)
	{
		if ((xA > xB) && (xC > xA ))
			return false;
		if ((xA < xB) && (xC > xB ))
			return false;
		if ((xA < xB) && (xC < xA ))
			return false;
		if ((xA > xB) && (xC < xB ))
			return false;
		double distAB = Math.sqrt((xB-xA)*(xB-xA) + (yB-yA)*(yB-yA));
		double scalar = ((xB-xA)*(xC-xA) + (yB-yA)*(yC-yA))/distAB;
		double distAC = Math.sqrt((xC-xA)*(xC-xA) + (yC-yA)*(yC-yA));
		double distToSegment = Math.sqrt(Math.abs(distAC*distAC - scalar*scalar));
		if (distToSegment <= Metrics.MESSAGE_SELECTION_TOLERANCE)
			return true;
		return false;
	}
	
	public boolean contains (int x, int y)
	{	
		//Used to create a rectangle which contains the message label to allow selection when clicking the label
		int tempHeight = Metrics.MESSAGES_NAME_SPACING + Metrics.getMessageFontHeigth();
		
		// Is it a self message?
		if (startLifeline == endLifeline)
		{
			/*
			 * Rectangle.contains(x,y, width, height) does not works with negative height or width
			 * We check here if the rectangle width is negative. 
			 */
 			if (getName().length()*Metrics.getAverageCharWidth()>
							Metrics.swimmingLaneWidth() - Metrics.EXECUTION_OCCURRENCE_WIDTH/2+
							- Metrics.INTERNAL_MESSAGE_WIDTH)
			{
				if (Frame.contains(getX() + Metrics.INTERNAL_MESSAGE_WIDTH +10,
							getY(), Metrics.swimmingLaneWidth() - Metrics.EXECUTION_OCCURRENCE_WIDTH/2+
							- Metrics.INTERNAL_MESSAGE_WIDTH, Metrics.getMessageFontHeigth(),
							x, y))
					return true;							 
			}
			else 
			{
				if (Frame.contains(getX() + Metrics.INTERNAL_MESSAGE_WIDTH +10,
					getY(), getName().length()*Metrics.getAverageCharWidth(), Metrics.getMessageFontHeigth(),
					x, y))
				return true;	
			}
			
			//Test if the point is in part 1 of the self message
			//see: 	"private void drawMessage (NGC context)" method for self message drawing schema
			if (Frame.contains(getX() , getY() - Metrics.MESSAGE_SELECTION_TOLERANCE/2, 
					Metrics.INTERNAL_MESSAGE_WIDTH/2, Metrics.MESSAGE_SELECTION_TOLERANCE,x, y))
				return true;
			//Test if the point is in part 3 of the self message
			if (Frame.contains(getX() + Metrics.INTERNAL_MESSAGE_WIDTH - Metrics.MESSAGE_SELECTION_TOLERANCE/2,
					 getY(), Metrics.MESSAGE_SELECTION_TOLERANCE, getHeight()+ Metrics.SYNC_INTERNAL_MESSAGE_HEIGHT,
					 x, y) )
				return true;
			//Test if the point is in part 5 of the self message
			if (Frame.contains(getX(), getY() + getHeight() - Metrics.MESSAGE_SELECTION_TOLERANCE/2 +
					Metrics.SYNC_INTERNAL_MESSAGE_HEIGHT,
					Metrics.INTERNAL_MESSAGE_WIDTH/2, Metrics.MESSAGE_SELECTION_TOLERANCE,x, y))
				return true;
			//false otherwise		
			return false;
		}
		if (isNearSegment(getX(), getY(),getX() + getWidth(),
				getY() + getHeight(), x, y))
			return true;
		int messageMaxWidth = Metrics.swimmingLaneWidth() - Metrics.EXECUTION_OCCURRENCE_WIDTH;
		int nameWidth = getName().length()*Metrics.getAverageCharWidth();
		if (getName().length()*Metrics.getAverageCharWidth()>messageMaxWidth)
		{
			if (Frame.contains(getX(), getY() -Metrics.MESSAGES_NAME_SPACING-Metrics.getMessageFontHeigth(), 
						messageMaxWidth, Metrics.getMessageFontHeigth(),x, y))
				return true;					 
		}
		else 
		{
			if (Frame.contains(getX()+ (messageMaxWidth - nameWidth)/2, 
							getY() -Metrics.MESSAGES_NAME_SPACING-Metrics.getMessageFontHeigth(), 
							nameWidth, Metrics.getMessageFontHeigth(),x, y))
				return true;	
		}
		
		return false;
	}
	
	private void drawAsyncMessage(IGC context)
	{
		if (startLifeline !=null && endLifeline != null && startLifeline == endLifeline &&(startEventOccurrence!=eventOccurrence))
		{
			int x = getX();
			int y = getY();
			int width = getWidth();
			int height = getHeight();
			int tempx = 0;
			boolean startInActivation = isMessageStartInActivation(startEventOccurrence);
			boolean endInActivation   = isMessageEndInActivation(eventOccurrence);
			
			if (endInActivation&&!startInActivation)
				tempx = Metrics.EXECUTION_OCCURRENCE_WIDTH/2;
			if (startInActivation&&!endInActivation)
				tempx = - Metrics.EXECUTION_OCCURRENCE_WIDTH/2;
				
			int tempy = Metrics.INTERNAL_MESSAGE_WIDTH/2;
			if (getHeight()<=Metrics.INTERNAL_MESSAGE_WIDTH)
				tempy = getHeight()/2;
			 
			context.drawLine( x , y, x + Metrics.INTERNAL_MESSAGE_WIDTH/2, y);
			context.drawLine( x + Metrics.INTERNAL_MESSAGE_WIDTH, y + tempy, x + Metrics.INTERNAL_MESSAGE_WIDTH,
							  y + height - tempy);
			context.drawLine( x + tempx, y + height, x + Metrics.INTERNAL_MESSAGE_WIDTH/2, y + height);
	
			Double xt=new Double(Math.cos(0.75)*7);
			Double yt=new Double(Math.sin(0.75)*7);

			context.drawLine( x + xt.intValue()+ tempx, y + height + yt.intValue(), 
							  x + tempx, y + height);						
			context.drawArc( x, y , 
							 Metrics.INTERNAL_MESSAGE_WIDTH, 2*tempy, 0,90);
			context.drawArc( x, y + height, 
						     Metrics.INTERNAL_MESSAGE_WIDTH, -2*tempy, 0,-90);
			context.drawLine( x + xt.intValue()+ tempx, y + height - yt.intValue(), 
							  x + tempx, y + height);
								
			context.drawTextTruncated(getName(),x + Metrics.INTERNAL_MESSAGE_WIDTH + Metrics.INTERNAL_MESSAGE_V_MARGIN,
								y, Metrics.swimmingLaneWidth() - Metrics.EXECUTION_OCCURRENCE_WIDTH+
								- Metrics.INTERNAL_MESSAGE_WIDTH, 
								+ Metrics.MESSAGES_NAME_SPACING - Metrics.getMessageFontHeigth(),!isSelected());
		}
		else super.draw(context);
	}
	
	/**
	 * Draws the asynchrous message in the given GC
	 */
	public void draw(IGC context)
	{
		if (!isVisible())
			return;
		//Draw it selected?
		if (isSelected())
		{
			/*
			 * Draw it twice
			 * First time, bigger inverting selection colors
			 * Second time, regular drawing using selection colors
			 * This create the highlight effect
			 */
			context.setForeground(Frame.getUserPref().getBackGroundColorSelection());
			context.setLineWidth(Metrics.SELECTION_LINE_WIDTH);
			drawAsyncMessage(context);
			context.setBackground(Frame.getUserPref().getBackGroundColorSelection());
			context.setForeground(Frame.getUserPref().getForeGroundColorSelection());
			//Second drawing is done after the else
		}
		else 
		{
			context.setBackground(Frame.getUserPref().getBackGroundColor(prefId));
			context.setForeground(Frame.getUserPref().getForeGroundColor(prefId));
		}
		context.setLineWidth(Metrics.NORMAL_LINE_WIDTH);
		drawAsyncMessage(context);
	}
	
	/**
	 * Set the time when the message end
	 * @param time the time when the message end
	 */
	public void setEndTime(double time)
	{
		endTime = time;
		hasTime = true;
		if (getStartLifeline() != null&& getStartLifeline().getFrame() !=null)
		  getStartLifeline().getFrame().setHasTimeInfo(true);
		else if (getEndLifeline() != null&& getEndLifeline().getFrame() !=null)
		  getEndLifeline().getFrame().setHasTimeInfo(true);
	 }
	 
	/**
	 * Set the time when the message start
	 * @param time the time when the message start
	 */
	public void setStartTime(double time)
	{
		startTime = time;
		hasTime = true;
		if (getStartLifeline() != null&& getStartLifeline().getFrame() !=null)
		  getStartLifeline().getFrame().setHasTimeInfo(true);
		else if (getEndLifeline() != null&& getEndLifeline().getFrame() !=null)
		  getEndLifeline().getFrame().setHasTimeInfo(true);
	 }
	
	/**
	 * Returns the time when the message begin
	 * @return the time
	 */	
	 public double getLastTime()
	 {
		return endTime;
	 }
	
	/**
	 * Returns the time when the message end
	 * @return the time
	 */	
	 public double getFirstTime()
	 {
		return startTime;
	 }
	 
	 public boolean hasTimeInfo()
	 {
	    return hasTime;
	 }
}
